package ru.yandex.direct.internaltools.utils;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.validation.constraints.NotNull;

import ru.yandex.direct.utils.PassportUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;

/**
 * Утилитные методы для парсинга и валидации параметров для внутренних инструментов
 */
public class ToolParameterUtils {

    private static final String ID_DELIMITER = "[,\\s]+";
    private static final String PAIRS_DELIMITER = "[|\\r\\n]+";

    private ToolParameterUtils() {
    }

    /**
     * Проверка, что строка содержит валидные Long значения кол-во которых >= {@code minCount}
     */
    public static Constraint<String, Defect> isStringWithValidLongIds(int minCount) {
        return Constraint.fromPredicate(isValidLongIds(minCount), CommonDefects.validId());
    }

    private static Predicate<String> isValidLongIds(int minCount) {
        return s -> {
            try {
                Set<Long> ids = getLongIdsFromString(s);
                for (Long id : ids) {
                    // Идентификатор должен быть неотрицательным
                    if (id < 0) {
                        return false;
                    }
                }
                return ids.size() >= minCount;
            } catch (NumberFormatException e) {
                return false;
            }
        };
    }

    /**
     * Получить список Long идентификаторов из строки. Разделителем является запятая
     *
     * @param campaignIds строка с идентификаторами разделенными запятыми. Пробелы между числами игнорируются
     * @return список Long идентификаторов полученных и строки
     */
    public static Set<Long> getLongIdsFromString(@NotNull String campaignIds) {
        return Stream.of(campaignIds.trim().split(ID_DELIMITER))
                .map(Long::parseLong)
                .collect(Collectors.toSet());
    }

    /**
     * Получить сет уникальных строковых значений разделенных запятой из строки. Разделителем является запятая
     *
     * @param listInString строка со значениями разделенными запятыми. Пробелы игнорируются
     * @return сет уникальных строковых значений полученных и строки
     */
    public static Set<String> parseCommaSeparatedString(String listInString) {
        if (listInString == null || listInString.equals("")) {
            return Collections.emptySet();
        }
        return Stream.of(listInString.trim().split(ID_DELIMITER))
                .collect(Collectors.toSet());
    }

    public static Set<String> parseVideoFilesString(String listInString) {
        return parseCommaSeparatedString(listInString).stream().map(ToolParameterUtils::removeQuotes)
                .collect(Collectors.toSet());
    }

    private static String removeQuotes(String line) {
        return line.replace("\"", "");
    }

    /**
     * Получить сет уникальных логинов из строки. Разделителем является запятая
     *
     * @param loginsList строка с логинами разделенными запятыми. Пробелы игнорируются
     * @return сет логинов полученных и строки
     */
    public static Set<String> parseLogins(String loginsList) {
        return parseCommaSeparatedString(loginsList).stream().map(PassportUtils::normalizeLogin)
                .collect(Collectors.toSet());
    }

    private static Long longLongKeyMapper(String pair) {
        return Long.parseLong(pair.trim().split(ID_DELIMITER)[0]);
    }

    private static Long longLongValueMapper(String pair) {
        return Long.parseLong(pair.trim().split(ID_DELIMITER)[1]);
    }

    /**
     * Разобрать строку из пар Long, разделённых переносами строк, на словарь из перового во второй Long.
     * Пары разделены запятыми И/ИЛИ переносами строк.
     * Не разрешает конфликты по ключам, кидает исключения.
     *
     * @param mapping — строка для разбора
     * @return словарь маппинга первых Long во вторые Long
     */
    public static Map<Long, Long> parseLongLongMap(String mapping) {
        return Stream.of(mapping.trim().split(PAIRS_DELIMITER)).collect(Collectors.toMap(
                ToolParameterUtils::longLongKeyMapper, ToolParameterUtils::longLongValueMapper
        ));
    }

    /**
     * Разобрать строку из упорядоченных пар Long, разделённых переносами строк, на словарь из первого во второй Long.
     * Пары разделены запятыми И/ИЛИ переносами строк.
     * Не разрешает конфликты по ключам, кидает исключения.
     *
     * @param mapping — строка для разбора
     * @return упорядоченный словарь маппинга первых Long во вторые Long
     */
    public static Map<Long, Long> parseLongLongLinkedMap(String mapping) {
        Map<Long, Long> result = new LinkedHashMap<>();
        for (var pair : mapping.trim().split(PAIRS_DELIMITER)) {
            var key = longLongKeyMapper(pair);
            if (result.containsKey(key)) {
                throw new IllegalStateException("Key " + key + " already exists in the map");
            }
            result.put(key, longLongValueMapper(pair));
        }
        return result;
    }
}
