package ru.yandex.direct.juggler.check;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.juggler.JugglerEvent;
import ru.yandex.direct.juggler.JugglerStatus;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Базовый класс для описания числовой juggler-проверки. Предполагается, что его наследники будут проаннотированы
 * {@link ru.yandex.direct.juggler.check.annotation.JugglerCheck}, что позволит автовайрить их в места использования
 * и генерировать по ним описания для создания плейбуков.
 */
@ParametersAreNonnullByDefault
public abstract class DirectNumericCheck {
    private static final String ASCENDING_MESSAGE_TEMPLATE = "%s >= %s";
    private static final String DESCENDING_MESSAGE_TEMPLATE = "%s <= %s";

    private final String serviceName;
    private final Long warnBorder;
    private final Long critBorder;
    private final String description;
    private final boolean ascending;
    private final String host;

    /**
     * @param serviceName основное имя проверки
     * @param warnBorder  лимит, при преодолении которого статус сгенерированного события станет WARN
     * @param critBorder  лимит, при преодолении которого статус сгенерированного события станет CRIT
     * @param description общее описание события
     * @param host        хост, на который будет отправляться событие и заводиться проверка
     * @param ascending   true, если статус сгенерированного события ухудшается при превышении лимита, false - если наоброт
     */
    public DirectNumericCheck(String serviceName, @Nullable Long warnBorder, @Nullable Long critBorder,
                              String host, String description, boolean ascending) {
        if (warnBorder == null && critBorder == null) {
            throw new IllegalArgumentException("Both warn and crit limits cannot be null");
        } else if (warnBorder != null && critBorder != null) {
            if (ascending && critBorder.compareTo(warnBorder) < 0) {
                throw new IllegalArgumentException(
                        String.format("Crit limit should not be less than warn limit (%s < %s)", critBorder,
                                warnBorder));
            } else if (!ascending && critBorder.compareTo(warnBorder) > 0) {
                throw new IllegalArgumentException(
                        String.format("Crit limit should not be more than warn limit (%s < %s)", critBorder,
                                warnBorder));
            }
        }

        this.serviceName = checkNotNull(serviceName);
        this.warnBorder = warnBorder;
        this.critBorder = critBorder;
        this.description = checkNotNull(description);
        this.ascending = ascending;
        this.host = checkNotNull(host);
    }

    /**
     * Получить джагглерное событие для переданного значения
     *
     * @param value          значение, которое нужно сравнить с пределами проверки
     * @param rawServiceName имя сырого события, одно из отдаваемых методом getRawServiceNames
     * @param subMessage     дополнительное сообщение для модификации базового описания созданного события
     */
    protected JugglerEvent getEvent(@Nullable Long value, String rawServiceName, @Nullable String subMessage) {
        checkArgument(getRawServiceNames().contains(rawServiceName), "Service name should be predefined");

        String message;
        JugglerStatus status;
        if (value != null) {
            String template = ascending ? ASCENDING_MESSAGE_TEMPLATE : DESCENDING_MESSAGE_TEMPLATE;
            int passCompareResult = ascending ? -1 : 1;

            if (critBorder != null && value.compareTo(critBorder) != passCompareResult) {
                message = String.format(template, value, critBorder);
                status = JugglerStatus.CRIT;
            } else if (warnBorder != null && value.compareTo(warnBorder) != passCompareResult) {
                message = String.format(template, value, warnBorder);
                status = JugglerStatus.WARN;
            } else {
                message = String.valueOf(value);
                status = JugglerStatus.OK;
            }
        } else {
            message = "undefined value";
            status = JugglerStatus.CRIT;
        }

        message = String.format("%s: %s", description, message);
        if (subMessage != null) {
            message = String.format("%s (%s)", message, subMessage);
        }
        return new JugglerEvent(host, rawServiceName, status, message);
    }

    /**
     * Получить имя результирующей проверки
     */
    public String getServiceName() {
        return serviceName;
    }

    /**
     * Получить хост результирующей проверки
     */
    public String getHost() {
        return host;
    }

    /**
     * Получить набор названий сырых событий которые вместе составляют основную проверку
     */
    public abstract List<String> getRawServiceNames();
}
