package ru.yandex.direct.juggler.check;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.direct.env.EnvironmentTypeProvider;
import ru.yandex.direct.juggler.JugglerEvent;
import ru.yandex.direct.juggler.JugglerSender;
import ru.yandex.direct.juggler.check.checkinfo.NumericCheckInfo;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Клиент для отправки событий в джагглер, перед отправкой всегда добавляет имя среды к их названию
 */
@Lazy
@Component
@ParametersAreNonnullByDefault
public class JugglerNumericEventsClient {
    private static final Logger logger = LoggerFactory.getLogger(JugglerNumericEventsClient.class);

    private final JugglerSender jugglerSender;
    private final EnvironmentTypeProvider environmentTypeProvider;

    @Autowired
    public JugglerNumericEventsClient(JugglerSender jugglerSender, EnvironmentTypeProvider environmentTypeProvider) {
        this.jugglerSender = jugglerSender;
        this.environmentTypeProvider = environmentTypeProvider;
    }

    /**
     * Отправить одно событие в джагглер с заданным таимаутом, к названию события будет добавлено имя среды исполнения
     */
    public void sendEvent(JugglerEvent jugglerEvent, Duration timeout) {
        sendEvents(Collections.singleton(jugglerEvent), timeout);
    }

    /**
     * Отправить несколько событий в джагглер с заданным таимаутом, к названию события будет добавлено имя среды исполнения
     */
    public void sendEvents(Collection<JugglerEvent> jugglerEvents, Duration timeout) {
        String envName = environmentTypeProvider.get().getLegacyName();
        List<JugglerEvent> events = mapList(jugglerEvents,
                event -> event.withService(NumericCheckInfo.getNameWithEnv(event.getService(), envName)));
        logger.debug("Sending juggler events to juggler: {}", events);

        events.forEach(event -> jugglerSender.sendEvent(event, timeout));
    }
}
