package ru.yandex.direct.juggler.check.checkinfo;

import java.time.Duration;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import ru.yandex.direct.ansiblejuggler.model.meta.JugglerMetaUrl;
import ru.yandex.direct.ansiblejuggler.model.notifications.JugglerNotification;
import ru.yandex.direct.ansiblejuggler.model.notifications.OnChangeNotification;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.Url;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.utils.Condition;

/**
 * Контейнер с информацией, необходимой для создания шаблона juggler-проверки
 */
public abstract class AnnotationBasedJugglerCheckInfo implements BaseJugglerCheckInfo {
    private final JugglerCheck annotation;

    public AnnotationBasedJugglerCheckInfo(JugglerCheck annotation) {
        this.annotation = annotation;
    }

    @Override
    public Class<? extends Condition> getEnvironmentConditionClass() {
        return annotation.needCheck();
    }

    /**
     * @return TTL проверки
     */
    public Duration getTTL() {
        return Duration.ZERO.plusDays(annotation.ttl().days())
                .plusHours(annotation.ttl().hours())
                .plusMinutes(annotation.ttl().minutes())
                .plusSeconds(annotation.ttl().seconds());
    }

    /**
     * @return массив тегов для проверки
     */
    public List<CheckTag> getTags() {
        return Arrays.asList(annotation.tags());
    }

    /**
     * @return список нотификаций для текущей проверки
     */
    public List<JugglerNotification> getNotifications() {
        return Arrays.stream(annotation.notifications())
                .map(this::convertOnChangeNotification)
                .collect(Collectors.toList());
    }

    private OnChangeNotification convertOnChangeNotification(
            ru.yandex.direct.juggler.check.annotation.OnChangeNotification annotation) {
        return new OnChangeNotification(annotation.type())
                .withRecipients(Arrays.stream(annotation.recipient())
                        .map(NotificationRecipient::getName)
                        .collect(Collectors.toList()))
                .withNotificationMethods(annotation.method())
                .withStatuses(annotation.status());
    }

    /**
     * @return список полезных ссылок для данной проверки
     */
    public List<JugglerMetaUrl> getUrls() {
        return Arrays.stream(annotation.urls())
                .map(this::convertUrl)
                .collect(Collectors.toList());
    }

    private JugglerMetaUrl convertUrl(Url url) {
        return new JugglerMetaUrl(url.title(), url.url(), url.type());
    }
}
