package ru.yandex.direct.logviewercore.domain;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import com.google.common.collect.ImmutableList;

import ru.yandex.direct.logviewercore.domain.ppclog.LogField;
import ru.yandex.direct.logviewercore.domain.ppclog.LogRecord;
import ru.yandex.direct.logviewercore.domain.ppclog.LogTable;
import ru.yandex.direct.logviewercore.service.virtual.VirtualColumn;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Даёт возможность инспектировать классы таблиц в рантайме
 */
public class LogRecordInfo<T extends LogRecord> {
    private final Class<? extends T> cls;
    private final String tableName;
    private final String arrayJoin;
    private final String logTimeColumn;
    private final String logDateColumn;
    private final String reqidColumn;
    private final String traceIdColumn;
    private final Map<String, Field> fields;
    private final BiMap<String, String> fieldToDbColumn;
    private final Set<String> heavyFields;
    private final Set<String> prewhereFields;
    private final Set<String> selectiveFields;
    private final List<String> additionalSort;
    private final Map<String, Class<? extends VirtualColumn>> virtualColumns;
    private final boolean encodingBroken;
    private final String[] allowAccessFor;

    @SuppressWarnings("checkstyle:hiddenfield")
    public LogRecordInfo(Class<? extends T> cls) {
        LogTable logTable = cls.getAnnotation(LogTable.class);
        checkNotNull(logTable, "Class " + cls.getName() + " is missing a LogTable annotation");

        this.cls = cls;
        this.tableName = logTable.tableName();
        this.arrayJoin = logTable.arrayJoin();
        this.logTimeColumn = logTable.logTimeColumn();
        this.logDateColumn = logTable.logDateColumn();
        this.reqidColumn = logTable.reqidColumn();
        this.traceIdColumn = logTable.traceIdColumn();
        this.encodingBroken = logTable.encodingBroken();
        this.additionalSort = ImmutableList.copyOf(logTable.additionalSort());

        HashMap<String, Field> fields = new HashMap<>();
        HashSet<String> heavyFields = new HashSet<>();
        HashSet<String> prewhereFields = new HashSet<>();
        HashSet<String> selectiveFields = new HashSet<>();
        ImmutableBiMap.Builder<String, String> fieldToDbColumn = new ImmutableBiMap.Builder<>();
        ImmutableBiMap.Builder<String, Class<? extends VirtualColumn>> virtualColumnsBuilder =
                new ImmutableBiMap.Builder<>();
        for (Field field : cls.getFields()) {
            String name = field.getName();
            if (name.startsWith("_")) {
                name = name.substring(1);
            }
            fields.put(name, field);

            LogField ann = field.getAnnotation(LogField.class);
            if (ann != null) {
                fieldToDbColumn.put(name, ann.columnName().isEmpty() ? name : ann.columnName());
                if (ann.heavy()) {
                    heavyFields.add(name);
                }
                if (ann.prewhere()) {
                    prewhereFields.add(name);
                }
                if (ann.selective()) {
                    selectiveFields.add(name);
                }
                if (ann.virtual() != VirtualColumn.class) {
                    virtualColumnsBuilder.put(name, ann.virtual());
                }
            }
        }
        this.fields = Collections.unmodifiableMap(fields);
        this.heavyFields = Collections.unmodifiableSet(heavyFields);
        this.prewhereFields = Collections.unmodifiableSet(prewhereFields);
        this.selectiveFields = Collections.unmodifiableSet(selectiveFields);
        this.fieldToDbColumn = fieldToDbColumn.build();
        this.virtualColumns = virtualColumnsBuilder.build();
        this.allowAccessFor = logTable.allowAccessFor();
    }

    /**
     * @return новый экземпляр данных таблицы
     */
    public T newInstance() {
        try {
            return cls.newInstance();
        } catch (InstantiationException | IllegalAccessException e) {
            throw new IllegalArgumentException("Cannot create instances of " + cls.getName(), e);
        }
    }

    /**
     * @return класс для данных таблицы
     */
    public Class<? extends T> getInstanceClass() {
        return cls;
    }

    /**
     * @return имя лога
     */
    public String getName() {
        return cls.getSimpleName();
    }

    /**
     * @return имя таблицы
     */
    public String getTableName() {
        return tableName;
    }

    /**
     * @return имя столбца, с которым нужно делать array join
     */
    public String getArrayJoin() {
        return arrayJoin;
    }

    /**
     * Название столбца с дата/время
     *
     * @return
     */
    public String getLogTimeColumn() {
        return logTimeColumn;
    }

    /**
     * Название столбца с датой
     *
     * @return
     */
    public String getLogDateColumn() {
        return logDateColumn;
    }

    public String getReqidColumn() {
        return reqidColumn;
    }

    public String getTraceIdColumn() {
        return traceIdColumn;
    }

    public String getDbColumn(String column) {
        return fieldToDbColumn.getOrDefault(column, column);
    }

    public String getColimnByDbColumn(String column) {
        return fieldToDbColumn.inverse().getOrDefault(column, column);
    }

    /**
     * @return broken encoding flag {@see LogTable}
     */
    public boolean isEncodingBroken() {
        return encodingBroken;
    }

    /**
     * @return список известных колонок таблицы
     */
    public List<String> getColumnNames() {
        List<String> columns = new ArrayList<>();
        for (Field field : cls.getFields()) {
            String name = field.getName();
            if (name.startsWith("_")) {
                name = name.substring(1);
            }
            columns.add(name);
        }
        return columns;
    }

    /**
     * @return Field для колонки с именем logName
     */
    public Field getColumnField(String name) {
        Field field = fields.get(name);
        if (field == null) {
            throw new IllegalArgumentException("Unknown column " + name);
        }
        return field;
    }

    /**
     * Returns whether field marked as "heavy" {@see LogField}
     */
    public boolean isHeavyField(String name) {
        return heavyFields.contains(name);
    }

    /**
     * Returns whether table has any field, marked as "selective" {@see LogField}
     */
    public boolean hasSelectiveFields() {
        return !selectiveFields.isEmpty();
    }

    /**
     * Returns set of field names, marked as selective
     */
    public Set<String> getSelectiveFields() {
        return selectiveFields;
    }

    /**
     * {@see LogField#prewhere}
     */
    public boolean isPrewhereField(String name) {
        return prewhereFields.contains(name);
    }

    public Optional<Class<? extends VirtualColumn>> findVirtual(String name) {
        return Optional.ofNullable(virtualColumns.get(name));
    }

    /**
     * {@see LogTable#additionalSort}
     */
    public List<String> getAdditionalSort() {
        return additionalSort;
    }

    public String[] getAllowAccessFor() {
        return allowAccessFor;
    }
}
