package ru.yandex.direct.logviewercore.domain;

import java.lang.reflect.Field;
import java.sql.Date;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.springframework.jdbc.core.RowMapper;

import ru.yandex.clickhouse.response.ClickHouseResultSet;
import ru.yandex.direct.common.util.EncodingUtils;
import ru.yandex.direct.logviewercore.domain.ppclog.LogRecord;

/**
 * RowMapper implementation for make partially initialized LogRecord* objects
 * from resultsets
 *
 * @param <T> - concrete LogRecord inheritor
 */
public class LogRecordRowMapper<T extends LogRecord> implements RowMapper<T> {
    private List<Field> selected = new ArrayList<>();
    private List<ColumnGetter> getters = new ArrayList<>();
    private LogRecordInfo<? extends T> info;

    public LogRecordRowMapper(List<String> fields, LogRecordInfo<? extends T> info) {
        for (String name : fields) {
            Field field = info.getColumnField(name);
            selected.add(field);
            getters.add(ColumnGetter.forClass(field.getType(), info.isEncodingBroken()));
        }
        this.info = info;
    }

    /**
     * Create T object from current row in resultset rs
     *
     * @param rs
     * @param i
     * @return
     * @throws SQLException
     */
    @Override
    public T mapRow(ResultSet rs, int i) throws SQLException {
        T row = info.newInstance();
        for (int index = 0; index < selected.size(); ++index) {
            Field field = selected.get(index);
            Object value = getters.get(index).getResultItem(rs, index + 1);
            try {
                selected.get(index).set(row, value);
            } catch (IllegalAccessException e) {
                throw new IllegalArgumentException("Cannot set field " + field.getName(), e);
            }
        }
        return row;
    }

    /**
     * Read result set row to list of objects (with respect to types)
     *
     * @param rs
     * @param i
     * @return
     * @throws SQLException
     */
    public List<Object> mapRowToObjectsList(ResultSet rs, int i) throws SQLException {
        List<Object> ret = new ArrayList<>(selected.size());
        for (int index = 0; index < selected.size(); ++index) {
            Object value = getters.get(index).getResultItem(rs, index + 1);
            ret.add(value);
        }
        return ret;
    }

    @FunctionalInterface
    private interface ColumnGetter {
        Object getResultItem(ResultSet rs, int index) throws SQLException;

        static ColumnGetter forClass(Class<?> cls, boolean encodingBroken) {
            if (cls.isAssignableFrom(String.class)) {
                return (rs, i) -> {
                    if (encodingBroken) {
                        return EncodingUtils.recoverUtf8Decode(rs.getString(i));
                    } else {
                        return rs.getString(i);
                    }
                };
            }
            if (cls.isAssignableFrom(Byte.class) || cls.isAssignableFrom(byte.class)) {
                return ResultSet::getByte;
            }
            if (cls.isAssignableFrom(Integer.class) || cls.isAssignableFrom(int.class)) {
                return ResultSet::getInt;
            }
            if (cls.isAssignableFrom(Long.class) || cls.isAssignableFrom(long.class)) {
                return ResultSet::getLong;
            }
            if (cls.isAssignableFrom(Float.class) || cls.isAssignableFrom(float.class)) {
                return ResultSet::getFloat;
            }
            if (cls.isAssignableFrom(Double.class) || cls.isAssignableFrom(double.class)) {
                return ResultSet::getDouble;
            }
            if (cls.isAssignableFrom(Date.class)) {
                return ResultSet::getDate;
            }
            if (cls.isAssignableFrom(Timestamp.class)) {
                return ResultSet::getTimestamp;
            }
            if (cls.isAssignableFrom(int[].class)) {
                return (rs, i) -> {
                    String v = rs.getString(i);
                    if ("[]".equals(v)) {
                        return new int[0];
                    }
                    return Arrays
                            .stream(v.substring(1, v.length() - 1).split(","))
                            .map(String::trim)
                            .mapToInt(Integer::parseInt)
                            .toArray();
                };
            }
            if (cls.isAssignableFrom(long[].class)) {
                return (rs, i) -> {
                    // FIXME: clickhouse-jdbc cannot decode empty arrays
                    String v = rs.getString(i);
                    if ("[]".equals(v)) {
                        return new long[0];
                    }
                    return getClickHouseResultSet(rs).getLongArray(i);
                };
            }
            if (cls.isAssignableFrom(double[].class)) {
                return (rs, i) -> {
                    String v = rs.getString(i);
                    if ("[]".equals(v)) {
                        return new double[0];
                    }
                    return Arrays
                            .stream(v.substring(1, v.length() - 1).split(","))
                            .map(String::trim)
                            .mapToDouble(Double::parseDouble)
                            .toArray();
                };
            }
            if (cls.isAssignableFrom(String[].class)) {
                return (rs, i) -> (String[]) getClickHouseResultSet(rs).getArray(i).getArray();
            }
            if (cls.isAssignableFrom(boolean.class)) {
                return ResultSet::getBoolean;
            }
            throw new IllegalArgumentException("Cannot decode " + cls.toString());
        }
    }

    private static ClickHouseResultSet getClickHouseResultSet(ResultSet rs) throws SQLException {
        ClickHouseResultSet chrs;
        if (rs instanceof ClickHouseResultSet) {
            chrs = (ClickHouseResultSet) rs;
        } else {
            chrs = rs.unwrap(ClickHouseResultSet.class);
        }
        return chrs;
    }
}
