package ru.yandex.direct.logviewercore.domain;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.jetbrains.annotations.NotNull;
import org.reflections.Reflections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.logviewercore.domain.ppclog.LogRecord;
import ru.yandex.direct.logviewercore.domain.ppclog.LogTable;

/**
 * Information about log types (depends on loaded classes)
 */
public class LogTablesInfoManager {
    private LogTablesInfoManager() {
    }

    /**
     * Lazy collector of loaded log types
     */
    private static class LazyHolder {
        private static final Logger logger = LoggerFactory.getLogger(LazyHolder.class);
        private static Map<String, Class<? extends LogRecord>> tables = getTables();

        private LazyHolder() {
        }

        private static Map<String, Class<? extends LogRecord>> getTables() {
            logger.debug("map initialization");
            Map<String, Class<? extends LogRecord>> tmp = new HashMap<>();
            Package logRecordPackage = LogRecord.class.getPackage();
            Reflections reflections = new Reflections(logRecordPackage.getName());

            for (Class<? extends LogRecord> cls : reflections.getSubTypesOf(LogRecord.class)) {
                LogTable ann = cls.getAnnotation(LogTable.class);
                if (ann != null) {
                    tmp.put(ann.logName(), cls);
                }
            }
            return Collections.unmodifiableMap(tmp);
        }
    }

    /**
     * Get all loaded log types
     *
     * @return Map of logNames refers to relevant log record class
     */
    public static Map<String, Class<? extends LogRecord>> allTableClasses() {
        return LazyHolder.tables;
    }

    /**
     * Get LogRecordInfo object for defined logName
     *
     * @param logName
     * @return
     */
    public static LogRecordInfo<LogRecord> getLogRecordInfo(String logName) {
        Class<? extends LogRecord> clazz = getTableClass(logName);
        if (clazz == null) {
            throw new IllegalArgumentException("unsupported log " + logName);
        }
        return new LogRecordInfo<>(clazz);
    }

    public static <T extends LogRecord> LogRecordInfo<T> getLogRecordInfo(@NotNull Class<T> clazz) {
        return new LogRecordInfo<>(clazz);
    }

    private static Class<? extends LogRecord> getTableClass(String logName) {
        return LazyHolder.tables.get(logName);
    }
}
