package ru.yandex.direct.logviewercore.domain.web;

import java.util.List;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.direct.tracing.Trace;

/**
 * Response to logviewer filter query
 */
public class FilterResponse {
    private final String error;
    private final List<List<Object>> data;
    private final long totalCount;
    private final String sql;
    private final long reqid = Trace.current().getSpanId();
    private String queryId;

    /**
     * Create response object with result data grid and empty error
     *
     * @param data
     * @param totalCount
     */
    public FilterResponse(List<List<Object>> data, long totalCount, String sql) {
        this(null, data, totalCount, sql, null);
    }

    /**
     * Create response object with error message and without data
     *
     * @param error
     */
    public FilterResponse(String error) {
        this(error, null, 0, null, null);
    }

    @JsonCreator
    private FilterResponse(
            @JsonProperty("error") String error,
            @JsonProperty("data") List<List<Object>> data,
            @JsonProperty("totalCount") long totalCount,
            @JsonProperty("sql") String sql,
            @JsonProperty("queryId") String queryId
    ) {
        this.error = error;
        this.data = data;
        this.totalCount = totalCount;
        this.sql = sql;
        this.queryId = queryId;
    }

    /**
     * Get resulting data grid, can be null if error occurred
     *
     * @return
     */
    public List<List<Object>> getData() {
        return data;
    }

    /**
     * Get error message, null if no error
     *
     * @return
     */
    public String getError() {
        return error;
    }

    public long getTotalCount() {
        return totalCount;
    }

    public String getSql() {
        return sql;
    }

    public long getReqid() {
        return reqid;
    }

    public String getQueryId() {
        return queryId;
    }

    public void setQueryId(long queryId) {
        this.queryId = Long.toString(queryId);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }
}
