package ru.yandex.direct.logviewercore.repository;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.builder.QueryAndParams;
import ru.yandex.direct.ydb.builder.predicate.Predicate;
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder;
import ru.yandex.direct.ydb.builder.querybuilder.OrderByBuilder;
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder;
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder;
import ru.yandex.direct.ydb.client.DataQueryResultWrapper;
import ru.yandex.direct.ydb.client.ResultSetReaderWrapped;
import ru.yandex.direct.ydb.client.YdbClient;
import ru.yandex.direct.ydb.column.Column;

import static ru.yandex.direct.common.configuration.FrontDbYdbConfiguration.FRONTDB_YDB_CLIENT_BEAN;
import static ru.yandex.direct.common.configuration.FrontDbYdbConfiguration.FRONTDB_YDB_PATH_BEAN;
import static ru.yandex.direct.logviewercore.repository.Tables.LOGVIEWER_HISTORY;
import static ru.yandex.direct.ydb.table.temptable.TempTable.tempTable;

@Lazy
@Repository
@ParametersAreNonnullByDefault
public class LogviewerRepository {

    private final YdbPath path;
    private final YdbClient ydbClient;

    @Autowired
    public LogviewerRepository(
            @Qualifier(FRONTDB_YDB_CLIENT_BEAN) YdbClient ydbClient,
            @Qualifier(FRONTDB_YDB_PATH_BEAN) YdbPath path
    ) {
        this.path = path;
        this.ydbClient = ydbClient;
    }

    public void save(LogviewerHistoryRow row) {
        var tempTable = tempTable(
                LOGVIEWER_HISTORY.QUERY_ID,
                LOGVIEWER_HISTORY.LIMIT,
                LOGVIEWER_HISTORY.OFFSET,
                LOGVIEWER_HISTORY.CLIENT_ID,
                LOGVIEWER_HISTORY.START_TIME,
                LOGVIEWER_HISTORY.STATUS,
                LOGVIEWER_HISTORY.TEMPLATE,
                LOGVIEWER_HISTORY.PERSISTENT,
                LOGVIEWER_HISTORY.FAVOURITE,
                LOGVIEWER_HISTORY.NAME,
                LOGVIEWER_HISTORY.EXECUTION_TIME,
                LOGVIEWER_HISTORY.VERSION,
                LOGVIEWER_HISTORY.REQUEST,
                LOGVIEWER_HISTORY.COMPRESSED_RESPONSE)
                .createValues()
                .fill(
                        row.queryId,
                        row.limit,
                        row.offset,
                        row.clientId,
                        row.startTime,
                        row.status,
                        row.template,
                        row.persistent,
                        row.favourite,
                        row.name,
                        row.executionTime,
                        row.version,
                        row.request,
                        row.compressedResponse
                );
        var queryAndParams = InsertBuilder.upsertInto(LOGVIEWER_HISTORY)
                .selectAll()
                .from(tempTable)
                .queryAndParams(path);
        ydbClient.executeQuery(queryAndParams);
    }

    public LogviewerHistoryRow get(long queryId, long offset, long limit) {
        QueryAndParams queryAndParams = SelectBuilder.select(LOGVIEWER_HISTORY.getColumns())
                .from(LOGVIEWER_HISTORY)
                .where(LOGVIEWER_HISTORY.QUERY_ID.eq(queryId))
                .orderBy(LOGVIEWER_HISTORY.OFFSET, LOGVIEWER_HISTORY.LIMIT)
                .queryAndParams(path);

        DataQueryResultWrapper result = ydbClient.executeQuery(queryAndParams);
        ResultSetReaderWrapped resultSet = result.getResultSet(0);

        LogviewerHistoryRow row = null;
        while (resultSet.next()) {
            LogviewerHistoryRow current = resultSetToLogviewerHistoryRow(resultSet);
            if (row == null) {
                row = current;
            }
            if (current.offset > offset || (current.offset == offset && current.limit > limit)) {
                break;
            }
            if (current.offset == offset && current.limit == limit) {
                row = current;
                break;
            }
        }

        return row;
    }

    public LogviewerHistoryRow get(long queryId) {
        QueryAndParams queryAndParams = SelectBuilder.select(LOGVIEWER_HISTORY.getColumns())
                .from(LOGVIEWER_HISTORY)
                .where(LOGVIEWER_HISTORY.QUERY_ID.eq(queryId))
                .orderBy(LOGVIEWER_HISTORY.OFFSET, LOGVIEWER_HISTORY.LIMIT)
                .limit(1)
                .queryAndParams(path);

        DataQueryResultWrapper result = ydbClient.executeQuery(queryAndParams);
        ResultSetReaderWrapped resultSet = result.getResultSet(0);

        if (resultSet.next()) {
            return resultSetToLogviewerHistoryRow(resultSet);
        }

        return null;
    }

    private static LogviewerHistoryRow resultSetToLogviewerHistoryRow(ResultSetReaderWrapped resultSet) {
        LogviewerHistoryRow row = new LogviewerHistoryRow();
        row.queryId = resultSet.getValueReader(LOGVIEWER_HISTORY.QUERY_ID).getUint64();
        row.limit = resultSet.getValueReader(LOGVIEWER_HISTORY.LIMIT).getInt32();
        row.offset = resultSet.getValueReader(LOGVIEWER_HISTORY.OFFSET).getInt32();
        row.clientId = resultSet.getValueReader(LOGVIEWER_HISTORY.CLIENT_ID).getUint64();
        row.startTime = resultSet.getValueReader(LOGVIEWER_HISTORY.START_TIME).getUint64();
        row.status = resultSet.getValueReader(LOGVIEWER_HISTORY.STATUS).getUtf8();
        row.template = resultSet.getValueReader(LOGVIEWER_HISTORY.TEMPLATE).getUtf8();
        row.persistent = resultSet.getValueReader(LOGVIEWER_HISTORY.PERSISTENT).getBool();
        row.favourite = resultSet.getValueReader(LOGVIEWER_HISTORY.FAVOURITE).getBool();
        row.name = resultSet.getValueReader(LOGVIEWER_HISTORY.NAME).getUtf8();
        row.executionTime = resultSet.getValueReader(LOGVIEWER_HISTORY.EXECUTION_TIME).getInt32();
        row.version = resultSet.getValueReader(LOGVIEWER_HISTORY.VERSION).getInt32();
        row.request = resultSet.getValueReader(LOGVIEWER_HISTORY.REQUEST).getJson();
        row.compressedResponse = resultSet.getValueReader(LOGVIEWER_HISTORY.COMPRESSED_RESPONSE).getString();
        return row;
    }

    public List<LogviewerHistoryRow> get(
            long clientId, long startAfter, long startBefore,
            List<String> statuses, @Nullable String template, boolean onlyFavourite, int limit
    ) {
        Predicate condition = LOGVIEWER_HISTORY.CLIENT_ID.eq(clientId)
                .and(LOGVIEWER_HISTORY.START_TIME.ge(startAfter))
                .and(LOGVIEWER_HISTORY.START_TIME.le(startBefore));
        if (!statuses.isEmpty()) {
            condition = condition.and(LOGVIEWER_HISTORY.STATUS.in(statuses));
        }
        if (onlyFavourite) {
            condition = condition.and(LOGVIEWER_HISTORY.FAVOURITE.eq(true));
        }
        if (StringUtils.isNotEmpty(template) && StringUtils.isNotEmpty(template.trim())) {
            condition = condition.and(LOGVIEWER_HISTORY.TEMPLATE.eq(template.trim()));
        }
        QueryAndParams queryAndParams = SelectBuilder.select(
                LOGVIEWER_HISTORY.QUERY_ID, LOGVIEWER_HISTORY.START_TIME,
                LOGVIEWER_HISTORY.STATUS, LOGVIEWER_HISTORY.FAVOURITE,
                LOGVIEWER_HISTORY.NAME, LOGVIEWER_HISTORY.TEMPLATE,
                LOGVIEWER_HISTORY.OFFSET, LOGVIEWER_HISTORY.LIMIT
        )
                .from(LOGVIEWER_HISTORY)
                .where(condition)
                .orderBy(OrderByBuilder.OrderType.DESC, LOGVIEWER_HISTORY.START_TIME)
                .limit(limit)
                .queryAndParams(path);

        DataQueryResultWrapper result = ydbClient.executeQuery(queryAndParams);
        ResultSetReaderWrapped resultSet = result.getResultSet(0);

        List<LogviewerHistoryRow> rows = new ArrayList<>(resultSet.getRowCount());
        Set<Long> queryIds = new HashSet<>();
        while (resultSet.next()) {
            Long queryId = resultSet.getValueReader(LOGVIEWER_HISTORY.QUERY_ID).getUint64();
            if (!queryIds.add(queryId)) {
                continue;
            }
            LogviewerHistoryRow row = new LogviewerHistoryRow();
            row.queryId = queryId;
            row.startTime = resultSet.getValueReader(LOGVIEWER_HISTORY.START_TIME).getUint64();
            row.status = resultSet.getValueReader(LOGVIEWER_HISTORY.STATUS).getUtf8();
            row.favourite = resultSet.getValueReader(LOGVIEWER_HISTORY.FAVOURITE).getBool();
            row.name = resultSet.getValueReader(LOGVIEWER_HISTORY.NAME).getUtf8();
            row.template = resultSet.getValueReader(LOGVIEWER_HISTORY.TEMPLATE).getUtf8();
            rows.add(row);
        }

        return rows;
    }

    public void update(
            long queryId,
            @Nullable ClientId clientId,
            @Nullable String name,
            @Nullable Boolean persistent,
            @Nullable Boolean favourite
    ) {
        if (clientId == null) {
            return;
        }
        if (name == null && persistent == null && favourite == null) {
            return;
        }
        Map<Column, Object> values = EntryStream.<Column, Object>of(
                LOGVIEWER_HISTORY.NAME, name,
                LOGVIEWER_HISTORY.PERSISTENT, persistent,
                LOGVIEWER_HISTORY.FAVOURITE, favourite
        ).nonNullValues().toMap();
        UpdateBuilder.SetStatement setStatement = null;
        for (Map.Entry<Column, Object> entry : values.entrySet()) {
            if (setStatement == null) {
                setStatement = UpdateBuilder.set(entry.getKey(), entry.getValue());
            } else {
                setStatement.set(entry.getKey(), entry.getValue());
            }
        }
        QueryAndParams queryAndParams = UpdateBuilder.update(LOGVIEWER_HISTORY, setStatement)
                .where(LOGVIEWER_HISTORY.QUERY_ID.eq(queryId)
                        .and(LOGVIEWER_HISTORY.CLIENT_ID.eq(clientId.asLong()))
                ).queryAndParams(path);
        ydbClient.executeQuery(queryAndParams);
    }

    public void cleanResponse(Long startLimitNotShared, Long startLimitShared) {
        if (startLimitNotShared == null && startLimitShared == null) {
            return;
        }
        Predicate condition = null;
        if (startLimitNotShared != null) {
            condition = LOGVIEWER_HISTORY.PERSISTENT.eq(false)
                    .and(LOGVIEWER_HISTORY.START_TIME.le(startLimitNotShared));
        }
        if (startLimitShared != null) {
            var appendCondition = LOGVIEWER_HISTORY.PERSISTENT.eq(true)
                    .and(LOGVIEWER_HISTORY.START_TIME.le(startLimitShared));
            condition = condition == null ? appendCondition : condition.or(appendCondition);
        }
        QueryAndParams queryAndParams = UpdateBuilder
                .update(LOGVIEWER_HISTORY, UpdateBuilder.set(LOGVIEWER_HISTORY.COMPRESSED_RESPONSE, new byte[0]))
                .where(condition)
                .queryAndParams(path);
        ydbClient.executeQuery(queryAndParams);
    }
}
