package ru.yandex.direct.logviewercore.service;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.logviewercore.domain.web.FilterResponse;
import ru.yandex.direct.logviewercore.domain.web.HistoryItem;
import ru.yandex.direct.logviewercore.domain.web.LogViewerFilterForm;
import ru.yandex.direct.logviewercore.repository.LogviewerHistoryRow;
import ru.yandex.direct.logviewercore.repository.LogviewerRepository;
import ru.yandex.direct.tracing.Trace;

import static ru.yandex.direct.common.db.PpcPropertyNames.DAYS_TO_LIVE_OF_RESPONSE_FOR_NOT_SHARED_LOGVIEWER_QUERY;
import static ru.yandex.direct.common.db.PpcPropertyNames.DAYS_TO_LIVE_OF_RESPONSE_FOR_SHARED_LOGVIEWER_QUERY;

@Service
public class CacheLogViewerResultService {

    private static final int MILLIS_IN_DAY = 24 * 60 * 60 * 1000;
    private static final DateTimeFormatter DATA_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd'T'HHmmss");

    private static final Logger logger = LoggerFactory.getLogger(CacheLogViewerResultService.class);

    private final LogviewerRepository logviewerRepository;
    private final PpcProperty<Integer> daysToLiveNotSharedProperty;
    private final PpcProperty<Integer> daysToLiveSharedProperty;

    @Autowired
    public CacheLogViewerResultService(
            LogviewerRepository logviewerRepository,
            PpcPropertiesSupport ppcPropertiesSupport
    ) {
        this.logviewerRepository = logviewerRepository;
        daysToLiveNotSharedProperty = ppcPropertiesSupport.get(DAYS_TO_LIVE_OF_RESPONSE_FOR_NOT_SHARED_LOGVIEWER_QUERY);
        daysToLiveSharedProperty = ppcPropertiesSupport.get(DAYS_TO_LIVE_OF_RESPONSE_FOR_SHARED_LOGVIEWER_QUERY);
    }

    public long initHistoryRow(String logName, LogViewerFilterForm form, long startTime) {
        long queryId = Trace.current().getSpanId();
        try {
            logviewerRepository.save(LogviewerHistoryRow.createStartedRow(
                    queryId, logName, getClientId(), form, startTime
            ));
        } catch (RuntimeException ex) {
            logger.warn(String.format("Query[id=%s] is running, but status is not saved in history.", queryId), ex);
        }
        return queryId;
    }

    public void save(long queryId, String logName, LogViewerFilterForm form, FilterResponse response, long startTime, int executionTime) {
        try {
            logviewerRepository.save(LogviewerHistoryRow.createFinishedRow(
                    queryId, logName, getClientId(), form, response, startTime, executionTime
            ));
        } catch (RuntimeException ex) {
            logger.warn(String.format("Query[id=%s] is finished, but status is not updated in history.", queryId), ex);
        }
    }

    public List<HistoryItem> get(
            long startAfter, long startBefore,
            List<String> statuses, @Nullable String template, boolean onlyFavourite, int limit
    ) {
        ClientId clientId = getClientId();
        return clientId == null ? Collections.emptyList() : StreamEx.of(logviewerRepository.get(
                clientId.asLong(), startAfter, startBefore,
                statuses, template, onlyFavourite, limit
                ))
                .map(HistoryItem::createInstance)
                .toList();
    }

    public HistoryItem get(long queryId) {
        return HistoryItem.createInstance(logviewerRepository.get(queryId));
    }

    public HistoryItem get(long queryId, long offset, long limit) {
        return HistoryItem.createInstance(logviewerRepository.get(queryId, offset, limit));
    }

    private ClientId getClientId() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (authentication instanceof DirectAuthentication) {
            return ((DirectAuthentication) authentication).getSubjectUser().getClientId();
        }
        return null;
    }

    public void updateQuery(long queryId, String name, Boolean shared, Boolean favourite) {
        Boolean persistent = Boolean.TRUE.equals(shared) ? shared : null;
        logviewerRepository.update(queryId, getClientId(), name, persistent, favourite);
    }

    public void clean() {
        int ttlNotShared = daysToLiveNotSharedProperty.getOrDefault(30);
        int ttlShared = daysToLiveSharedProperty.getOrDefault(365);
        if (ttlNotShared < 0 && ttlShared < 0) {
            return;
        }
        long now = System.currentTimeMillis();
        Long startLimitNotShared = ttlNotShared < 0 ? null : now - ttlNotShared * MILLIS_IN_DAY;
        Long startLimitShared = ttlShared < 0 ? null : now - ttlShared * MILLIS_IN_DAY;
        logviewerRepository.cleanResponse(startLimitNotShared, startLimitShared);
    }

    public String getWebQueryForm(long queryId) {
        HistoryItem cached = get(queryId);
        if (cached == null) {
            return null;
        }
        var form = cached.getRequest();
        StringBuilder query = new StringBuilder("")
                .append("~(logType~'").append(cached.getLogName())
                .append("~form~(from~'").append(form.getFrom().format(DATA_TIME_FORMATTER))
                .append("~to~'").append(form.getTo().format(DATA_TIME_FORMATTER))
                .append("~fields~(");
        StreamEx.of(form.getFields())
                .forEach(field -> query
                        .append("~'")
                        .append(field));
        query
                .append(")~conditions~(");
        EntryStream.of(form.getConditions())
                .forEach(cond -> query
                        .append(cond.getKey())
                        .append("~'")
                        .append(decodeValue(cond.getValue()))
                        .append("~"));
        if (form.getConditions().size() > 0) {
            query.setLength(query.length() - 1);
        }
        query
                .append(")~limit~").append(form.getLimit())
                .append("~offset~").append(form.getOffset())
                .append("~reverseOrder~").append(form.isReverseOrder())
                .append("~showTraceIdRelated~").append(form.isShowTraceIdRelated())
                .append("~showStats~").append(form.isShowStats())
                .append("~sortByCount~").append(form.isSortByCount())
                .append("~logTimeGroupBy~'").append(form.getLogTimeGroupBy())
                .append("))");
        return query.toString();
    }

    private static String decodeValue(String value) {
        try {
            return URLEncoder
                    .encode(value, StandardCharsets.UTF_8)
                    .replaceAll("\\+", "%20")
                    .replaceAll("\\%", "*");
        } catch (Exception ex) {
            logger.warn("Can't decode condition " + value, ex);
            return "";
        }
    }
}
