package ru.yandex.direct.logviewercore.service;

import java.lang.reflect.Field;
import java.sql.Date;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.clickhouse.ClickHousePreparedStatement;
import ru.yandex.clickhouse.ClickHouseUtil;
import ru.yandex.clickhouse.response.ClickHouseResponse;
import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.common.util.EncodingUtils;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.dbutil.wrapper.SimpleDb;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.logviewercore.container.RowsWithCount;
import ru.yandex.direct.logviewercore.container.StatsRows;
import ru.yandex.direct.logviewercore.domain.Condition;
import ru.yandex.direct.logviewercore.domain.LogRecordInfo;
import ru.yandex.direct.logviewercore.domain.LogRecordRowMapper;
import ru.yandex.direct.logviewercore.domain.LogTablesInfoManager;
import ru.yandex.direct.logviewercore.domain.ppclog.BinlogQueriesV2;
import ru.yandex.direct.logviewercore.domain.ppclog.BinlogRowsFieldsV2;
import ru.yandex.direct.logviewercore.domain.ppclog.BinlogRowsV2;
import ru.yandex.direct.logviewercore.domain.ppclog.LogField;
import ru.yandex.direct.logviewercore.domain.ppclog.LogRecord;
import ru.yandex.direct.logviewercore.domain.ppclog.LogTable;
import ru.yandex.direct.logviewercore.domain.ppclog.TraceRecord;
import ru.yandex.direct.logviewercore.domain.web.InfoResponse;
import ru.yandex.direct.logviewercore.domain.web.LogViewerFilterForm;
import ru.yandex.direct.logviewercore.service.virtual.VirtualColumn;
import ru.yandex.direct.utils.PassportUtils;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class LogViewerService {
    private static final Logger logger = LoggerFactory.getLogger(LogViewerService.class);
    public static final int MAX_ROWS = 10000;
    private static final List<String> OPERATORS = List.of("<=", ">=", "<", ">", "!");

    private static final String TRACE_ID_COLUMN_NAME = "trace_id";
    /**
     * При запросе к бинлогам и заданному условию на trace_id не должны ограничивать поиск span_id лимитом,
     * переданным в основном запросе к бинлогам
     * По умолчанию берем это число -- почти наверное его достаточно для того, чтобы получить все span_id
     * для заданного условия на trace_id
     */
    private static final int BINLOG_TRACE_LIMIT = 10000;

    DatabaseWrapperProvider dbProvider;
    ShardHelper shardHelper;
    Map<Class, VirtualColumn> virtualColumnsMap;
    private final FeatureService featureService;
    private final FeatureManagingService clientFeatureService;

    static class ConditionWithOperator {
        final String operator;
        final String condition;

        ConditionWithOperator(String condition) {
            this.operator = "";
            this.condition = condition;
        }

        ConditionWithOperator(String operator, String condition) {
            this.operator = operator;
            this.condition = condition;
        }

        public String getOperator() {
            return operator;
        }

        public String getCondition() {
            return condition;
        }

        public static ConditionWithOperator fromString(String string) {
            for (String oper : OPERATORS) {
                if (string.startsWith(oper)) {
                    return new ConditionWithOperator(oper, string.substring(oper.length()));
                }
            }
            return new ConditionWithOperator(string);
        }
    }

    @Autowired
    public LogViewerService(DatabaseWrapperProvider dbProvider,
                            ShardHelper shardHelper,
                            List<VirtualColumn> virtualColumns,
                            FeatureService featureService,
                            FeatureManagingService clientFeatureService
    ) {
        this.dbProvider = dbProvider;
        this.shardHelper = shardHelper;
        this.featureService = featureService;
        this.clientFeatureService = clientFeatureService;
        this.virtualColumnsMap = new HashMap<>();
        for (VirtualColumn col : virtualColumns) {
            this.virtualColumnsMap.put(col.getClass(), col);
        }
    }

    /**
     * Creates {@link SqlBuilder.ExpressionWithBinds} for filtering data by one column with multiple conditions.
     *
     * <p>
     * Conditions, which use {@link #OPERATORS}, are joined using {@code AND}; conditions, which don't use
     * {@link #OPERATORS}, considered as with equal operator and are joined using {@code OR}.
     *
     * <p>
     * For example, list of vals {@code ['>7', '4', '12', '!33']} will produce the following expression:
     * {@code WHERE ((`columnName` > 7 AND `columnName` != 33) OR `columnName` = 4 OR `columnName` = 12)}.
     */
    private static SqlBuilder.ExpressionWithBinds buildFieldConditionExpression(SqlBuilder sql, String columnName,
                                                                                Field field, List<String> vals) {
        Preconditions.checkArgument(vals != null && !vals.isEmpty(), "Empty values list");

        SqlBuilder.Column column = SqlBuilder.column(columnName);
        Class<?> fieldType = field.getType();

        var conditions = mapList(vals, ConditionWithOperator::fromString);

        Predicate<ConditionWithOperator> isAndExpression =
                c -> !c.operator.isEmpty() || isLikeExpression(fieldType, c.condition);

        var andExpressions = conditions.stream()
                .filter(isAndExpression)
                .map(c -> buildFieldConditionEntryExpression(column, fieldType, c.operator, c.condition))
                .collect(toList());

        var orExpressions = conditions.stream()
                .filter(isAndExpression.negate())
                .map(c -> buildFieldConditionEntryExpression(column, fieldType, "=", c.condition))
                .collect(toList());

        if (andExpressions.isEmpty()) {
            return SqlBuilder.joinExpressions(" OR ", orExpressions);
        } else if (orExpressions.isEmpty()) {
            return SqlBuilder.joinExpressions(" AND ", andExpressions);
        } else {
            return SqlBuilder.joinExpressions(" OR ",
                    ListUtils.union(
                            singletonList(SqlBuilder.joinExpressions(" AND ", andExpressions)),
                            orExpressions)
            );
        }
    }

    static SqlBuilder.ExpressionWithBinds buildFieldConditionEntryExpression(
            SqlBuilder.Column column, Class<?> fieldType, String operator, String value) {
        String expr;
        operator = operator.isEmpty() ? "="
                : operator.equals("!") ? "!="
                : operator;
        if ((operator.equals("=") || operator.equals("!=")) && isLikeExpression(fieldType, value)) {
            // Strings are matched using "like" mode instead of equality if they contain SQL wildcard symbols % and _
            expr = operator.equals("!=") ? "not like(" + column + ", ?)" : "like(" + column + ", ?)";
        } else if (fieldType.isArray()) {
            if (operator.equals("=") || operator.equals("!=")) {
                // Arrays are matched using "in" mode instead of equality
                expr = operator.equals("!=") ? "not has(" + column + ", ?)" : "has(" + column + ", ?)";
            } else {
                throw new LogViewerInvalidInputException("Column " + column + " with array type " +
                        fieldType.getCanonicalName() + " can't be compared using operator " + operator);
            }
        } else {
            expr = column + " " + operator + " ?";
        }

        return new SqlBuilder.ExpressionWithBinds(expr, convertArrayOrValue(fieldType, value));
    }

    private static boolean isLikeExpression(Class<?> fieldType, String value) {
        return String.class.isAssignableFrom(fieldType)
                && (value.contains("%") || value.contains("_"));
    }

    static Object convertArrayOrValue(Class<?> fieldType, String value) {
        if (fieldType.isArray()) {
            return convertValue(fieldType.getComponentType(), value);
        } else {
            return convertValue(fieldType, value);
        }
    }

    private static Object convertValue(Class<?> fieldType, String value) {
        if (String.class.isAssignableFrom(fieldType)) {
            return value;
        }
        if (Timestamp.class.isAssignableFrom(fieldType)) {
            return value.trim();
        }
        try {
            if (Integer.class.isAssignableFrom(fieldType) || int.class.isAssignableFrom(fieldType)) {
                return Integer.valueOf(value);
            }
            if (Long.class.isAssignableFrom(fieldType) || long.class.isAssignableFrom(fieldType)) {
                return Long.valueOf(value);
            }
            if (Float.class.isAssignableFrom(fieldType) || float.class.isAssignableFrom(fieldType)) {
                return Float.valueOf(value);
            }
            if (Double.class.isAssignableFrom(fieldType) || double.class.isAssignableFrom(fieldType)) {
                return Double.valueOf(value);
            }
            if (Boolean.class.isAssignableFrom(fieldType) || boolean.class.isAssignableFrom(fieldType)) {
                return Boolean.valueOf(value);
            }
        } catch (NumberFormatException e) {
            throw new LogViewerInvalidInputException(
                    "Can't convert value " + value + " to type " + fieldType.getCanonicalName(),
                    e
            );
        }
        throw new LogViewerInvalidInputException(
                "Unsupported type " + fieldType.getCanonicalName() + " with value " + value);
    }

    /**
     * Enhances conditions allowing a richer set of operations
     * <p>
     * At the moment logins in uid fields are converted to numeric uids if possible.
     * Operators are concatenated with the next values through {@link #normalizeOperators}.
     */
    public Map<String, List<String>> enhanceConditions(Map<String, String> conditions) {
        Map<String, List<String>> result = new HashMap<>();
        Splitter dateTimeSplitter = Splitter.on(Pattern.compile("[,]+")).omitEmptyStrings();
        Splitter splitter = Splitter.on(Pattern.compile("[, ]+")).omitEmptyStrings();
        for (Map.Entry<String, String> condition : conditions.entrySet()) {
            if (condition.getValue().trim().isEmpty()) {
                continue;
            }
            Splitter spl = "log_time".equals(condition.getKey()) ? dateTimeSplitter : splitter;
            List<String> vals = spl.splitToList(condition.getValue());
            if ("uid".equals(condition.getKey()) || "cluid".equals(condition.getKey())) {
                result.put(condition.getKey(), vals.stream().map(x -> loginToUid(x).toString()).collect(toList()));
            } else {
                result.put(condition.getKey(), normalizeOperators(vals));
            }
        }
        return result;
    }

    /**
     * Joins single operators with the next values into one string.
     * <p>
     * Example: {@code [">", "12", "4", "<70"] -> [">12", "4", "<70"]}
     * <p>
     * Throws exception when operators and values are in incorrect order.
     */
    static List<String> normalizeOperators(List<String> vals) {
        List<String> result = new ArrayList<>();
        String lastOperator = "";
        for (String val : vals) {
            if (lastOperator.isEmpty()) {
                if (OPERATORS.stream().anyMatch(val::equals)) {
                    lastOperator = val;
                } else {
                    result.add(val);
                }
            } else {
                if (OPERATORS.stream().anyMatch(val::startsWith)) {
                    throw new LogViewerInvalidInputException(
                            "Incorrect operators format, two operators in a row: " + lastOperator + " and " + val);
                } else {
                    result.add(lastOperator + val);
                    lastOperator = "";
                }
            }
        }
        if (!lastOperator.isEmpty()) {
            throw new LogViewerInvalidInputException("Incorrect operators format, hanging operator: " + lastOperator);
        }
        return result;
    }

    public List<LogRecord> mergeDataWithTraceIdRows(
            LogRecordInfo<? extends LogRecord> info,
            List<? extends LogRecord> rows,
            List<? extends LogRecord> traceIdRows
    ) {
        List<LogRecord> resultRows = new ArrayList<>();
        Map<String, ? extends List> traceIdDataRowsMap = StreamEx.of(traceIdRows)
                .groupingBy(
                        record -> uncheckedFieldGet(record, info.getColumnField(info.getTraceIdColumn())).toString());
        rows.forEach(rec -> {
            String traceId = uncheckedFieldGet(rec, info.getColumnField(info.getTraceIdColumn())).toString();
            //удаляем значение из мапы, чтобы при повторах trace_id не дублировать записи
            List<? extends LogRecord> trIdRows = traceIdDataRowsMap.remove(traceId);
            resultRows.add(rec);
            if (trIdRows != null) {
                resultRows.addAll(trIdRows);
            }
        });


        return resultRows;
    }

    /**
     * Ищем в {@code ret} все значения с traceId. Исключаем уже найденые записи по reqid
     */
    public Map<String, List<Condition>> getTraceIdCondition(LogRecordInfo<? extends LogRecord> info,
                                                            RowsWithCount<? extends LogRecord> ret) {
        String reqIdColumn = info.getReqidColumn();
        String traceIdColumn = info.getTraceIdColumn();
        List<Condition> reqIdValues = mapList(ret.rows(),
                val -> new Condition(true, uncheckedFieldGet(val, info.getColumnField(reqIdColumn)).toString()));
        List<Condition> traceIdValues = StreamEx.of(ret.rows())
                .map(row -> uncheckedFieldGet(row, info.getColumnField(traceIdColumn)).toString())
                .remove("0"::equals)
                .map(Condition::new)
                .toList();
        Map<String, List<Condition>> result = new HashMap<>();
        result.put(traceIdColumn, traceIdValues);
        result.put(reqIdColumn, reqIdValues);
        return result;
    }

    // если число - используем его
    // если непохоже на число - считаем, что это логин и резолвим по метабазе
    private Long loginToUid(String login) {
        try {
            return Long.valueOf(login);
        } catch (NumberFormatException ex) {
            // normalize login
            // replace login string with uid if possible
            Long uid = shardHelper.getUidByLogin(PassportUtils.normalizeLogin(login));
            return uid == null ? -1 : uid;
        }
    }

    public <T extends LogRecord> RowsWithCount<T> getLogRows(
            SimpleDb dbName, LogRecordInfo<? extends T> info,
            List<String> fields, LocalDateTime dateFrom, LocalDateTime dateTo,
            Map<String, List<String>> conditions,
            int limit, int offset, boolean reverseOrder) {
        Map<String, List<Condition>> conditionsMap = EntryStream.of(conditions)
                .mapValues(t -> mapList(t, Condition::new))
                .toMap();
        if (isBinlogQuery(info.getInstanceClass())) {
            String reqidColumn = info.getReqidColumn();
            var ret = preprocessBinlogRequest(conditionsMap, reqidColumn, dbName, dateFrom, dateTo, reverseOrder);
            if (ret != null && ret.totalCount() == 0L) {
                return new RowsWithCount<>(0, ret.query(), emptyList());
            }
        }
        return getLogRowsWithConditions(dbName, info, fields, dateFrom, dateTo, conditionsMap, limit, offset,
                reverseOrder);
    }

    private <T extends LogRecord> boolean isBinlogQuery(Class<? extends T> logClass) {
        return logClass.isAssignableFrom(BinlogQueriesV2.class)
                || logClass.isAssignableFrom(BinlogRowsFieldsV2.class)
                || logClass.isAssignableFrom(BinlogRowsV2.class);
    }

    /**
     * Предобработать запрос к бинлогам в случае, когда поиск запрашивается с условием по trace_id
     * <p>
     * trace_id -- фиктивная колонка в фильтрации запросов к бинлогам. Добавлена для удобного поиска
     * всех изменений в базе по условиям на trace_id
     * Под капотом ищутся все span_id, связанные с переданными trace_id, и добавляются в условие фильтрации
     * к исходному запросу
     * <p>
     * (!) метод мутирует переданную {@code modifiedConditions}
     *
     * @return результат запроса за span_id по переданным trace_id или {@code null}, если запроса в trace логи не было
     * (например, не было условий по trace_id)
     */
    @Nullable
    private RowsWithCount<TraceRecord> preprocessBinlogRequest(
            Map<String, List<Condition>> modifiedConditions,
            String reqidColumn,
            SimpleDb dbName,
            LocalDateTime dateFrom,
            LocalDateTime dateTo,
            boolean reverseOrder
    ) {
        if (!modifiedConditions.containsKey(TRACE_ID_COLUMN_NAME)) {
            return null;
        }
        var info = LogTablesInfoManager.getLogRecordInfo(TraceRecord.class);
        var ret = getLogRowsWithConditions(
                dbName,
                info,
                List.of(info.getReqidColumn()),
                dateFrom,
                dateTo,
                Map.of(info.getTraceIdColumn(), modifiedConditions.get(TRACE_ID_COLUMN_NAME)),
                BINLOG_TRACE_LIMIT,
                0,
                reverseOrder
        );
        Set<Long> spanIds = listToSet(ret.rows(), x -> x.span_id);
        spanIds.forEach(s -> {
            var spanIdCondition = new Condition(String.valueOf(s));
            //Условия для reqid и trace_id объединяются по ИЛИ -- считаем это фичей
            modifiedConditions.computeIfAbsent(reqidColumn, v -> new ArrayList<>()).add(spanIdCondition);
        });
        modifiedConditions.remove(TRACE_ID_COLUMN_NAME);
        return ret;
    }

    public <T extends LogRecord> RowsWithCount<T> getLogRowsWithConditions(
            SimpleDb dbName, LogRecordInfo<? extends T> info,
            List<String> fields, LocalDateTime dateFrom, LocalDateTime dateTo,
            Map<String, List<Condition>> conditions,
            int limit, int offset, boolean reverseOrder) {
        // заглушка SqlBuilder-a - со всеми условиями, но без полей / order by / limit
        Supplier<SqlBuilder> builderSupplier = () -> {
            SqlBuilder builder = new SqlBuilder()
                    .from(info.getTableName());
            if (!info.getArrayJoin().isEmpty()) {
                builder.arrayJoin(info.getArrayJoin());
            }
            addDateTimeConditions(builder, info, dateFrom, dateTo);
            addFieldsComplexConditions(builder, info, conditions);
            return builder;
        };

        SqlBuilder.Order sortOrder = getSortOrder(reverseOrder);

        // формируем эквивалентный запрос, для отображения в интерфейсе
        SqlBuilder fullQuery = builderSupplier.get()
                .select(mapList(fields, info::getDbColumn))
                .limit(offset, limit);
        addOrderBy(fullQuery, sortOrder, info);
        String fullSql = generateFullSql(dbName, fullQuery);

        /*
        Есть две проблемы:
        - для сортировки CH сначала выбирает все поля всех подходящих строк, и только после это сортирует.
          если полей много или поля толстые - это очень дорого
        - для пейджинга нам нужно узнать полное число строк - в CH нет аналога SQL_CALC_FOUND_ROWS, а
          rows_before_limit_at_least работает только для запросов c group by и не работает через distibuted

        Делаем два запроса:
        - select log_time, count(*) ... group by log_time with totals limit
          из него мы в totals можем узнать полное число строк, и найти все подходящие нам log_time
          log_time - селективное поле, его хорошо дальше использовать в prewhere
        - формируем запрос с полными условиями и log_time
         SELECT *
             FROM tbl
             WHERE log_time in (...полученные log_time...)
               AND ...все условия...
        */
        SqlBuilder logTimeSql = builderSupplier.get()
                .select(info.getLogTimeColumn())
                .selectExpression("count()", "cnt");
        logTimeSql.groupBy(info.getLogTimeColumn()).withTotals();
        logTimeSql.orderBy(SqlBuilder.column(info.getLogTimeColumn()), sortOrder);
        // выбираем все времена, а выбор именно нашей страницы будет уже в основном запросе
        logTimeSql.limit(0, offset + limit);

        ClickHouseResponse response = dbProvider.get(dbName)
                .clickhouseQuery(logTimeSql.toString(), logTimeSql.getBindings());
        if (response.getData().isEmpty()) {
            return new RowsWithCount<>(0, fullSql, emptyList());
        }

        // выбираем log_time, пока не превысим offset + limit
        List<String> times = new ArrayList<>();
        long runningCount = 0;
        for (List<String> row : response.getData()) {
            if (runningCount >= offset + limit) {
                break;
            }
            times.add(row.get(0));
            runningCount += Long.parseLong(row.get(1));
        }
        long totalRowsCount = Long.parseLong(response.getTotals().get(1));

        // вторая часть запроса - выбираем собственно данные
        SqlBuilder sql = builderSupplier.get()
                .select(mapList(fields, info::getDbColumn));

        sql.prewhere(info.getLogTimeColumn()
                        + " in ("
                        + StreamEx.of(times).map(t -> "?").joining(", ")
                        + ")",
                times.toArray());

        addOrderBy(sql, sortOrder, info);
        sql.limit(offset, limit);

        List<? extends T> rows = dbProvider.get(dbName).query(sql.toString(), sql.getBindings(),
                new LogRecordRowMapper<>(fields, info));

        return new RowsWithCount<T>(totalRowsCount, fullSql, rows);
    }

    protected String generateFullSql(SimpleDb db, SqlBuilder query) {
        return generateFullSql(dbProvider, db, query);
    }

    public static String generateFullSql(DatabaseWrapperProvider dbProvider, SimpleDb db, SqlBuilder query) {
        String querySql;
        try (
                var connection = dbProvider.get(db).getDataSource().getConnection();
                var ps = (ClickHousePreparedStatement) connection.prepareStatement(query.generateSql(true));
        ) {
            Object[] bindings = query.getBindings();
            for (int i = 0; i < bindings.length; i++) {
                ps.setObject(i + 1, bindings[i]);
            }
            querySql = ps.asSql();
        } catch (SQLException e) {
            throw new IllegalArgumentException("Can't generate sql", e);
        }
        return querySql;
    }

    private SqlBuilder.Order getSortOrder(boolean reverseOrder) {
        return reverseOrder ? SqlBuilder.Order.DESC : SqlBuilder.Order.ASC;
    }

    private SqlBuilder.Order getCountSortOrder(boolean reverseOrder) {
        return reverseOrder ? SqlBuilder.Order.ASC : SqlBuilder.Order.DESC;
    }

    private <T extends LogRecord> void addOrderBy(SqlBuilder sql, SqlBuilder.Order order,
                                                  LogRecordInfo<? extends T> info) {
        sql.orderBy(SqlBuilder.column(info.getLogTimeColumn()), order);
        for (String col : info.getAdditionalSort()) {
            sql.orderBy(SqlBuilder.column(col), order);
        }
    }

    @SuppressWarnings("checkstyle:parameternumber")
    public <T extends LogRecord> StatsRows getLogStats(
            SimpleDb dbName,
            LogRecordInfo<? extends T> info,
            List<String> fields,
            LocalDateTime dateFrom,
            LocalDateTime dateTo,
            Map<String, List<String>> conditions,
            LogViewerFilterForm.TimeGroupingType timeGroupingType,
            boolean sortByCount,
            boolean reverseOrder,
            int limit,
            int offset) {
        LogRecordRowMapper<T> mapper = new LogRecordRowMapper<>(fields, info);

        SqlBuilder sql = new SqlBuilder()
                .from(info.getTableName());

        if (!info.getArrayJoin().isEmpty()) {
            sql.arrayJoin(info.getArrayJoin());
        }

        for (String field : fields) {
            if (timeGroupingType != null && (field.equals(info.getLogTimeColumn()) || info.getDbColumn(field).toString().equals(info.getLogTimeColumn()))) {
                String roundExpression = makeTimeRoundExpression(info.getLogTimeColumn(), timeGroupingType);
                sql.selectExpression(roundExpression,
                                info.getLogTimeColumn() + "_" + timeGroupingType.name().toLowerCase())
                        .groupByExpression(roundExpression);
                if (!sortByCount) {
                    sql.orderByExpression(roundExpression, getSortOrder(reverseOrder));
                }
            } else {
                String dbColumn = info.getDbColumn(field);
                sql.select(dbColumn).groupBy(dbColumn);
                if (!sortByCount) {
                    sql.orderBy(dbColumn, getSortOrder(reverseOrder));
                }
            }
        }
        sql.selectExpression("count(*)", "count");

        Map<String, List<Condition>> conditionsMap = EntryStream.of(conditions)
                .mapValues(t -> mapList(t, Condition::new))
                .toMap();

        if (isBinlogQuery(info.getInstanceClass())) {
            String reqidColumn = info.getReqidColumn();
            var ret = preprocessBinlogRequest(conditionsMap, reqidColumn, dbName, dateFrom, dateTo, reverseOrder);
            if (ret != null && ret.totalCount() == 0L) {
                return new StatsRows(emptyList(), ret.query());
            }
        }

        addDateTimeConditions(sql, info, dateFrom, dateTo);
        addFieldsComplexConditions(sql, info, conditionsMap);

        if (sortByCount) {
            sql.orderBy("count", getCountSortOrder(reverseOrder));
        }
        sql.limit(offset, limit);

        String fullSql = generateFullSql(dbName, sql);

        List<List<Object>> rows = dbProvider.get(dbName).query(sql.toString(), sql.getBindings(), (rs, i) -> {
            List<Object> ret = mapper.mapRowToObjectsList(rs, i);
            ret.add(rs.getLong("count"));
            return ret;
        });

        return new StatsRows(rows, fullSql);
    }

    private String makeTimeRoundExpression(String fieldName, LogViewerFilterForm.TimeGroupingType timeGroupingType) {
        String quotedField = ClickHouseUtil.quoteIdentifier(fieldName);
        switch (timeGroupingType) {
            case MINUTE:
                return String.format("toStartOfMinute(%s)", quotedField);
            case HOUR:
                return String.format("toStartOfHour(%s)", quotedField);
            case DAY:
                return String.format("toDateTime(toDate(%s))", quotedField);
            case MONTH:
                return String.format("toDateTime(toStartOfMonth(%s))", quotedField);
            case YEAR:
                return String.format("toDateTime(toStartOfYear(%s))", quotedField);
            default:
                throw new IllegalArgumentException("Unsupported TimeGroupingType: " + timeGroupingType);
        }
    }

    private <T extends LogRecord> void addFieldsComplexConditions(SqlBuilder sql, LogRecordInfo<? extends T> info,
                                                                  Map<String, List<Condition>> conditions) {
        for (Map.Entry<String, List<Condition>> condition : conditions.entrySet()) {
            String name = condition.getKey();
            if (info.findVirtual(name).isPresent()) {
                continue;
            }

            List<String> conditionValues = StreamEx.of(condition.getValue())
                    .remove(Condition::isInvert)
                    .map(Condition::getValue)
                    .toList();
            List<String> invertedConditionValues = StreamEx.of(condition.getValue())
                    .filter(Condition::isInvert)
                    .map(Condition::getValue)
                    .toList();
            addFieldCondition(sql, info, name, conditionValues);
            addFieldNotCondition(sql, info, name, invertedConditionValues);
        }
    }

    private <T extends LogRecord> void addFieldCondition(SqlBuilder sql, LogRecordInfo<? extends T> info,
                                                         String name, List<String> conditionValues) {
        String columnName = info.getDbColumn(name);
        Field field = info.getColumnField(name);
        List<String> vals = info.isEncodingBroken()
                ? conditionValues.stream().map(EncodingUtils::doubleUtf8Encode).collect(toList())
                : conditionValues;
        if (!vals.isEmpty()) {
            sql.where(buildFieldConditionExpression(sql, columnName, field, vals));
        }
    }

    private <T extends LogRecord> void addFieldNotCondition(SqlBuilder sql, LogRecordInfo<? extends T> info,
                                                            String name, List<String> conditionValues) {
        String columnName = info.getDbColumn(name);
        Field field = info.getColumnField(name);
        List<String> vals = info.isEncodingBroken()
                ? conditionValues.stream().map(EncodingUtils::doubleUtf8Encode).collect(toList())
                : conditionValues;
        if (!vals.isEmpty()) {
            sql.whereNot(buildFieldConditionExpression(sql, columnName, field, vals));
        }
    }

    private void addDateTimeConditions(SqlBuilder sql, LogRecordInfo<?> tableInfo,
                                       LocalDateTime dateFrom, LocalDateTime dateTo) {

        SqlBuilder.Column dateColumn = SqlBuilder.column(tableInfo.getLogDateColumn());

        if (dateFrom.toLocalDate().equals(dateTo.toLocalDate())) {
            sql.where(dateColumn, "=", Date.valueOf(dateFrom.toLocalDate()));
        } else {
            sql.where(
                    dateColumn + " BETWEEN ? AND ?",
                    Date.valueOf(dateFrom.toLocalDate()),
                    Date.valueOf(dateTo.toLocalDate())
            );
        }

        sql.where(SqlBuilder.column(tableInfo.getLogTimeColumn()) + " BETWEEN ? AND ?",
                Timestamp.valueOf(dateFrom),
                Timestamp.valueOf(dateTo));
    }

    private Optional<VirtualColumn> findVirtualColumn(LogRecordInfo<? extends LogRecord> info, String field) {
        Optional<Class<? extends VirtualColumn>> virtual = info.findVirtual(field);
        if (!virtual.isPresent()) {
            return Optional.empty();
        }
        VirtualColumn virtualColumn = virtualColumnsMap.get(virtual.get());
        if (virtualColumn == null) {
            throw new IllegalArgumentException();
        }
        return Optional.of(virtualColumn);
    }

    @SuppressWarnings("unchecked")
    public void calculateVirtualColumns(
            LogRecordInfo<? extends LogRecord> info,
            List<String> fields,
            List<? extends LogRecord> rows) {
        for (String field : fields) {
            findVirtualColumn(info, field).ifPresent(v -> v.calculate(rows));
        }
    }

    public List<String> devirtualizeFields(LogRecordInfo<? extends LogRecord> info, List<String> fields) {
        Set<String> result = new LinkedHashSet<>();
        for (String field : fields) {
            Optional<VirtualColumn> virtualColumn1 = findVirtualColumn(info, field);
            if (virtualColumn1.isPresent()) {
                result.addAll(asList(virtualColumn1.get().sourceColumns()));
            } else {
                result.add(field);
            }
        }
        return new ArrayList<>(result);
    }

    /**
     * Enhances results by modifying values of outgoing columns
     * <p>
     * At the moment numeric uid values are converted to logins if possible
     */
    public void enhanceResults(List<List<Object>> results, List<String> fields) {
        for (int i = 0; i < fields.size(); ++i) {
            final int fieldIndex = i;
            String fieldName = fields.get(fieldIndex);
            if ("uid".equals(fieldName)) {
                // Convert uids to logins if there's a uid column
                // First cut all values as a new stream
                List<Object> allUids = results.stream().map(row -> row.get(fieldIndex)).collect(toList());
                // Use the metabase to lookup any existing mappings from uid to login
                List<List<String>> allLogins = shardHelper.getLoginsByUids(allUids);
                // The mappings are unique, but some of them might not have been found
                for (int rowIndex = 0; rowIndex < allLogins.size(); ++rowIndex) {
                    List<String> logins = allLogins.get(rowIndex);
                    if (!logins.isEmpty()) {
                        results.get(rowIndex).set(fieldIndex, logins.get(0));
                    }
                }
            }
        }
    }

    /**
     * Делаем из списка объектов список списков полей
     */
    public List<List<Object>> cutColumns(LogRecordInfo<? extends LogRecord> info,
                                         List<? extends LogRecord> rows,
                                         List<String> fields) {
        List<Field> rawFields = mapList(fields, info::getColumnField);
        return rows
                .stream()
                .map(row -> mapList(rawFields, f -> uncheckedFieldGet(row, f)))
                .collect(toList());
    }

    private static Object uncheckedFieldGet(Object obj, Field field) {
        try {
            return field.get(obj);
        } catch (IllegalAccessException e) {
            throw new IllegalArgumentException(
                    "Can't get field " + field.getName() + " of " + obj.getClass().getCanonicalName(), e);
        }
    }

    /**
     * Get information about available logs
     *
     * @param filterPredicate предикат фильтрации логов
     */
    public InfoResponse getLogTypesInfo(Function<LogTable, Boolean> filterPredicate) {
        List<InfoResponse.LogInfo> logs = new ArrayList<>();
        LogTablesInfoManager.allTableClasses().forEach((logName, clazz) -> {
            LogRecordInfo<? extends LogRecord> logInfo = new LogRecordInfo<>(clazz);

            List<InfoResponse.ColumnInfo> columns = new ArrayList<>();
            for (String colName : logInfo.getColumnNames()) {
                LogField ann = logInfo.getColumnField(colName).getAnnotation(LogField.class);
                String desc = ann == null ? null : ann.desc();
                boolean hidden = ann != null && ann.hidden();
                boolean strictlyHidden = ann != null && ann.strictlyHidden();
                boolean heavy = ann != null && ann.heavy();
                int scale = ann != null ? ann.scale() : 0;
                boolean virtual = logInfo.findVirtual(colName).isPresent();
                columns.add(new InfoResponse.ColumnInfo(colName, desc, hidden, strictlyHidden, heavy, scale, virtual));
            }

            LogTable ann = clazz.getAnnotation(LogTable.class);

            if (filterPredicate.apply(ann)) {
                String dbReqIdColumn = ann.reqidColumn();
                String reqIdColumn = logInfo.getColimnByDbColumn(dbReqIdColumn);
                String dbTraceIdColumn = ann.traceIdColumn();
                String traceIdColumn = logInfo.getColimnByDbColumn(dbTraceIdColumn);
                logs.add(new InfoResponse.LogInfo(logName, ann.desc(), reqIdColumn, traceIdColumn, columns));
            }
        });
        logs.sort((a, b) -> a.getName().compareToIgnoreCase(b.getName()));

        return new InfoResponse(logs);
    }

    public boolean isNewUiEnabled(@Nullable User operator) {
        return operator != null && featureService.isEnabled(operator.getUid(), FeatureName.NEW_LOGVIEWER_ENABLED);
    }

    /**
     * Переключить UI интерфейс
     *
     * @param operator  оператор, для которого происходит переключение
     * @param enableNew {@code true}, если нужно переключиться на новый интерфейс. Иначе, на старый
     * @return {@code true}, если переключение произошло успешно
     */
    public boolean switchUi(@Nullable User operator, Boolean enableNew) {
        if (operator == null) {
            return false;
        }
        var clientId = operator.getClientId();
        if (enableNew) {
            clientFeatureService.enableFeatureForClient(clientId, FeatureName.NEW_LOGVIEWER_ENABLED);
        } else {
            clientFeatureService.disableFeatureForClient(clientId, FeatureName.NEW_LOGVIEWER_ENABLED);
        }
        return true;
    }
}
