package ru.yandex.direct.logviewercore.service;

import java.sql.Timestamp;
import java.util.Arrays;
import java.util.List;
import java.util.stream.IntStream;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.DataFormat;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.stereotype.Service;

@Service
public class LogViewerXlsService {
    public static final int MAX_VALUE_SIZE = 32766;
    public static final long MAX_LONG_VALUE = 1L << 53;

    public Workbook createXlsReport(String logName, List<String> fieldNames, List<List<Object>> data) {
        Workbook workbook = new XSSFWorkbook();
        Sheet sheet = workbook.createSheet(logName);

        Row firstRow = sheet.createRow(0);
        for (int i = 0; i < fieldNames.size(); ++i) {
            Cell cell = firstRow.createCell(i);
            cell.setCellValue(fieldNames.get(i));
            cell.setCellStyle(getBoldStyle(workbook));
        }

        for (int i = 0; i < data.size(); ++i) {
            Row row = sheet.createRow(i + 1);
            for (int j = 0; j < data.get(i).size(); ++j) {
                Cell cell = row.createCell(j);
                convertValueAndWriteIntoCell(cell, data.get(i).get(j));
            }
        }

        IntStream.range(0, firstRow.getLastCellNum()).forEach(sheet::autoSizeColumn);

        return workbook;
    }

    private void convertValueAndWriteIntoCell(Cell cell, Object value) {
        if (value instanceof String) {
            setTrimmedValue(cell, (String) value);
        } else if (value instanceof Timestamp) {
            cell.setCellValue((Timestamp) value);
            cell.setCellStyle(getDatetimeStyle(cell.getSheet().getWorkbook()));
        } else if (value instanceof Integer) {
            cell.setCellValue((Integer) value);
        } else if (value instanceof Long) {
            Long val = (Long) value;
            if (val < MAX_LONG_VALUE) {
                cell.setCellValue(val);
            } else {
                cell.setCellValue(val.toString());
            }
        } else if (value instanceof Float) {
            cell.setCellValue((Float) value);
        } else if (value instanceof Double) {
            cell.setCellValue((Double) value);
        } else if (value instanceof int[]) {
            setTrimmedValue(cell, Arrays.toString((int[]) value));
        } else if (value instanceof long[]) {
            setTrimmedValue(cell, Arrays.toString((long[]) value));
        } else if (value instanceof float[]) {
            setTrimmedValue(cell, Arrays.toString((float[]) value));
        } else if (value instanceof double[]) {
            setTrimmedValue(cell, Arrays.toString((double[]) value));
        } else if (value.getClass().isArray()) {
            setTrimmedValue(cell, Arrays.toString((Object[]) value));
        } else {
            throw new LogViewerInvalidInputException("Unsupported class " + value.getClass() + " of value " + value);
        }
    }

    private void setTrimmedValue(Cell cell, String value) {
        String trimmed = value.length() <= MAX_VALUE_SIZE ? value : value.substring(0, MAX_VALUE_SIZE - 3) + "...";
        cell.setCellValue(trimmed);
    }

    private CellStyle getDatetimeStyle(Workbook workbook) {
        CellStyle datetimeStyle = workbook.createCellStyle();
        DataFormat datetimeFormat = workbook.createDataFormat();
        datetimeStyle.setDataFormat(datetimeFormat.getFormat("yyyy-mm-dd hh:mm:ss"));
        return datetimeStyle;
    }

    private CellStyle getBoldStyle(Workbook workbook) {
        CellStyle boldStyle = workbook.createCellStyle();
        Font headerFont = workbook.createFont();
        headerFont.setBold(true);
        boldStyle.setFont(headerFont);
        return boldStyle;
    }
}
