package ru.yandex.direct.logviewercore.service.virtual;

import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.springframework.stereotype.Component;

import ru.yandex.direct.logviewercore.domain.ppclog.TraceRecord;

@Component
public class TraceProfileFieldsColumn implements VirtualColumn<TraceRecord, String> {
    @Override
    public String[] sourceColumns() {
        return new String[]{"profile_func", "profile_tags", "profile_all_ela",
                "profile_childs_ela", "profile_calls", "profile_obj_num"};
    }

    @Override
    public void calculate(List<TraceRecord> recs) {
        for (TraceRecord rec : recs) {
            List<Integer> indices = IntStream.range(0, rec.profile_func.length)
                    .boxed()
                    .sorted(Comparator.<Integer>comparingDouble(i -> rec.profile_all_ela[i]).reversed())
                    .collect(Collectors.toList());

            StringBuilder builder = new StringBuilder();
            for (int i : indices) {
                builder.append(rec.profile_func[i]);

                if (rec.profile_tags[i] != null && !rec.profile_tags[i].isEmpty()) {
                    builder.append("/").append(rec.profile_tags[i]);
                }

                builder.append("\tela:").append(formatEla(rec.profile_all_ela[i]));

                if (rec.profile_childs_ela[i] > 0) {
                    builder.append("(-").append(formatEla(rec.profile_childs_ela[i])).append(')');
                }

                if (rec.profile_calls[i] > 1) {
                    builder.append("\tcalls:").append(rec.profile_calls[i]);
                }

                if (rec.profile_obj_num[i] > 0) {
                    builder.append("\tobj:").append(rec.profile_obj_num[i]);
                }

                builder.append("\n");
            }
            rec.profile = builder.toString();
        }
    }

    private String formatEla(double ela) {
        if (ela >= 0.001 || ela < 1e-6) {
            return String.format("%.3f", ela);
        } else {
            return String.format("%.6f", ela);
        }
    }
}
