package ru.yandex.direct.market.client.http

import io.netty.handler.codec.http.HttpHeaders
import ru.yandex.direct.asynchttp.ParallelFetcherFactory
import ru.yandex.direct.http.smart.core.Call
import ru.yandex.direct.http.smart.core.Smart
import ru.yandex.direct.http.smart.error.ErrorUtils
import ru.yandex.direct.market.client.MarketClient
import ru.yandex.direct.market.client.MarketClient.FeedFeaturesInfo
import ru.yandex.direct.market.client.MarketClient.FileFeedInfo
import ru.yandex.direct.market.client.MarketClient.SendFeedResult
import ru.yandex.direct.market.client.MarketClient.SiteFeedInfo
import ru.yandex.direct.market.client.MarketClient.SitePreviewIds
import ru.yandex.direct.market.client.MarketClient.SitePreviewInfo
import ru.yandex.direct.market.client.MarketClient.UrlFeedInfo
import ru.yandex.direct.market.client.exception.MarketClientException
import ru.yandex.direct.market.client.http.MarketMbiApi.DeleteSitePreviewsRequest
import ru.yandex.direct.market.client.http.MarketMbiApi.DirectSitePreviewIds
import ru.yandex.direct.market.client.http.MarketMbiApi.PreviewSiteRequest
import ru.yandex.direct.market.client.http.MarketMbiApi.RefreshFeedRequest
import ru.yandex.direct.market.client.http.MarketMbiApi.RefreshFeedResponse
import ru.yandex.direct.market.client.http.MarketMbiApi.RefreshFileRequest
import ru.yandex.direct.market.client.http.MarketMbiApi.RefreshSiteRequest
import ru.yandex.direct.market.client.http.MarketMbiApi.SetFeedFeatureRequest
import ru.yandex.direct.tvm.TvmIntegration
import ru.yandex.direct.tvm.TvmService

/**
 * Реализует работу с севисами Маркета.
 *
 * @see [mbi-api](https://wiki.yandex-team.ru/mbi/newdesign/components/mbi-api/)
 */
open class MarketHttpClient(configuration: MarketConfiguration,
                            parallelFetcherFactory: ParallelFetcherFactory,
                            tvmIntegration: TvmIntegration) : MarketClient {

    data class MarketConfiguration(val tvmService: TvmService, val marketApiUrl: String)

    private val marketMbiApi: MarketMbiApi = Smart.builder()
            .withParallelFetcherFactory(parallelFetcherFactory)
            .useTvm(tvmIntegration, configuration.tvmService)
            .withProfileName("mbi-api")
            .withBaseUrl(configuration.marketApiUrl)
            .addHeaderConfigurator { headers: HttpHeaders -> headers.add("Content-type", "application/xml") }
            .build()
            .create(MarketMbiApi::class.java)

    override fun sendUrlFeedToMbi(urlFeedInfo: UrlFeedInfo, enableDirectPlacement: Boolean?): SendFeedResult {
        val request = RefreshFeedRequest(urlFeedInfo.chiefUid, urlFeedInfo.feedId, urlFeedInfo.clientId.asLong(),
                urlFeedInfo.feedUrl, urlFeedInfo.login, urlFeedInfo.password, enableDirectPlacement)
        val call = marketMbiApi.refreshDirectFeed(request)
        return executeSendFeedCall(call, request)
    }

    override fun sendSiteFeedToMbi(siteFeedInfo: SiteFeedInfo, enableDirectPlacement: Boolean?): SendFeedResult {
        val request = RefreshSiteRequest(siteFeedInfo.chiefUid, siteFeedInfo.feedId, siteFeedInfo.clientId.asLong(),
                siteFeedInfo.schema.name.lowercase(), siteFeedInfo.domain, "/", enableDirectPlacement)
        val call = marketMbiApi.refreshDirectSite(request)
        return executeSendFeedCall(call, request)
    }

    override fun sendFileFeedToMbi(fileFeedInfo: FileFeedInfo, enableDirectPlacement: Boolean?): SendFeedResult {
        val request = RefreshFileRequest(fileFeedInfo.chiefUid, fileFeedInfo.feedId, fileFeedInfo.clientId.asLong(),
                fileFeedInfo.fileMdsUrl, fileFeedInfo.refresh, enableDirectPlacement)
        val call = marketMbiApi.refreshDirectFile(request)
        return executeSendFeedCall(call, request)
    }

    override fun addSitePreviewToMBI(sitePreviewInfo: SitePreviewInfo): SendFeedResult {
        val request = PreviewSiteRequest(sitePreviewInfo.schema.name.lowercase(), sitePreviewInfo.domain, "/",
            sitePreviewInfo.numOffers)
        val call = marketMbiApi.addDirectSitePreview(request)
        return executeSendFeedCall(call, request)
    }

    override fun deleteSitePreviewsFromMBI(deleteSitePreviewInfo: List<SitePreviewIds>): List<SitePreviewIds> {
        val request = DeleteSitePreviewsRequest(deleteSitePreviewInfo.map {
            DirectSitePreviewIds(it.shopId, it.feedId)
        })

        val call = marketMbiApi.deleteDirectSitePreview(request)
        val executeResult = call.execute()
        ErrorUtils.checkResultForErrors(executeResult, { msg: String? -> MarketClientException(msg) }, request)

        return executeResult.success.feedIds.map {
            SitePreviewIds(it.partnerId, it.feedId)
        }
    }

    override fun setFeedFeaturesToMbi(feedFeaturesInfo: FeedFeaturesInfo) {
        val request = SetFeedFeatureRequest(feedFeaturesInfo.shopId, feedFeaturesInfo.isStandby,
                feedFeaturesInfo.isGoodsAds, feedFeaturesInfo.isSearchSnippetGallery)
        val call = marketMbiApi.setDirectFeedFeatures(request)
        val executeResult = call.execute()
        ErrorUtils.checkResultForErrors(executeResult, { msg: String? -> MarketClientException(msg) }, request)
    }

    private fun executeSendFeedCall(call: Call<RefreshFeedResponse>, request: Any): SendFeedResult {
        val executeResult = call.execute()
        ErrorUtils.checkResultForErrors(executeResult, { msg: String? -> MarketClientException(msg) }, request)
        val response: RefreshFeedResponse = executeResult.success

        if (response.marketFeedId == null) {
            throw MarketClientException("marketFeedId wasn't returned")
        }
        if (response.partnerId == null) {
            throw MarketClientException("partnerId wasn't returned")
        }
        if (response.businessId == null) {
            throw MarketClientException("businessId wasn't returned")
        }
        return SendFeedResult(response.marketFeedId!!, response.partnerId!!, response.businessId!!)
    }

}
