package ru.yandex.direct.market.client.http

import com.fasterxml.jackson.annotation.JsonIgnoreProperties
import com.fasterxml.jackson.annotation.JsonInclude
import com.fasterxml.jackson.annotation.JsonProperty
import com.fasterxml.jackson.dataformat.xml.annotation.JacksonXmlProperty
import com.fasterxml.jackson.dataformat.xml.annotation.JacksonXmlRootElement
import ru.yandex.direct.http.smart.annotations.Json
import ru.yandex.direct.http.smart.annotations.Xml
import ru.yandex.direct.http.smart.core.Call
import ru.yandex.direct.http.smart.http.Body
import ru.yandex.direct.http.smart.http.Headers
import ru.yandex.direct.http.smart.http.POST
import ru.yandex.direct.http.smart.http.Path
import ru.yandex.direct.http.smart.http.Query

interface MarketMbiApi {

    /**
     * Обёртка над ручкой /direct/feed/refresh
     * http://mbi-api.market.yandex.net:80/direct/feed/refresh
     *
     * Ручка отправляет урл фида на закачку товаров в DataCamp и возвращает идентификаторы бизнеса, партнера и фида
     * к которым этот урл привязан в Маркете.
     * Документация: https://wiki.yandex-team.ru/mbi/newdesign/components/mbi-api/direct/feed/refresh/
     */
    @POST("/direct/feed/refresh")
    @Xml
    fun refreshDirectFeed(@Body @Xml request: RefreshFeedRequest): Call<RefreshFeedResponse>

    @JacksonXmlRootElement(localName = "add-direct-feed-request")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    data class RefreshFeedRequest(
        @field:JacksonXmlProperty(localName = "uid") val uid: Long,
        @field:JacksonXmlProperty(localName = "direct-feed-id") val directFeedId: Long,
        @field:JacksonXmlProperty(localName = "client-id") val clientId: Long,
        @field:JacksonXmlProperty(localName = "feed-url") val feedUrl: String,
        @field:JacksonXmlProperty(localName = "feed-http-login") val login: String?,
        @field:JacksonXmlProperty(localName = "feed-http-password") val password: String?,
        @field:JacksonXmlProperty(localName = "enable-direct-placement") val enableDirectPlacement: Boolean?)

    @JacksonXmlRootElement(localName = "add-direct-feed-response")
    @JsonIgnoreProperties(ignoreUnknown = true)
    data class RefreshFeedResponse(
        @field:JacksonXmlProperty(localName = "market-feed-id") var marketFeedId: Long? = null,
        @field:JacksonXmlProperty(localName = "partner-id") var partnerId: Long? = null,
        @field:JacksonXmlProperty(localName = "business-id") var businessId: Long? = null
    )

    /**
     * Обёртка над ручкой /direct/site/refresh
     * http://mbi-api.market.yandex.net:80/direct/site/refresh
     *
     * Ручка отправляет урл сайта на закачку товаров в DataCamp и возвращает идентификаторы бизнеса, партнера и фида
     * к которым этот урл привязан в Маркете.
     * Документация: https://wiki.yandex-team.ru/mbi/newdesign/components/mbi-api/direct/site/refresh/
     */
    @POST("/direct/site/refresh")
    @Xml
    fun refreshDirectSite(@Body @Xml request: RefreshSiteRequest): Call<RefreshFeedResponse>

    @JacksonXmlRootElement(localName = "add-direct-site-request")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    data class RefreshSiteRequest(
        @field:JacksonXmlProperty(localName = "uid") val uid: Long,
        @field:JacksonXmlProperty(localName = "direct-feed-id") val directFeedId: Long,
        @field:JacksonXmlProperty(localName = "client-id") val clientId: Long,
        @field:JacksonXmlProperty(localName = "schema") val schema: String,
        @field:JacksonXmlProperty(localName = "domain") val domain: String,
        @field:JacksonXmlProperty(localName = "path") val path: String,
        @field:JacksonXmlProperty(localName = "enable-direct-placement") val enableDirectPlacement: Boolean?)

    /**
     * Обёртка над ручкой /direct/file/refresh
     * http://mbi-api.market.yandex.net:80/direct/file/refresh
     *
     * Ручка отправляет ссылку на файл фида в MDS на закачку товаров в DataCamp и возвращает идентификаторы бизнеса,
     * партнера и фида к которым этот файл привязан в Маркете.
     * Документация: https://wiki.yandex-team.ru/mbi/newdesign/components/mbi-api/direct/file/refresh/
     */
    @POST("/direct/file/refresh")
    @Xml
    fun refreshDirectFile(@Body @Xml request: RefreshFileRequest): Call<RefreshFeedResponse>

    @JacksonXmlRootElement(localName = "add-direct-file-request")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    data class RefreshFileRequest(
        @field:JacksonXmlProperty(localName = "uid") val uid: Long,
        @field:JacksonXmlProperty(localName = "direct-feed-id") val directFeedId: Long,
        @field:JacksonXmlProperty(localName = "client-id") val clientId: Long,
        @field:JacksonXmlProperty(localName = "feed-url") val feedUrl: String,
        @field:JacksonXmlProperty(localName = "refresh") val refresh: Boolean,
        @field:JacksonXmlProperty(localName = "enable-direct-placement") val enableDirectPlacement: Boolean?)

    /**
     * Обёртка над ручкой /direct/site/preview
     * http://mbi-api.market.yandex.net:80/direct/site/preview
     *
     * Ручка отправляет урл сайта на закачку превью товаров в DataCamp и возвращает идентификаторы бизнеса,
     * партнера и фида по которым отправленный сайт можно будет идентифицировать в Маркете.
     */
    @POST("/direct/site/preview")
    @Xml
    fun addDirectSitePreview(@Body @Xml request: PreviewSiteRequest): Call<RefreshFeedResponse>

    @JacksonXmlRootElement(localName = "add-direct-site-preview-request")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    data class PreviewSiteRequest(
        @field:JacksonXmlProperty(localName = "schema") val schema: String,
        @field:JacksonXmlProperty(localName = "domain") val domain: String,
        @field:JacksonXmlProperty(localName = "path") val path: String,
        @field:JacksonXmlProperty(localName = "num-offers") val numOffers: Int
    )

    /**
     * Обертка над ручкой /direct/site/delete-preview
     *
     * Ручка позволяющая удалять из MBI сайты для превью, добавленные ранее через /direct/site/preview,
     * по списку идентификаторов (shop_id + feed_id). Количество сайтов для удаления ограничено 1000.
     * Возвращает список удаленных сайтов для превью. Если id сайта отсутствует в ответе - он не был найден в MBI
     */
    @POST("/direct/site/delete-preview")
    @Json
    @Headers("Content-Type: application/json")
    fun deleteDirectSitePreview(@Body @Json request: DeleteSitePreviewsRequest): Call<DeleteSitePreviewsResponse>

    data class DeleteSitePreviewsRequest(
        @field:JsonProperty("feedIds") val feedIds: List<DirectSitePreviewIds>
    )

    data class DeleteSitePreviewsResponse(
        @field:JsonProperty("feedIds") val feedIds: List<DirectSitePreviewIds>
    )

    data class DirectSitePreviewIds(
        @field:JsonProperty("partnerId") val partnerId: Long,
        @field:JsonProperty("feedId") val feedId: Long
    )

    /**
     * Обёртка над ручкой /direct/shop/features
     * http://mbi-api.market.yandex.net:80/direct/shop/features
     *
     * Ручка позволяющая установить свойства уже зарегистрированного в MBI фида
     * Планируется использовать ее для нотификации о том, в каких типах групп используются фиды, а соответственно,
     * какой для их офферов должен быть выбран pipeline
     *
     * Документация: TBD
     */
    @POST("/direct/shop/features")
    @Xml
    fun setDirectFeedFeatures(@Body @Xml request: SetFeedFeatureRequest): Call<Any>

    @JacksonXmlRootElement(localName = "set-direct-feed-feature-request")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    data class SetFeedFeatureRequest(
        @field:JacksonXmlProperty(localName = "shop-id") val shopId: Long,
        @field:JacksonXmlProperty(localName = "is-standby") val standby: Boolean,
        @field:JacksonXmlProperty(localName = "is-goods-ads") val goodsAds: Boolean,
        @field:JacksonXmlProperty(localName = "is-search-snippet-gallery") val searchSnippetGallery: Boolean,
    )


    /**
     * Обертка над ручкой /partner/{partnerId}/direct/disable
     * http://mbi-api.market.yandex.net:80/partner/{partnerId}/direct/disable
     *
     * Ручка для выключения размещения магазина директа.
     * В случае выключения магазина директа MBI передает эту информацию в datacamp через shops.dat.
     * Если товары магазина размещаются только в директе, то магазин выключается целиком. Если товары маркетного
     * магазина размещаются в директе, то магазин отключается только от директа и сохраняет размещение
     * на маркете.
     */
    @POST("/partner/{partnerId}/direct/disable")
    fun disableShop(@Path("partnerId") shopId: Long, @Query("uid") uid: Long): Call<Any>

    /**
     * Обертка над ручкой /partner/{partnerId}/direct/enable
     * http://mbi-api.market.yandex.net:80/partner/{partnerId}/direct/enable
     *
     * Ручка для включения размещения магазина директа.
     * В случае включения магазина директа MBI передает эту информацию в datacamp через shops.dat.
     * Если товары магазина размещаются только в директе, то магазин включается целиком. Если товары маркетного
     * магазина размещаются в директе и магазин включен на маркете, то магазин включится для директа. Если магазин
     * выключен в маркете, то для директа он также останется выключенным до включения в маркете.
     */
    @POST("/partner/{partnerId}/direct/enable")
    fun enableShop(@Path("partnerId") shopId: Long, @Query("uid") uid: Long): Call<Any>

}
