package ru.yandex.direct.mysql2grut.enummappers

import NYT.NYson.NProto.ProtobufInterop
import com.google.protobuf.ProtocolMessageEnum
import org.jooq.EnumType
import org.slf4j.LoggerFactory
import java.lang.invoke.MethodHandles
import java.lang.reflect.Method
import java.lang.reflect.Modifier

private val logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass())

/**
 * Создать словарь соответствия для двух Enum'ов.
 * Пытается автомагически сопоставить элементы по их литеральному значению (см. createLiteralAccessor),
 * плюс добавляет маппинг, заданный вручную (нужен, если у некоторых значений отличаются литералы).
 *
 * Логирует значения исходного Enum, для которых не нашлось пары.
 *
 * @param fromClass                 тип исходного Enum
 * @param toClass                   тип целевого Enum
 * @param customMapping             вручную заданные соответствия
 * @param customFromLiteralAccessor вручную заданный способ получения литерального значения для From
 * @param customToLiteralAccessor   вручную заданный способ получения литерального значения для To
 * @return словарь пар исходный-целевой Enum. может содержать меньше значений, чем исходный Enum (если не нашлась пара)
 */
fun <From : Enum<From>, To : Enum<To>> buildEnumMapping(
    fromClass: Class<From>,
    toClass: Class<To>,
    customMapping: Map<From, To> = emptyMap(),
    customFromLiteralAccessor: Function1<From, String>? = null,
    customToLiteralAccessor: Function1<To, String>? = null,
): Map<From, To> {
    val toLiteralAccessor = customToLiteralAccessor ?: createLiteralAccessor(toClass)
    val candidatesTo = toClass.enumConstants
        .associateBy(toLiteralAccessor)
        .filterKeys { it != null }

    val fromLiteralAccessor = customFromLiteralAccessor ?: createLiteralAccessor(fromClass)
    val result = fromClass.enumConstants
        .associateWith(fromLiteralAccessor)
        .filterValues { it != null }
        .filterValues { it in candidatesTo }
        .mapValues { candidatesTo[it.value]!! }
        .plus(customMapping)

    fromClass.enumConstants
        .filterNot { it in result.keys }
        .forEach {
            logger.info("{}.{} doesn't have mapped value from {}", fromClass.simpleName, it, toClass.simpleName)
        }

    return result
}

/**
 * Создает функцию, получающую из Enum'а его литеральное значение (или null).
 *
 * Поддерживает следующие типы:
 *      1. Enum, созданный model-generator'ом из MySQL'ного (отличается по методу toSource). Значение конвертируется
 * в sql-тип, и у него берется getLiteral()
 *      2. Enum, сгенерированный по proto описанию, в котором задано расширение NYT.NYson.NProto.enum_value_name.
 * "Литералом" в данном случае считается значение этого расширения.
 *      3. Enum, у которого есть метод getLiteral, возвращающий строку (пример: AdgroupAdditionalTargetingsTargetingType)
 *      4. Enum, у которого есть метод getTypedValue, возвращающий строку (пример: InterfaceLangs)
 */
private fun <T : Enum<T>> createLiteralAccessor(enumClass: Class<T>): Function1<T, String?> {
    if (ProtocolMessageEnum::class.java.isAssignableFrom(enumClass)) {
        return { enumValue: T ->
            val protoValue = enumValue as ProtocolMessageEnum
            if (protoValue.valueDescriptor.options.hasExtension(ProtobufInterop.enumValueName)) {
                protoValue.valueDescriptor.options.getExtension(ProtobufInterop.enumValueName)
            } else {
                null
            }
        }
    }

    val toSourceMethod = enumClass.declaredMethods
        .filter { m: Method -> Modifier.isStatic(m.modifiers) && Modifier.isPublic(m.modifiers) }
        .filter { m: Method -> EnumType::class.java.isAssignableFrom(m.returnType) }
        .firstOrNull { m: Method -> "toSource" == m.name }
    if (toSourceMethod != null) {
        // direct db-mapped enum
        return { enumValue: T ->
            try {
                toSourceMethod.invoke(null, enumValue)
                    .let { it as EnumType }
                    .literal
            } catch (e: Exception) {
                logger.debug("Failed to get literal value for {}.{}", enumClass.simpleName, enumValue, e)
                null
            }
        }
    }
    val literalMethod = enumClass.declaredMethods
        .filter { m: Method -> Modifier.isPublic(m.modifiers) }
        .firstOrNull { m: Method -> "getLiteral" == m.name && m.returnType == String::class.java }
    if (literalMethod != null) {
        return invokeStringResultMethod(literalMethod, enumClass.simpleName)
    }
    val typedValueMethod = enumClass.declaredMethods
        .filter { m: Method -> Modifier.isPublic(m.modifiers) }
        .firstOrNull { m: Method -> "getTypedValue" == m.name && m.returnType == String::class.java }    // других эвристик (пока?) не делаем
    if (typedValueMethod != null) {
        return invokeStringResultMethod(typedValueMethod, enumClass.simpleName)
    }
    return { _: T -> null }
}

private fun <T> invokeStringResultMethod(method: Method, className: String): (T) -> String? {
    return { enumValue: T ->
        try {
            method.invoke(enumValue) as String
        } catch (e: Exception) {
            logger.debug("Failed to get literal value for {}.{}", className, enumValue, e)
            null
        }
    }
}

/**
 * Общий метод конвертации Enum'а из одного типа в другой с использованием словаря с маппингом.
 * При остутствии маппинга — пишет debug сообщение в лог, и возвращает переданный unknownValue.
 */
fun <From : Enum<From>, To : Enum<To>> convert(mapping: Map<From, To>, unknownValue: To, value: From): To {
    return mapping[value]
        ?: unknownValue.also { logger.debug("Can't map enum value: ${value.javaClass.simpleName}.$value") }
}

/**
 * Общий метод конвертации Enum'а из одного типа в другой с использованием словаря с маппингом.
 * При остутствии маппинга — возращает null
 */
fun <From : Enum<From>, To : Enum<To>> convertWithDefaultNull(mapping: Map<From, To>, value: From): To? {
    return mapping[value]
}
