package ru.yandex.direct.mysql.slowlog.writer.states;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.utils.JsonUtilsWithMapper;

public class MySqlClusterSlowLogsWriterStateProvider {
    private static final Logger logger = LoggerFactory.getLogger(MySqlClusterSlowLogsWriterStateProvider.class);

    public static final String PPC_PROPERTY_PREFIX = "mysql_slow_query_logs_writer_".toLowerCase();
    public static final String PPC_PROPERTY_SUFFIX = "_cluster_state".toLowerCase();

    private final PpcPropertiesSupport ppcPropertiesSupport;

    public MySqlClusterSlowLogsWriterStateProvider(PpcPropertiesSupport ppcPropertiesSupport) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    public List<String> getClustersNames() {
        List<String> propertiesNames = ppcPropertiesSupport.getAllNamesByPrefix(PPC_PROPERTY_PREFIX);
        List<String> clusterNames = new ArrayList<>(propertiesNames.size());
        for (String propertyName: propertiesNames) {
            propertyName = propertyName.toLowerCase();
            if (!propertyName.endsWith(PPC_PROPERTY_SUFFIX)) {
                logger.warn("MySQL slow logs writer cluster state property '{}' do not ends with {}, skipping",
                        propertyName, PPC_PROPERTY_SUFFIX);
                continue;
            }
            String clusterName = propertyName.substring(
                    PPC_PROPERTY_PREFIX.length(), propertyName.length() - PPC_PROPERTY_SUFFIX.length());
            clusterNames.add(clusterName);
        }
        return clusterNames;
    }

    public Map<String, MySqlClusterSlowLogsWriterFullStateInfo> getClustersStates() {
        Map<String, String> propertiesMap = ppcPropertiesSupport.getByPrefix(PPC_PROPERTY_PREFIX);
        Map<String, MySqlClusterSlowLogsWriterFullStateInfo> clustersMap = new HashMap<>();
        for (Map.Entry<String, String> entry: propertiesMap.entrySet()) {
            String propertyName = entry.getKey().toLowerCase();
            if (!propertyName.endsWith(PPC_PROPERTY_SUFFIX)) {
                logger.warn("MySQL slow logs writer cluster state property '{}' do not ends with {}, skipping",
                        propertyName, PPC_PROPERTY_SUFFIX);
                continue;
            }
            String clusterName = propertyName.substring(
                    PPC_PROPERTY_PREFIX.length(), propertyName.length() - PPC_PROPERTY_SUFFIX.length());
            MySqlClusterSlowLogsWriterStateInfo clusterState =
                    JsonUtilsWithMapper.JSON_UTILS_WITH_HUMAN_READABLE_DATES.fromJson(
                            entry.getValue(), MySqlClusterSlowLogsWriterStateInfo.class);
            clustersMap.put(clusterName, new MySqlClusterSlowLogsWriterFullStateInfo(entry.getValue(), clusterState));
        }
        return clustersMap;
    }

    @NotNull
    public String createPropertyName(String clusterName) {
        return PPC_PROPERTY_PREFIX + clusterName.toLowerCase() + PPC_PROPERTY_SUFFIX;
    }

    public MySqlClusterSlowLogsWriterFullStateInfo getClusterState(String clusterName) {
        String propertyName = createPropertyName(clusterName);
        String clusterStateAsString = ppcPropertiesSupport.get(propertyName);
        if (clusterStateAsString == null) {
            return null;
        }
        MySqlClusterSlowLogsWriterStateInfo clusterState = JsonUtilsWithMapper.JSON_UTILS_WITH_HUMAN_READABLE_DATES.
                fromJson(clusterStateAsString, MySqlClusterSlowLogsWriterStateInfo.class);
        return new MySqlClusterSlowLogsWriterFullStateInfo(clusterStateAsString, clusterState);
    }

    public MySqlClusterSlowLogsWriterFullStateInfo compareAndSwap(
            String clusterName, MySqlClusterSlowLogsWriterFullStateInfo newStateWithOldCasValue) {
        String propertyName = createPropertyName(clusterName);
        String newValue = JsonUtilsWithMapper.JSON_UTILS_WITH_HUMAN_READABLE_DATES
                .toJson(newStateWithOldCasValue.getClusterState());
        boolean success = ppcPropertiesSupport.cas(
                propertyName, newStateWithOldCasValue.getCompareAndSwapValue(), newValue);
        if (!success) {
            return null;
        }
        return new MySqlClusterSlowLogsWriterFullStateInfo(newValue, newStateWithOldCasValue.getClusterState());
    }

    public boolean remove(String clusterName) {
        String propertyName = createPropertyName(clusterName);
        return ppcPropertiesSupport.remove(propertyName);
    }
}
