package ru.yandex.direct.oneshot.core.entity.oneshot.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.dbschema.ppcdict.enums.OneshotLaunchDataLaunchStatus;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.oneshot.core.model.LaunchStatus;
import ru.yandex.direct.oneshot.core.model.OneshotLaunchData;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.stringSetProperty;
import static ru.yandex.direct.dbschema.ppcdict.tables.OneshotLaunchData.ONESHOT_LAUNCH_DATA;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class OneshotLaunchDataRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<OneshotLaunchData> jooqMapper;
    private final Collection<Field<?>> allReadableFields;

    private static OneshotLaunchDataLaunchStatus[] inProgressStatuses = new OneshotLaunchDataLaunchStatus[]{
            OneshotLaunchDataLaunchStatus.in_progress,
            OneshotLaunchDataLaunchStatus.pause_requested
    };

    public OneshotLaunchDataRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(OneshotLaunchData::new)
                .map(property(OneshotLaunchData.ID, ONESHOT_LAUNCH_DATA.LAUNCH_DATA_ID))
                .map(property(OneshotLaunchData.LAUNCH_ID, ONESHOT_LAUNCH_DATA.LAUNCH_ID))

                .map(convertibleProperty(OneshotLaunchData.SHARD, ONESHOT_LAUNCH_DATA.SHARD,
                        OneshotLaunchDataRepository::convertShardFromSource,
                        OneshotLaunchDataRepository::convertShardToSource))
                .map(property(OneshotLaunchData.SPAN_ID, ONESHOT_LAUNCH_DATA.SPAN_ID))
                .map(property(OneshotLaunchData.LAUNCH_TIME, ONESHOT_LAUNCH_DATA.LAUNCH_TIME))
                .map(property(OneshotLaunchData.LAST_ACTIVE_TIME, ONESHOT_LAUNCH_DATA.LAST_ACTIVE_TIME))
                .map(stringSetProperty(OneshotLaunchData.LAUNCHED_REVISIONS, ONESHOT_LAUNCH_DATA.LAUNCHED_REVISIONS))
                .map(convertibleProperty(OneshotLaunchData.LAUNCH_STATUS, ONESHOT_LAUNCH_DATA.LAUNCH_STATUS,
                        LaunchStatus::fromSource, LaunchStatus::toSource))

                .map(property(OneshotLaunchData.STATE, ONESHOT_LAUNCH_DATA.STATE))
                .map(property(OneshotLaunchData.FINISH_TIME, ONESHOT_LAUNCH_DATA.FINISH_TIME))
                .build();
        this.allReadableFields = jooqMapper.getFieldsToRead();
    }

    public LaunchStatus getLaunchStatus(Long launchDataId) {
        return dslContextProvider.ppcdict()
                .select(ONESHOT_LAUNCH_DATA.LAUNCH_STATUS)
                .from(ONESHOT_LAUNCH_DATA)
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_DATA_ID.eq(launchDataId))
                .fetchOne(rec -> LaunchStatus.fromSource(rec.get(ONESHOT_LAUNCH_DATA.LAUNCH_STATUS)));
    }

    public List<OneshotLaunchData> getAll() {
        return dslContextProvider.ppcdict()
                .select(allReadableFields)
                .from(ONESHOT_LAUNCH_DATA)
                .orderBy(ONESHOT_LAUNCH_DATA.LAUNCH_ID.desc(), ONESHOT_LAUNCH_DATA.SHARD)
                .fetch(jooqMapper::fromDb);
    }

    public OneshotLaunchData get(long id) {
        List<OneshotLaunchData> datas = get(singletonList(id));
        return datas.isEmpty() ? null : datas.get(0);
    }

    public List<OneshotLaunchData> get(List<Long> ids) {
        return dslContextProvider.ppcdict()
                .select(allReadableFields)
                .from(ONESHOT_LAUNCH_DATA)
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_DATA_ID.in(ids))
                .fetch(jooqMapper::fromDb);
    }


    @Nullable
    public OneshotLaunchData getReadyLaunchDataForUpdate(DSLContext dslContext) {
        List<OneshotLaunchData> readyDatas = getReadyLaunchDatasForUpdate(dslContext);
        return readyDatas.isEmpty() ? null : readyDatas.get(0);
    }

    public List<OneshotLaunchData> getReadyLaunchDatasForUpdate(DSLContext dslContext) {
        return dslContext
                .select(allReadableFields)
                .from(ONESHOT_LAUNCH_DATA)
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_STATUS.eq(OneshotLaunchDataLaunchStatus.ready))
                .forUpdate()
                .fetch(jooqMapper::fromDb);
    }

    public Map<Long, List<LaunchStatus>> getLaunchStatusesByLaunchId(Collection<Long> launchIds) {
        return EntryStream.of(dslContextProvider.ppcdict()
                .select(allReadableFields)
                .from(ONESHOT_LAUNCH_DATA)
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_ID.in(launchIds))
                .fetchGroups(ONESHOT_LAUNCH_DATA.LAUNCH_ID))
                .mapValues(r -> mapList(r.getValues(ONESHOT_LAUNCH_DATA.LAUNCH_STATUS), LaunchStatus::fromSource))
                .toMap();
    }

    public Map<Long, List<OneshotLaunchData>> getLaunchDataByLaunchId(Collection<Long> launchIds) {
        return dslContextProvider.ppcdict()
                .select(allReadableFields)
                .from(ONESHOT_LAUNCH_DATA)
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_ID.in(launchIds))
                .fetchGroups(ONESHOT_LAUNCH_DATA.LAUNCH_ID, jooqMapper::fromDb);
    }

    public List<OneshotLaunchData> selectNotActiveLaunchDatasForUpdate(DSLContext dslContext,
                                                                       LocalDateTime lastActiveTimeThreshold) {
        return dslContext
                .select(allReadableFields)
                .from(ONESHOT_LAUNCH_DATA)
                .where(ONESHOT_LAUNCH_DATA.LAST_ACTIVE_TIME.lessThan(lastActiveTimeThreshold))
                .and(ONESHOT_LAUNCH_DATA.LAUNCH_STATUS.in(inProgressStatuses))
                .forUpdate()
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Обновляет статусы потоков заданного запуска на новый. Обновляются те статусы, которые можно обновить на
     * заданный, что задается в {@code statusesAllowedForChanging}
     *
     * @param launchId                   - id запуска
     * @param newStatus                  - новый статус
     * @param statusesAllowedForChanging - статусы, которые можно изменить
     */
    public void updateLaunchDataStatuses(DSLContext dslContext, Long launchId,
                                         LaunchStatus newStatus,
                                         Set<LaunchStatus> statusesAllowedForChanging) {
        Set<OneshotLaunchDataLaunchStatus> dbStatuses = listToSet(statusesAllowedForChanging, LaunchStatus::toSource);
        dslContext
                .update(ONESHOT_LAUNCH_DATA)
                .set(ONESHOT_LAUNCH_DATA.LAUNCH_STATUS, LaunchStatus.toSource(newStatus))
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_ID.eq(launchId))
                .and(ONESHOT_LAUNCH_DATA.LAUNCH_STATUS.in(dbStatuses))
                .execute();
    }

    public void add(OneshotLaunchData oneshotLaunchData) {
        add(dslContextProvider.ppcdict(), singletonList(oneshotLaunchData));
    }

    public void add(List<OneshotLaunchData> oneshotLaunchDataList) {
        add(dslContextProvider.ppcdict(), oneshotLaunchDataList);
    }

    public void add(DSLContext dslContext, List<OneshotLaunchData> oneshotLaunchDataList) {
        var newIds = new InsertHelper<>(dslContext, ONESHOT_LAUNCH_DATA)
                .addAll(jooqMapper, oneshotLaunchDataList)
                .executeIfRecordsAddedAndReturn(ONESHOT_LAUNCH_DATA.LAUNCH_DATA_ID);
        StreamEx.of(newIds)
                .zipWith(StreamEx.of(oneshotLaunchDataList))
                .forKeyValue((id, launchData) -> launchData.setId(id));
    }

    public OneshotLaunchData selectForUpdate(DSLContext dslContext, Long launchDataId) {
        return dslContext.select(allReadableFields)
                .from(ONESHOT_LAUNCH_DATA)
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_DATA_ID.eq(launchDataId))
                .forUpdate()
                .fetchOne(jooqMapper::fromDb);
    }

    public int updateState(DSLContext dslContext, OneshotLaunchData oneshotLaunchData) {
        return dslContext
                .update(ONESHOT_LAUNCH_DATA)
                .set(ONESHOT_LAUNCH_DATA.STATE, oneshotLaunchData.getState())
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_DATA_ID.eq(oneshotLaunchData.getId()))
                .execute();
    }

    public int updateStatus(DSLContext dslContext, OneshotLaunchData oneshotLaunchData) {
        return dslContext
                .update(ONESHOT_LAUNCH_DATA)
                .set(ONESHOT_LAUNCH_DATA.LAUNCH_STATUS, LaunchStatus.toSource(oneshotLaunchData.getLaunchStatus()))
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_DATA_ID.eq(oneshotLaunchData.getId()))
                .execute();
    }

    public int updateStatusAndFinishTime(DSLContext dslContext, OneshotLaunchData oneshotLaunchData) {
        return dslContext
                .update(ONESHOT_LAUNCH_DATA)
                .set(ONESHOT_LAUNCH_DATA.LAUNCH_STATUS, LaunchStatus.toSource(oneshotLaunchData.getLaunchStatus()))
                .set(ONESHOT_LAUNCH_DATA.FINISH_TIME, oneshotLaunchData.getFinishTime())
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_DATA_ID.eq(oneshotLaunchData.getId()))
                .execute();
    }

    public int updateFinishTime(DSLContext dslContext, OneshotLaunchData oneshotLaunchData) {
        return dslContext
                .update(ONESHOT_LAUNCH_DATA)
                .set(ONESHOT_LAUNCH_DATA.FINISH_TIME, oneshotLaunchData.getFinishTime())
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_DATA_ID.eq(oneshotLaunchData.getId()))
                .execute();
    }

    public void updateLaunchDataForStart(DSLContext dslContext,
                                         OneshotLaunchData launchData) {
        dslContext
                .update(ONESHOT_LAUNCH_DATA)
                .set(ONESHOT_LAUNCH_DATA.LAUNCH_STATUS, LaunchStatus.toSource(launchData.getLaunchStatus()))
                .set(ONESHOT_LAUNCH_DATA.LAUNCH_TIME, launchData.getLaunchTime())
                .set(ONESHOT_LAUNCH_DATA.LAST_ACTIVE_TIME, launchData.getLastActiveTime())
                .set(ONESHOT_LAUNCH_DATA.LAUNCHED_REVISIONS,
                        RepositoryUtils.setToDb(launchData.getLaunchedRevisions(), Function.identity()))
                .set(ONESHOT_LAUNCH_DATA.SPAN_ID, launchData.getSpanId())
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_DATA_ID.eq(launchData.getId()))
                .execute();
    }

    public void updateLastActiveTime(DSLContext dslContext, List<OneshotLaunchData> launchDatas,
                                     LocalDateTime lastActiveTime) {
        var idsForUpdate = StreamEx.of(launchDatas).map(OneshotLaunchData::getId).toSet();
        dslContext
                .update(ONESHOT_LAUNCH_DATA)
                .set(ONESHOT_LAUNCH_DATA.LAST_ACTIVE_TIME, lastActiveTime)
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_DATA_ID.in(idsForUpdate))
                .execute();
        // update local model
        launchDatas.forEach(launchData -> launchData.setLastActiveTime(lastActiveTime));
    }

    public void pauseLaunchDatas(DSLContext dslContext, List<OneshotLaunchData> launchDatas) {
        var idsForUpdate = StreamEx.of(launchDatas).map(OneshotLaunchData::getId).toSet();
        dslContext.update(ONESHOT_LAUNCH_DATA)
                .set(ONESHOT_LAUNCH_DATA.LAUNCH_STATUS, OneshotLaunchDataLaunchStatus.paused)
                .where(ONESHOT_LAUNCH_DATA.LAUNCH_DATA_ID.in(idsForUpdate))
                .execute();
        // update local model
        launchDatas.forEach(launchData -> launchData.setLaunchStatus(LaunchStatus.PAUSED));
    }

    private static Integer convertShardFromSource(Long shard) {
        return shard == 0L ? null : shard.intValue();
    }

    private static Long convertShardToSource(Integer shard) {
        return shard == null ? Long.valueOf(0) : Long.valueOf(shard);
    }
}
