package ru.yandex.direct.oneshot.core.entity.oneshot.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppcdict.enums.OneshotLaunchesValidationStatus;
import ru.yandex.direct.dbschema.ppcdict.tables.records.OneshotLaunchesRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.oneshot.core.model.OneshotLaunch;
import ru.yandex.direct.oneshot.core.model.OneshotLaunchValidationStatus;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singleton;
import static ru.yandex.direct.dbschema.ppcdict.tables.OneshotLaunches.ONESHOT_LAUNCHES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class OneshotLaunchRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<OneshotLaunch> jooqMapper;
    private final Collection<Field<?>> allReadableFields;

    public OneshotLaunchRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(OneshotLaunch::new)
                .map(property(OneshotLaunch.ID, ONESHOT_LAUNCHES.LAUNCH_ID))
                .map(property(OneshotLaunch.ONESHOT_ID, ONESHOT_LAUNCHES.ONESHOT_ID))
                .map(property(OneshotLaunch.LAUNCH_CREATOR, ONESHOT_LAUNCHES.LAUNCH_CREATOR))
                .map(property(OneshotLaunch.PARAMS, ONESHOT_LAUNCHES.PARAMS))
                .map(property(OneshotLaunch.TRACE_ID, ONESHOT_LAUNCHES.TRACE_ID))
                .map(convertibleProperty(OneshotLaunch.VALIDATION_STATUS, ONESHOT_LAUNCHES.VALIDATION_STATUS,
                        OneshotLaunchValidationStatus::fromSource, OneshotLaunchValidationStatus::toSource))
                .map(property(OneshotLaunch.APPROVER, ONESHOT_LAUNCHES.APPROVER))
                .map(property(OneshotLaunch.APPROVED_REVISION, ONESHOT_LAUNCHES.APPROVED_REVISION))
                .map(property(OneshotLaunch.LAUNCH_REQUEST_TIME, ONESHOT_LAUNCHES.LAUNCH_REQUEST_TIME))
                .build();
        allReadableFields = jooqMapper.getFieldsToRead();
    }

    public OneshotLaunch getSafe(Long id) {
        OneshotLaunch launch = get(id);
        checkState(launch != null, "there is no launch in database (launch_id = %s)", id);
        return launch;
    }

    public OneshotLaunch get(Long id) {
        return get(dslContextProvider.ppcdict(), id);
    }

    public OneshotLaunch getForUpdate(DSLContext dslContext, Long id) {
        return dslContext.select(allReadableFields)
                .from(ONESHOT_LAUNCHES)
                .where(ONESHOT_LAUNCHES.LAUNCH_ID.eq(id))
                .forUpdate()
                .fetchOne(jooqMapper::fromDb);
    }

    public OneshotLaunch get(DSLContext dslContext, Long id) {
        List<OneshotLaunch> launches = get(dslContext, singleton(id));
        return launches.isEmpty() ? null : launches.get(0);
    }

    public List<OneshotLaunch> get(DSLContext dslContext, Collection<Long> ids) {
        return getManyWhere(dslContext, ONESHOT_LAUNCHES.LAUNCH_ID.in(ids));
    }

    public List<OneshotLaunch> getByOneshotId(Long oneshotId) {
        return getManyWhere(ONESHOT_LAUNCHES.ONESHOT_ID.eq(oneshotId));
    }

    public List<OneshotLaunch> getByOneshotIds(Collection<Long> oneshotIds) {
        return getManyWhere(ONESHOT_LAUNCHES.ONESHOT_ID.in(oneshotIds));
    }

    private List<OneshotLaunch> getManyWhere(Condition whereCondition) {
        return getManyWhere(dslContextProvider.ppcdict(), whereCondition);
    }

    private List<OneshotLaunch> getManyWhere(DSLContext dslContext, Condition whereCondition) {
        return dslContext
                .select(allReadableFields)
                .from(ONESHOT_LAUNCHES)
                .where(whereCondition)
                .fetch(jooqMapper::fromDb);
    }

    @Nullable
    public OneshotLaunch getReadyLaunchForUpdate(DSLContext dslContext) {
        List<OneshotLaunch> readyLaunches = getReadyLaunchesForUpdate(dslContext, 1);
        return readyLaunches.isEmpty() ? null : readyLaunches.get(0);
    }

    public List<OneshotLaunch> getReadyLaunchesForUpdate(DSLContext dslContext, int limit) {
        return dslContext
                .select(allReadableFields)
                .from(ONESHOT_LAUNCHES)
                .where(ONESHOT_LAUNCHES.VALIDATION_STATUS.in(OneshotLaunchesValidationStatus.ready))
                .limit(limit)
                .forUpdate()
                .fetch(jooqMapper::fromDb);
    }

    public void updateLaunch(DSLContext dslContext, AppliedChanges<OneshotLaunch> changes) {
        JooqUpdateBuilder<OneshotLaunchesRecord, OneshotLaunch> builder =
                new JooqUpdateBuilder<>(ONESHOT_LAUNCHES.LAUNCH_ID, Collections.singletonList(changes));
        builder.processProperty(OneshotLaunch.VALIDATION_STATUS, ONESHOT_LAUNCHES.VALIDATION_STATUS,
                OneshotLaunchValidationStatus::toSource);
        builder.processProperty(OneshotLaunch.TRACE_ID, ONESHOT_LAUNCHES.TRACE_ID);
        builder.processProperty(OneshotLaunch.APPROVER, ONESHOT_LAUNCHES.APPROVER);
        builder.processProperty(OneshotLaunch.APPROVED_REVISION, ONESHOT_LAUNCHES.APPROVED_REVISION);
        builder.processProperty(OneshotLaunch.LAUNCH_REQUEST_TIME, ONESHOT_LAUNCHES.LAUNCH_REQUEST_TIME);

        dslContext.update(ONESHOT_LAUNCHES)
                .set(builder.getValues())
                .where(ONESHOT_LAUNCHES.LAUNCH_ID.in(builder.getChangedIds()))
                .execute();
    }

    public Long add(OneshotLaunch oneshotLaunch) {
        return new InsertHelper<>(dslContextProvider.ppcdict(), ONESHOT_LAUNCHES)
                .add(jooqMapper, oneshotLaunch).newRecord()
                .executeIfRecordsAddedAndReturn(ONESHOT_LAUNCHES.LAUNCH_ID).get(0);
    }

    public void updateValidationStatus(DSLContext dslContext, Long launchId, OneshotLaunchValidationStatus status) {
        dslContext
                .update(ONESHOT_LAUNCHES)
                .set(ONESHOT_LAUNCHES.VALIDATION_STATUS,
                        OneshotLaunchValidationStatus.toSource(status))
                .where(ONESHOT_LAUNCHES.LAUNCH_ID.eq(launchId))
                .execute();
    }

    public int deleteLaunch(DSLContext dslContext, Long launchId) {
        return dslContext
                .deleteFrom(ONESHOT_LAUNCHES)
                .where(ONESHOT_LAUNCHES.LAUNCH_ID.eq(launchId))
                .and(ONESHOT_LAUNCHES.LAUNCH_REQUEST_TIME.isNull())
                .execute();
    }
}
