package ru.yandex.direct.oneshot.core.entity.oneshot.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.oneshot.core.model.Oneshot;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.stringSetProperty;
import static ru.yandex.direct.dbschema.ppcdict.tables.Oneshots.ONESHOTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class OneshotRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<Oneshot> jooqMapper;
    private final Collection<Field<?>> allReadableFields;

    public OneshotRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(Oneshot::new)
                .map(property(Oneshot.ID, ONESHOTS.ID))
                .map(property(Oneshot.CLASS_NAME, ONESHOTS.CLASS_NAME))
                .map(stringSetProperty(Oneshot.APPROVERS, ONESHOTS.APPROVERS))
                .map(property(Oneshot.TICKET, ONESHOTS.TICKET))
                .map(property(Oneshot.CREATE_TIME, ONESHOTS.CREATE_TIME))
                .map(integerProperty(Oneshot.RETRIES, ONESHOTS.RETRIES))
                .map(integerProperty(Oneshot.RETRY_TIMEOUT_SECONDS, ONESHOTS.RETRY_TIMEOUT))
                .map(booleanProperty(Oneshot.PAUSED_STATUS_ON_FAIL, ONESHOTS.IS_PAUSED_ON_FAIL))
                .map(booleanProperty(Oneshot.MULTI_LAUNCH, ONESHOTS.IS_MULTI_LAUNCH))
                .map(booleanProperty(Oneshot.SAFE_ONESHOT, ONESHOTS.IS_SAFE_ONESHOT))
                .map(booleanProperty(Oneshot.SHARDED, ONESHOTS.IS_SHARDED))
                .map(booleanProperty(Oneshot.DELETED, ONESHOTS.IS_DELETED))
                .build();
        this.allReadableFields = jooqMapper.getFieldsToRead();
    }

    public Oneshot getSafe(Long id) {
        Oneshot oneshot = get(id);
        checkState(oneshot != null, "there is no oneshot in database (oneshot id = %s)", id);
        return oneshot;
    }

    public Oneshot get(Long id) {
        return get(dslContextProvider.ppcdict(), id);
    }

    public Oneshot get(DSLContext dslContext, Long oneshotId) {
        return getOneWhere(dslContext, ONESHOTS.ID.eq(oneshotId));
    }

    public Oneshot getByClassName(String className) {
        return getOneWhere(ONESHOTS.CLASS_NAME.eq(className));
    }

    private Oneshot getOneWhere(Condition whereCondition) {
        return getOneWhere(dslContextProvider.ppcdict(), whereCondition);
    }

    private Oneshot getOneWhere(DSLContext dslContext, Condition whereCondition) {
        List<Oneshot> oneshots = getManyWhere(dslContext, whereCondition);
        return oneshots.isEmpty() ? null : oneshots.get(0);
    }

    public List<Oneshot> get(DSLContext dslContext, Collection<Long> ids) {
        return getManyWhere(dslContext, ONESHOTS.ID.in(ids));
    }

    public List<Oneshot> getAll() {
        return getManyWhere(DSL.noCondition());
    }

    public List<Oneshot> getExisting() {
        return getManyWhere(ONESHOTS.IS_DELETED.eq(0L));
    }

    private List<Oneshot> getManyWhere(Condition whereCondition) {
        return getManyWhere(dslContextProvider.ppcdict(), whereCondition);
    }

    private List<Oneshot> getManyWhere(DSLContext dslContext, Condition whereCondition) {
        return dslContext
                .select(allReadableFields)
                .from(ONESHOTS)
                .where(whereCondition)
                .fetch(jooqMapper::fromDb);
    }

    public Map<Long, Oneshot> getExistingById() {
        return dslContextProvider.ppcdict()
                .select(allReadableFields)
                .from(ONESHOTS)
                .where(ONESHOTS.IS_DELETED.isFalse())
                .fetchMap(ONESHOTS.ID, jooqMapper::fromDb);
    }

    public Oneshot lockingRead(DSLContext ctx, Long oneshotId) {
        return ctx.select(allReadableFields)
                .from(ONESHOTS)
                .where(ONESHOTS.ID.eq(oneshotId))
                .forUpdate()
                .fetchOne(jooqMapper::fromDb);
    }

    public void add(DSLContext dslContext, List<Oneshot> oneshots) {
        new InsertHelper<>(dslContext, ONESHOTS)
                .addAll(jooqMapper, oneshots)
                .executeIfRecordsAdded();
    }

    public Long add(Oneshot oneshot) {
        return new InsertHelper<>(dslContextProvider.ppcdict(), ONESHOTS)
                .add(jooqMapper, oneshot).newRecord()
                .executeIfRecordsAddedAndReturn(ONESHOTS.ID).get(0);
    }

    public void update(DSLContext dslContext, List<AppliedChanges<Oneshot>> appliedChanges) {
        if (appliedChanges.isEmpty()) {
            return;
        }

        var ub = new JooqUpdateBuilder<>(ONESHOTS.ID, appliedChanges);
        ub.processProperty(
                Oneshot.APPROVERS, ONESHOTS.APPROVERS,
                set -> RepositoryUtils.setToDb(set, Function.identity()));
        ub.processProperty(Oneshot.RETRIES, ONESHOTS.RETRIES, RepositoryUtils::intToLong);
        ub.processProperty(Oneshot.RETRY_TIMEOUT_SECONDS, ONESHOTS.RETRY_TIMEOUT, RepositoryUtils::intToLong);
        ub.processProperty(Oneshot.PAUSED_STATUS_ON_FAIL, ONESHOTS.IS_PAUSED_ON_FAIL, RepositoryUtils::booleanToLong);
        ub.processProperty(Oneshot.MULTI_LAUNCH, ONESHOTS.IS_MULTI_LAUNCH, RepositoryUtils::booleanToLong);
        ub.processProperty(Oneshot.SHARDED, ONESHOTS.IS_SHARDED, RepositoryUtils::booleanToLong);
        ub.processProperty(Oneshot.SAFE_ONESHOT, ONESHOTS.IS_SAFE_ONESHOT, RepositoryUtils::booleanToLong);
        ub.processProperty(Oneshot.DELETED, ONESHOTS.IS_DELETED, RepositoryUtils::booleanToLong);

        dslContext.update(ONESHOTS)
                .set(ub.getValues())
                .where(ONESHOTS.ID.in(ub.getChangedIds()))
                .execute();
    }
}
