package ru.yandex.direct.oneshot.core.entity.oneshot.service;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.stereotype.Service;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.core.entity.oneshot.model.ComplexOneshot;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotLaunchRepository;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotRepository;
import ru.yandex.direct.oneshot.core.model.Oneshot;
import ru.yandex.direct.oneshot.core.model.OneshotLaunch;
import ru.yandex.direct.oneshot.core.model.OneshotLaunchValidationStatus;
import ru.yandex.direct.tracing.util.TraceUtil;

import static com.google.common.collect.Iterables.isEmpty;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class OneshotService {
    private final DslContextProvider dslContextProvider;
    private final OneshotRepository oneshotRepository;
    private final OneshotLaunchRepository oneshotLaunchRepository;

    public OneshotService(
            DslContextProvider dslContextProvider,
            OneshotRepository oneshotRepository,
            OneshotLaunchRepository oneshotLaunchRepository
    ) {
        this.dslContextProvider = dslContextProvider;
        this.oneshotRepository = oneshotRepository;
        this.oneshotLaunchRepository = oneshotLaunchRepository;
    }

    public ComplexOneshot getByClassName(String className) {
        Oneshot oneshot = oneshotRepository.getByClassName(className);
        if (oneshot == null) {
            return null;
        }
        //noinspection OptionalGetWithoutIsPresent
        return getComplexOneshots(Collections.singletonList(oneshot)).findFirst().get();
    }

    public List<ComplexOneshot> getAll() {
        return getComplexOneshots(oneshotRepository.getAll()).collect(toList());
    }

    public List<ComplexOneshot> getLaunchableOneshots() {
        return getComplexOneshots(oneshotRepository.getExisting())
                .filter(ComplexOneshot::isLaunchable)
                .collect(toList());
    }

    private Stream<ComplexOneshot> getComplexOneshots(List<Oneshot> oneshots) {
        List<Long> ids = mapList(oneshots, Oneshot::getId);
        HashSet<Long> launchedOneshotIds = oneshotLaunchRepository.getByOneshotIds(ids).stream()
                .map(OneshotLaunch::getOneshotId)
                .collect(Collectors.toCollection(HashSet::new));

        return oneshots.stream()
                .map(oneshot -> makeComplexOneshot(oneshot, launchedOneshotIds.contains(oneshot.getId())));
    }

    private ComplexOneshot makeComplexOneshot(Oneshot oneshot, boolean hasLaunches) {
        return new ComplexOneshot()
                .withOneshot(oneshot)
                .withHasLaunches(hasLaunches);
    }

    public OneshotLaunch launch(Long oneshotId, String operatorLogin, @Nullable String param) {
        return dslContextProvider.ppcdict().transactionResult(conf -> {
            DSLContext ctx = conf.dsl();

            Oneshot oneshot = oneshotRepository.lockingRead(ctx, oneshotId);
            if (oneshot.getDeleted()) {
                return null;
            }
            List<OneshotLaunch> launches = oneshotLaunchRepository.getByOneshotId(oneshotId);
            if (!oneshot.getMultiLaunch() && !isEmpty(launches)) {
                return null;
            }

            OneshotLaunch launch = createLaunch(oneshot, operatorLogin, param);
            Long launchId = oneshotLaunchRepository.add(launch);
            launch.setId(launchId);
            return launch;
        });
    }

    private OneshotLaunch createLaunch(Oneshot oneshot, String operatorLogin, @Nullable String params) {
        return new OneshotLaunch()
                .withOneshotId(oneshot.getId())
                .withLaunchCreator(operatorLogin)
                .withParams(params)
                .withTraceId(TraceUtil.randomId())
                .withValidationStatus(OneshotLaunchValidationStatus.READY);
    }
}
