package ru.yandex.direct.oneshot.core.entity.oneshot.service;

import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.oneshot.core.model.Oneshot;
import ru.yandex.direct.utils.io.FileUtils;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.StartrekClientBuilder;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueCreate;
import ru.yandex.startrek.client.model.IssueUpdate;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.ThreadUtils.sleep;

@Service
@ParametersAreNonnullByDefault
public class OneshotStartrekService {
    private static final Logger logger = LoggerFactory.getLogger(OneshotStartrekService.class);

    private static final String QUEUE = "DIRECTONESHOT";

    private static final String TICKET_TYPE = "task";

    private final String internalToolsPrefix;
    private final String startrekUrl;

    private final Session stSession;

    public OneshotStartrekService(DirectConfig directConfig, EnvironmentType environmentType) {
        String oauthToken = "";
        try {
            oauthToken = FileUtils.slurp(FileUtils.expandHome(directConfig.getString(
                    "startrek.robot_dt_oneshots.token_file"))).trim();
        } catch (Exception e) {
            logger.warn("Can't read token for @robot-dt-oneshots", e);
        }

        String host = environmentType.equals(EnvironmentType.DEVTEST) ||
                environmentType.equals(EnvironmentType.DEV7) ?
                System.getProperty("beta.url") :
                directConfig.getString("web.host");
        this.internalToolsPrefix = host + "/internal_tools/";

        // используем тестовый стартрек, если окружение не продовое
        String apiUrl;
        if (environmentType.equals(EnvironmentType.PRODUCTION)) {
            apiUrl = "https://st-api.yandex-team.ru";
            this.startrekUrl = "https://st.yandex-team.ru";
        } else {
            apiUrl = "https://st-api.test.yandex-team.ru";
            this.startrekUrl = "https://st.test.yandex-team.ru";
        }
        this.stSession = StartrekClientBuilder.newBuilder()
                .uri(apiUrl)
                .maxConnections(10)
                .build(oauthToken);
    }

    public String createTicket(Oneshot oneshot) {
        checkState(oneshot.getTicket() == null, "Oneshot already has a ticket");

        IssueCreate.Builder builder = IssueCreate.builder()
                .queue(QUEUE)
                .type(TICKET_TYPE)
                .summary(oneshot.getClassName())
                .description(makeOneshotDescription(oneshot))
                .followers(oneshot.getApprovers().toArray(new String[]{}));

        Issue createdIssue = new RetryCommand<Issue>("Exception occurred during creating ticket for oneshot " +
                oneshot.getClassName()).run(() -> stSession.issues().create(builder.build()));

        if (createdIssue != null) {
            return createdIssue.getKey();
        }
        return null;
    }

    public void closeTicket(String ticketKey) {
        new RetryCommand<>("Exception occurred during closing ticket " + ticketKey).run(() ->
                stSession.transitions()
                        .execute(ticketKey, "close",
                                IssueUpdate
                                        .resolution("fixed")
                                        .comment(
                                                CommentCreate
                                                        .comment("Ваншот был удален из репозитория, тикет закрываю")
                                                        .build())
                                        .build()));
    }

    public void reopenTicket(String ticketKey) {
        new RetryCommand<>("Exception occurred during reopening ticket " + ticketKey).run(() ->
                stSession.transitions()
                        .execute(ticketKey, "reopen",
                                IssueUpdate
                                        .comment(
                                                CommentCreate
                                                        .comment("Ваншот снова появился в репозитории, переоткрываю тикет")
                                                        .build())
                                        .build()));
    }

    public void writeRequestLaunchComment(String ticketKey, Long launchId, String login, String toolName) {
        writeComment(ticketKey, "@" + login + " запросил запуск ваншота (id запуска: " + launchId +
                ")\n\nПодтвердить можно на ((" + makeInternalToolUrl(toolName) + " странице с запусками))");
    }

    public void writeLaunchApprovedComment(String ticketKey, Long launchId, String login, String toolName) {
        writeComment(ticketKey, "@" + login + " подтвердил запуск (id запуска: " + launchId +
                ")\n\nЗапустить можно на ((" + makeInternalToolUrl(toolName) + " странице с запусками))");
    }

    public void writeDeleteLaunchComment(String ticketKey, Long launchId, String login) {
        writeComment(ticketKey, "@" + login + " удалил запуск (id запуска: " + launchId + ")");
    }

    public void writeStartLaunchComment(String ticketKey, Long launchId, String login, String toolName) {
        writeComment(ticketKey, "@" + login + " запустил ваншот (id запуска: " + launchId +
                ")\n\nУправлять ходом выполнения ваншота можно на ((" + makeInternalToolUrl(toolName) +
                " странице с потоками выполнения))");
    }

    private void writeComment(String ticketKey, String comment) {
        new RetryCommand<>("Exception occurred during writing comment to ticket " + ticketKey).run(() ->
                stSession.comments().create(ticketKey, CommentCreate.comment(comment).build()));
    }

    private String makeInternalToolUrl(String toolName) {
        return this.internalToolsPrefix + "#" + toolName;
    }

    private String makeOneshotDescription(Oneshot oneshot) {
        StringBuilder sb = new StringBuilder();
        sb.append("Класс ваншота: ").append(oneshot.getClassName()).append("\n");
        sb.append("Тип: ").append(oneshot.getSharded() ? "Шардированный": "Нешардированный").append("\n");
        sb.append("Дата создания: ").append(oneshot.getCreateTime()).append("\n");
        sb.append("Можно ли запускать несколько раз: ").append(oneshot.getMultiLaunch() ? "Да": "Нет").append("\n");
        sb.append("Апруверы: ").append(oneshot.getApprovers());
        return sb.toString();
    }

    public String getTicketUrl(String ticketKey) {
        return this.startrekUrl + "/" + ticketKey;
    }

    private class RetryCommand<T> {
        private int retries = 3;
        private long delay = 500L;
        private final String errorMessage;

        private RetryCommand(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        private RetryCommand withRetries(int retries) {
            this.retries = retries;
            return this;
        }

        private RetryCommand withDelay(long delay) {
            this.delay = delay;
            return this;
        }

        private T run(Supplier<T> function) {
            for (int i = 0; i < retries; ++i) {
                try {
                    return function.get();
                } catch (Exception e) {
                    if (i + 1 >= retries) {
                        logger.error(String.format("%s \"%s\"", errorMessage, e));
                        break;
                    }
                    sleep(delay);
                }
            }
            return null;
        }
    }
}
