package ru.yandex.direct.rbac;

import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.direct.dbutil.model.ClientId;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableSet;

/**
 * Неизменяемый объект, содержащий базовую информацию о правах клиента
 */
public class ClientPerminfo {
    @Nonnull
    private final ClientId clientId;
    private final Long chiefUid;
    @Nonnull
    private final RbacRole role;
    @Nullable
    private final RbacSubrole subrole;
    private final ClientId agencyClientId;
    private final Set<Long> agencyUids;
    private final Long managerUid;
    private final Set<Long> managerUids;
    private final Set<ClientPerm> perms;
    private final boolean canHaveRelationship;
    private final Set<Long> mccClientIds;
    private final Set<Long> managerGroupIds;
    private final Set<Long> clientGroupIds;

    @SuppressWarnings("checkstyle:parameternumber")
    public ClientPerminfo(ClientId clientId, Long chiefUid,
                          RbacRole role, RbacSubrole subrole,
                          ClientId agencyClientId, Set<Long> agencyUids, Long managerUid, Set<Long> managerUids,
                          Set<ClientPerm> perms, boolean canHaveRelationship, Set<Long> mccClientIds,
                          Set<Long> managerGroupIds, Set<Long> clientGroupIds
    ) {
        this.clientId = checkNotNull(clientId);
        this.chiefUid = chiefUid;
        this.role = role != null ? role : RbacRole.EMPTY;
        this.subrole = subrole;
        this.agencyClientId = agencyClientId;
        this.agencyUids = agencyUids;
        this.managerUid = managerUid;
        this.managerUids = managerUids == null ? emptySet() : unmodifiableSet(managerUids);
        this.perms = perms == null ? emptySet() : unmodifiableSet(perms);
        this.canHaveRelationship = canHaveRelationship;
        this.mccClientIds = mccClientIds;
        this.managerGroupIds = managerGroupIds == null ? emptySet() : unmodifiableSet(managerGroupIds);
        this.clientGroupIds = clientGroupIds == null ? emptySet() : unmodifiableSet(clientGroupIds);
    }

    @Nonnull
    public ClientId clientId() {
        return clientId;
    }

    public Long chiefUid() {
        return chiefUid;
    }

    @Nonnull
    public RbacRole role() {
        return role;
    }

    @Nullable
    public RbacSubrole subrole() {
        return subrole;
    }

    public ClientId agencyClientId() {
        return agencyClientId;
    }

    public Set<Long> agencyUids() {
        return agencyUids;
    }

    public Long managerUid() {
        return managerUid;
    }

    public Set<Long> managerUids() {
        return managerUids;
    }

    public Set<ClientPerm> perms() {
        return perms;
    }

    /*
     * Признак того, что данный клиент может иметь доступ к данным других
     * клиентов через установку отношения клиент-клиент.
     * Сейчас такой доступ могут иметь только фрилансеры.
     **/
    public boolean canHaveRelationship() {
        return canHaveRelationship;
    }

    public Set<Long> mccClientIds() {
        return mccClientIds;
    }

    public boolean isMccForClient(Long clientId) {
        return mccClientIds.contains(clientId);
    }

    public boolean isMcc() {
        return !mccClientIds.isEmpty();
    }
    public boolean isAgencySubclient() {
        return role == RbacRole.CLIENT && agencyClientId != null;
    }

    public boolean isAgencySuperSubclient() {
        return isAgencySubclient() && hasPerm(ClientPerm.SUPER_SUBCLIENT);
    }

    /**
     * ID групп, имеющих доступ к текущему клиенту.
     *
     * @see #clientGroupIds()
     */
    public Set<Long> managerGroupIds() {
        return managerGroupIds;
    }

    /**
     * ID групп, в которых состоит клиент.
     * Только для внутренних пользователей!
     *
     * @see #managerGroupIds()
     */
    public Set<Long> clientGroupIds() {
        return clientGroupIds;
    }

    public boolean hasRole(RbacRole... expectedRoles) {
        for (RbacRole expectedRole : expectedRoles) {
            if (this.role == expectedRole) {
                return true;
            }
        }
        return false;
    }

    public boolean hasPerm(ClientPerm expectedPerm) {
        return perms.contains(expectedPerm);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ClientPerminfo that = (ClientPerminfo) o;
        return canHaveRelationship == that.canHaveRelationship &&
                clientId.equals(that.clientId) &&
                Objects.equals(chiefUid, that.chiefUid) &&
                role == that.role &&
                subrole == that.subrole &&
                Objects.equals(agencyClientId, that.agencyClientId) &&
                Objects.equals(agencyUids, that.agencyUids) &&
                Objects.equals(managerUid, that.managerUid) &&
                Objects.equals(managerUids, that.managerUids) &&
                Objects.equals(perms, that.perms) &&
                Objects.equals(managerGroupIds, that.managerGroupIds) &&
                Objects.equals(clientGroupIds, that.clientGroupIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(clientId, chiefUid, role, subrole, agencyClientId, agencyUids, managerUid, managerUids,
                perms, canHaveRelationship, managerGroupIds, clientGroupIds);
    }
}
