package ru.yandex.direct.rbac;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Functions;
import com.google.common.collect.ImmutableList;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Record2;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.ClientsAllowCreateScampBySubclient;
import ru.yandex.direct.dbschema.ppc.enums.ClientsRelationsType;
import ru.yandex.direct.dbschema.ppc.enums.ClientsRole;
import ru.yandex.direct.dbschema.ppc.enums.ReverseClientsRelationsType;
import ru.yandex.direct.dbschema.ppc.tables.Campaigns;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.rbac.cache.RbacCache;
import ru.yandex.direct.rbac.model.ClientsRelation;
import ru.yandex.direct.rbac.model.ClientsRelationType;
import ru.yandex.direct.rbac.model.RbacAccessType;
import ru.yandex.direct.rbac.model.RbacCampPerms;
import ru.yandex.direct.rbac.model.Representative;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonMap;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.dbschema.ppc.tables.AgencyLimRepClients.AGENCY_LIM_REP_CLIENTS;
import static ru.yandex.direct.dbschema.ppc.tables.AgencyManagers.AGENCY_MANAGERS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignsInternal.CAMPAIGNS_INTERNAL;
import static ru.yandex.direct.dbschema.ppc.tables.ClientManagers.CLIENT_MANAGERS;
import static ru.yandex.direct.dbschema.ppc.tables.Clients.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.tables.ClientsRelations.CLIENTS_RELATIONS;
import static ru.yandex.direct.dbschema.ppc.tables.Freelancers.FREELANCERS;
import static ru.yandex.direct.dbschema.ppc.tables.IdmGroupRoles.IDM_GROUP_ROLES;
import static ru.yandex.direct.dbschema.ppc.tables.IdmGroupsMembers.IDM_GROUPS_MEMBERS;
import static ru.yandex.direct.dbschema.ppc.tables.InternalAdManagerPlaceAccess.INTERNAL_AD_MANAGER_PLACE_ACCESS;
import static ru.yandex.direct.dbschema.ppc.tables.ManagerHierarchy.MANAGER_HIERARCHY;
import static ru.yandex.direct.dbschema.ppc.tables.ReverseClientsRelations.REVERSE_CLIENTS_RELATIONS;
import static ru.yandex.direct.dbschema.ppc.tables.Users.USERS;
import static ru.yandex.direct.dbschema.ppc.tables.UsersAgency.USERS_AGENCY;
import static ru.yandex.direct.dbschema.ppcdict.tables.PpcProperties.PPC_PROPERTIES;
import static ru.yandex.direct.rbac.ClientPerm.MONEY_TRANSFER;
import static ru.yandex.direct.rbac.ClientPerm.SUPER_SUBCLIENT;
import static ru.yandex.direct.rbac.ClientPerm.XLS_IMPORT;
import static ru.yandex.direct.rbac.RbacRepType.CHIEF;
import static ru.yandex.direct.rbac.RbacRepType.LIMITED;
import static ru.yandex.direct.rbac.RbacRepType.MAIN;
import static ru.yandex.direct.rbac.RbacRole.AGENCY;
import static ru.yandex.direct.rbac.RbacRole.CLIENT;
import static ru.yandex.direct.rbac.RbacRole.EMPTY;
import static ru.yandex.direct.rbac.RbacRole.LIMITED_SUPPORT;
import static ru.yandex.direct.rbac.RbacRole.MANAGER;
import static ru.yandex.direct.rbac.RbacRole.MEDIA;
import static ru.yandex.direct.rbac.RbacRole.PLACER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.ListUtils.uniqueList;

/**
 * Кусочки нового rbac - данные о ролях из ppcdata
 */
@ParametersAreNonnullByDefault
@Component
public class PpcRbac {
    public static final String CLIENTS_CACHE_BEAN_NAME = "rbacClientsPerminfoCache";
    public static final String USERS_CACHE_BEAN_NAME = "rbacUsersPerminfoCache";
    public static final String CAMP_PERMS_CACHE_BEAN_NAME = "rbacCampPermsCache";
    public static final Long NO_SUPERVISOR = 0L;
    /**
     * {@code RepositoryUtils.TRUE} недоступен, так как у rbac нет зависимости от common
     */
    private static final long TRUE = 1L;
    private static final Logger logger = LoggerFactory.getLogger(PpcRbac.class);
    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final RbacCache<ClientId, ClientPerminfo> clientsCache;
    private final RbacCache<Long, UserPerminfo> usersCache;
    private final RbacCache<OperatorCampIds, RbacCampPerms> campPermsCache;
    private final ObjectMapper objectMapper;
    private final RbacClientsRelationsStorage rbacClientsRelationsStorage;

    @Autowired
    public PpcRbac(
            DslContextProvider dslContextProvider,
            ShardHelper shardHelper,
            @Qualifier(CLIENTS_CACHE_BEAN_NAME) RbacCache<ClientId, ClientPerminfo> clientsCache,
            @Qualifier(CAMP_PERMS_CACHE_BEAN_NAME) RbacCache<OperatorCampIds, RbacCampPerms> campPermsCache,
            @Qualifier(USERS_CACHE_BEAN_NAME) RbacCache<Long, UserPerminfo> usersCache,
            RbacClientsRelationsStorage rbacClientsRelationsStorage) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.clientsCache = clientsCache;
        this.campPermsCache = campPermsCache;
        this.usersCache = usersCache;
        this.rbacClientsRelationsStorage = rbacClientsRelationsStorage;
        this.objectMapper = new ObjectMapper();
    }

    /**
     * По ClientId получить логин представителя-шефа (для любой роли)
     */
    @Nonnull
    public String getChiefLoginClientId(long clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(ClientId.fromLong(clientId));
        return dslContextProvider.ppc(shard)
                .select(USERS.LOGIN)
                .from(CLIENTS)
                .join(USERS).on(USERS.UID.eq(CLIENTS.CHIEF_UID))
                .where(CLIENTS.CLIENT_ID.eq(clientId))
                .fetchOptional(USERS.LOGIN)
                .orElseThrow(() -> new IllegalArgumentException(
                        "Incorrect clientid: " + clientId + ", no chief in database"));
    }

    /**
     * Получить роли для списка ClientID,
     * если такого клиента нет в Директе - роль EMPTY
     */
    @Nonnull
    public Map<ClientId, RbacRole> getClientsRoles(Collection<ClientId> clientIds) {
        return EntryStream.of(getClientsPerminfo(clientIds))
                .mapValues(o -> o.map(ClientPerminfo::role).orElse(EMPTY))
                .toMap();
    }

    /**
     * Получить роли для списка uids,
     * если такого клиента нет в Директе - роль EMPTY
     */
    @Nonnull
    public Map<Long, RbacRole> getUidsRoles(Collection<Long> uids) {
        return EntryStream.of(getUsersPerminfo(uids))
                .mapValues(o -> o.map(ClientPerminfo::role).orElse(EMPTY))
                .toMap();
    }

    /**
     * Получить базовую информацию о правах по ClientID
     */
    @Nonnull
    public Optional<ClientPerminfo> getClientPerminfo(@Nonnull ClientId clientId) {
        return getClientsPerminfo(singleton(clientId)).getOrDefault(clientId, Optional.empty());
    }

    /**
     * По списку клиентов получить ClientPerminfo
     */
    @Nonnull
    public Map<ClientId, Optional<ClientPerminfo>> getClientsPerminfo(Collection<ClientId> clientIds) {
        List<ClientId> uniqueClientIds = uniqueList(clientIds);
        Map<ClientId, ClientPerminfo> cacheResult =
                clientsCache.computeIfAbsent(uniqueClientIds, this::getClientsPerminfoInternal);
        return uniqueClientIds.stream().collect(toMap(id -> id, id -> Optional.ofNullable(cacheResult.get(id))));
    }

    @Nonnull
    private Map<ClientId, ClientPerminfo> getClientsPerminfoInternal(Collection<ClientId> clientIds) {
        if (clientIds.isEmpty()) {
            return emptyMap();
        }
        Map<ClientId, ClientPerminfo> result = new HashMap<>();
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong)
                .forEach((shard, chunk) -> result.putAll(getClientsPerminfoSharded(shard, chunk)));

        return result;
    }

    @Nonnull
    private Map<ClientId, ClientPerminfo> getClientsPerminfoSharded(int shard, List<ClientId> chunk) {
        return StreamEx.of(
                dslContextProvider.ppc(shard)
                        .select(CLIENTS.CLIENT_ID, CLIENTS.CHIEF_UID, CLIENTS.ROLE, CLIENTS.SUBROLE,
                                CLIENTS.AGENCY_CLIENT_ID, CLIENTS.AGENCY_UID, CLIENTS.PRIMARY_MANAGER_UID,
                                AGENCY_MANAGERS.MANAGER_UID, CLIENT_MANAGERS.MANAGER_UID, CLIENTS.PERMS,
                                FREELANCERS.CLIENT_ID, AGENCY_LIM_REP_CLIENTS.AGENCY_UID, REVERSE_CLIENTS_RELATIONS.CLIENT_ID_TO,
                                IDM_GROUP_ROLES.IDM_GROUP_ID, IDM_GROUPS_MEMBERS.IDM_GROUP_ID)
                        .from(CLIENTS)
                        .leftJoin(AGENCY_LIM_REP_CLIENTS).on(CLIENTS.CLIENT_ID.eq(AGENCY_LIM_REP_CLIENTS.CLIENT_ID))
                        .leftJoin(CLIENT_MANAGERS).on(CLIENTS.CLIENT_ID.eq(CLIENT_MANAGERS.CLIENT_ID))
                        .leftJoin(AGENCY_MANAGERS).on(CLIENTS.CLIENT_ID.eq(AGENCY_MANAGERS.AGENCY_CLIENT_ID))
                        .leftJoin(FREELANCERS)
                        .on(CLIENTS.CLIENT_ID.eq(FREELANCERS.CLIENT_ID)
                                .and(FREELANCERS.IS_DISABLED.eq(0L)))
                        .leftJoin(IDM_GROUP_ROLES)
                        .on(IDM_GROUP_ROLES.SUBJECT_CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                        .leftJoin(IDM_GROUPS_MEMBERS).on(IDM_GROUPS_MEMBERS.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                        .leftJoin(REVERSE_CLIENTS_RELATIONS).on(REVERSE_CLIENTS_RELATIONS.CLIENT_ID_FROM.eq(CLIENTS.CLIENT_ID)
                                .and(REVERSE_CLIENTS_RELATIONS.TYPE.eq(ReverseClientsRelationsType.mcc)))
                        .where(CLIENTS.CLIENT_ID.in(mapList(chunk, ClientId::asLong)))
                        .fetch(r -> new ClientPerminfo(
                                ClientId.fromLong(r.getValue(CLIENTS.CLIENT_ID)),
                                r.getValue(CLIENTS.CHIEF_UID),
                                RbacRole.fromSource(r.getValue(CLIENTS.ROLE)),
                                RbacSubrole.fromSource(r.getValue(CLIENTS.SUBROLE)),
                                ClientId.fromNullableLong(r.getValue(CLIENTS.AGENCY_CLIENT_ID)),
                                StreamEx.of(
                                        r.getValue(CLIENTS.AGENCY_UID),
                                        r.getValue(AGENCY_LIM_REP_CLIENTS.AGENCY_UID)
                                ).nonNull().toSet(),
                                r.getValue(CLIENTS.PRIMARY_MANAGER_UID),
                                Optional.ofNullable(RbacRole.fromSource(r.getValue(CLIENTS.ROLE)) == AGENCY
                                        ? r.getValue(AGENCY_MANAGERS.MANAGER_UID)
                                        : r.getValue(CLIENT_MANAGERS.MANAGER_UID))
                                        .map(Collections::singleton)
                                        .orElse(Collections.emptySet()),
                                ClientPerm.parse(r.getValue(CLIENTS.PERMS)),
                                r.getValue(FREELANCERS.CLIENT_ID) != null,
                                Optional.ofNullable(r.getValue(REVERSE_CLIENTS_RELATIONS.CLIENT_ID_TO))
                                        .map(Collections::singleton)
                                        .orElse(Collections.emptySet()),
                                Optional.ofNullable(r.getValue(IDM_GROUP_ROLES.IDM_GROUP_ID))
                                        .map(Collections::singleton)
                                        .orElse(Collections.emptySet()),
                                Optional.ofNullable(r.getValue(IDM_GROUPS_MEMBERS.IDM_GROUP_ID))
                                        .map(Collections::singleton)
                                        .orElse(Collections.emptySet())
                        )))
                .mapToEntry(ClientPerminfo::clientId, Function.identity())
                .collapseKeys()
                .mapValues(this::mergePermInfos)
                .toMap();
    }

    private ClientPerminfo mergePermInfos(List<ClientPerminfo> perminfos) {
        if (perminfos.size() == 1) {
            return perminfos.get(0);
        }
        ClientPerminfo first = perminfos.get(0);
        return new ClientPerminfo(
                first.clientId(),
                first.chiefUid(),
                first.role(),
                first.subrole(),
                first.agencyClientId(),
                perminfos.stream()
                        .map(ClientPerminfo::agencyUids)
                        .flatMap(Collection::stream).collect(toSet()),
                first.managerUid(),
                perminfos.stream()
                        .map(ClientPerminfo::managerUids)
                        .flatMap(Collection::stream).collect(toSet()),
                first.perms(),
                first.canHaveRelationship(),
                perminfos.stream().map(ClientPerminfo::mccClientIds)
                        .flatMap(Collection::stream).collect(toSet()),
                perminfos.stream().map(ClientPerminfo::managerGroupIds)
                        .flatMap(Collection::stream).collect(toSet()),
                perminfos.stream().map(ClientPerminfo::clientGroupIds)
                        .flatMap(Collection::stream).collect(toSet())
        );
    }

    @Nonnull
    public Optional<UserPerminfo> getUserPerminfo(long uid) {
        return getUsersPerminfo(singleton(uid)).get(uid);
    }

    /**
     * По списку uid получить UserPerminfo - базовую информацию о правах
     */
    @Nonnull
    public Map<Long, Optional<UserPerminfo>> getUsersPerminfo(Collection<Long> uids) {
        List<Long> uniqueUids = uniqueList(uids);
        Map<Long, UserPerminfo> cacheResult = usersCache.computeIfAbsent(uniqueUids, this::getUsersPerminfoInternal);
        return StreamEx.of(uniqueUids).toMap(id -> Optional.ofNullable(cacheResult.get(id)));
    }

    @Nonnull
    private Map<Long, UserPerminfo> getUsersPerminfoInternal(Collection<Long> uids) {
        if (uids.isEmpty()) {
            return emptyMap();
        }
        Map<Long, UserPerminfo> result = new HashMap<>();
        shardHelper.groupByShard(uids, ShardKey.UID)
                .forEach((shard, chunk) -> result.putAll(getUsersPerminfoSharded(shard, chunk)));

        return result;
    }

    @Nonnull
    private Map<Long, UserPerminfo> getUsersPerminfoSharded(int shard, List<Long> chunk) {
        return StreamEx.of(dslContextProvider.ppc(shard)
                .select(USERS.UID, USERS.REP_TYPE,
                        CLIENTS.CLIENT_ID, CLIENTS.CHIEF_UID, CLIENTS.ROLE, CLIENTS.SUBROLE,
                        CLIENTS.AGENCY_CLIENT_ID, CLIENTS.AGENCY_UID, CLIENTS.PRIMARY_MANAGER_UID,
                        AGENCY_MANAGERS.MANAGER_UID, CLIENT_MANAGERS.MANAGER_UID, CLIENTS.PERMS,
                        FREELANCERS.CLIENT_ID, AGENCY_LIM_REP_CLIENTS.AGENCY_UID,
                        IDM_GROUP_ROLES.IDM_GROUP_ID, IDM_GROUPS_MEMBERS.IDM_GROUP_ID,
                        REVERSE_CLIENTS_RELATIONS.CLIENT_ID_TO)
                .from(USERS)
                .join(CLIENTS).on(CLIENTS.CLIENT_ID.eq(USERS.CLIENT_ID))
                .leftJoin(AGENCY_LIM_REP_CLIENTS).on(CLIENTS.CLIENT_ID.eq(AGENCY_LIM_REP_CLIENTS.CLIENT_ID))
                .leftJoin(CLIENT_MANAGERS).on(CLIENTS.CLIENT_ID.eq(CLIENT_MANAGERS.CLIENT_ID))
                .leftJoin(AGENCY_MANAGERS).on(CLIENTS.CLIENT_ID.eq(AGENCY_MANAGERS.AGENCY_CLIENT_ID))
                .leftJoin(FREELANCERS).on(CLIENTS.CLIENT_ID.eq(FREELANCERS.CLIENT_ID)
                        .and(FREELANCERS.IS_DISABLED.eq(0L)))
                .leftJoin(IDM_GROUP_ROLES)
                .on(IDM_GROUP_ROLES.SUBJECT_CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .leftJoin(IDM_GROUPS_MEMBERS).on(IDM_GROUPS_MEMBERS.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                        .leftJoin(REVERSE_CLIENTS_RELATIONS).on(REVERSE_CLIENTS_RELATIONS.CLIENT_ID_FROM.eq(CLIENTS.CLIENT_ID)
                                .and(REVERSE_CLIENTS_RELATIONS.TYPE.eq(ReverseClientsRelationsType.mcc)))
                .where(USERS.UID.in(chunk))
                .orderBy(USERS.UID)
                .fetch(r -> new UserPerminfo(
                        ClientId.fromLong(r.getValue(CLIENTS.CLIENT_ID)),
                        r.getValue(CLIENTS.CHIEF_UID),
                        RbacRole.fromSource(r.getValue(CLIENTS.ROLE)),
                        RbacSubrole.fromSource(r.getValue(CLIENTS.SUBROLE)),
                        ClientId.fromNullableLong(r.getValue(CLIENTS.AGENCY_CLIENT_ID)),
                        StreamEx.of(
                                r.getValue(CLIENTS.AGENCY_UID),
                                r.getValue(AGENCY_LIM_REP_CLIENTS.AGENCY_UID)
                        ).nonNull().toSet(),
                        r.getValue(CLIENTS.PRIMARY_MANAGER_UID),
                        Optional.ofNullable(RbacRole.fromSource(r.getValue(CLIENTS.ROLE)) == AGENCY
                                ? r.getValue(AGENCY_MANAGERS.MANAGER_UID)
                                : r.getValue(CLIENT_MANAGERS.MANAGER_UID))
                                .map(Collections::singleton)
                                .orElse(Collections.emptySet()),
                        RbacRepType.fromSource(r.getValue(USERS.REP_TYPE)),
                        r.getValue(USERS.UID),
                        ClientPerm.parse(r.getValue(CLIENTS.PERMS)),
                        r.getValue(FREELANCERS.CLIENT_ID) != null,
                        Optional.ofNullable(r.getValue(REVERSE_CLIENTS_RELATIONS.CLIENT_ID_TO))
                                .map(Collections::singleton)
                                .orElse(Collections.emptySet()),
                        Optional.ofNullable(r.getValue(IDM_GROUP_ROLES.IDM_GROUP_ID))
                                .map(Collections::singleton)
                                .orElse(Collections.emptySet()),
                        Optional.ofNullable(r.getValue(IDM_GROUPS_MEMBERS.IDM_GROUP_ID))
                                .map(Collections::singleton)
                                .orElse(Collections.emptySet())
                )))
                .mapToEntry(UserPerminfo::userId, Function.identity())
                .collapseKeys()
                .mapValues(this::mergeUserPermInfos)
                .toMap();
    }

    /**
     * Получить список UID всех менеджеров на всех шардах.
     */
    public List<Long> getAllManagers() {
        return shardHelper.dbShards().stream()
                .flatMap(shard -> getAllManagersSharded(shard).stream())
                .collect(toList());
    }

    /**
     * Получить список UID всех менеджеров на данном шарде.
     */
    public List<Long> getAllManagersSharded(int shard) {
        return dslContextProvider.ppc(shard)
                .select(MANAGER_HIERARCHY.MANAGER_UID)
                .from(MANAGER_HIERARCHY)
                .join(CLIENTS)
                .on(MANAGER_HIERARCHY.MANAGER_CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .where(CLIENTS.ROLE.eq(ClientsRole.manager))
                .fetch(MANAGER_HIERARCHY.MANAGER_UID);
    }

    private UserPerminfo mergeUserPermInfos(List<UserPerminfo> perminfos) {
        if (perminfos.size() == 1) {
            return perminfos.get(0);
        }
        UserPerminfo first = perminfos.get(0);
        return new UserPerminfo(
                first.clientId(),
                first.chiefUid(),
                first.role(),
                first.subrole(),
                first.agencyClientId(),
                perminfos.stream()
                        .map(ClientPerminfo::agencyUids)
                        .flatMap(Collection::stream).collect(toSet()),
                first.managerUid(),
                perminfos.stream()
                        .map(ClientPerminfo::managerUids)
                        .flatMap(Collection::stream).collect(toSet()),
                first.repType(),
                first.userId(),
                first.perms(),
                first.canHaveRelationship(),
                perminfos.stream()
                        .map(ClientPerminfo::mccClientIds)
                        .flatMap(Collection::stream).collect(toSet()),
                perminfos.stream()
                        .map(ClientPerminfo::managerGroupIds)
                        .flatMap(Collection::stream).collect(toSet()),
                perminfos.stream()
                        .map(ClientPerminfo::clientGroupIds)
                        .flatMap(Collection::stream).collect(toSet())
        );
    }

    private RbacRole permsRole(Map<Long, ClientPerminfo> perms, long clientId) {
        return Optional.ofNullable(perms.get(clientId)).map(ClientPerminfo::role).orElse(EMPTY);
    }

    /**
     * Аналог RBACDirect::replace_manager_by_teamlead
     * Используется, чтобы все менеджеры одной группы имели одинаковые права (права тимлида)
     * Если uid не менеджерский, или нет тимлида - возвращается он же
     *
     * @param uid
     * @return
     */
    public long replaceManagerByTeamLead(long uid) {
        Optional<UserPerminfo> userPerminfo = getUserPerminfo(uid);
        return userPerminfo
                .filter(pi -> pi.role() == MANAGER && pi.subrole() == null)
                .flatMap(pi -> this.getManagerSupervisor(pi.chiefUid()))
                .orElse(uid);
    }

    @Nonnull
    public Optional<Long> getManagerSupervisor(Long uid) {
        return Optional.ofNullable(getManagersSupervisors(singleton(uid)).get(uid));
    }

    @Nonnull
    public Map<Long, Long> getManagersSupervisors(Collection<Long> uids) {
        Map<Long, Long> result = new HashMap<>();
        shardHelper.groupByShard(uids, ShardKey.UID).forEach((shard, chunk) ->
                dslContextProvider.ppc(shard)
                        .select(MANAGER_HIERARCHY.MANAGER_UID, MANAGER_HIERARCHY.SUPERVISOR_UID)
                        .from(MANAGER_HIERARCHY)
                        .where(MANAGER_HIERARCHY.MANAGER_UID.in(chunk)
                                .and(MANAGER_HIERARCHY.SUPERVISOR_CLIENT_ID.gt(NO_SUPERVISOR)))
                        .forEach(r -> result.put(r.value1(), r.value2())));
        return result;
    }

    /**
     * Возвращает список прямых подчиненных тимлида/супертимлида.
     * - тимлидов для супертимлида
     * - менеджеров для тимлида
     */
    @Nonnull
    public List<Long> getSupervisorDirectSubordinates(Long uid) {
        return getSupervisorsDirectSubordinates(singleton(uid)).getOrDefault(uid, emptyList());
    }

    /**
     * Для каждого тимлида/супертимлида возвращает список его прямых подчиненных.
     * - тимлидов для супертимлида
     * - менеджеров для тимлида
     * <p>
     * Аналог методов rbac_get_team_of_superteamleader и rbac_get_managers_of_teamleader
     */
    @Nonnull
    public Map<Long, List<Long>> getSupervisorsDirectSubordinates(Collection<Long> uids) {
        return StreamEx.of(shardHelper.dbShards().stream().flatMap(shard ->
                dslContextProvider.ppc(shard)
                        .select(MANAGER_HIERARCHY.MANAGER_UID, MANAGER_HIERARCHY.SUPERVISOR_UID)
                        .from(MANAGER_HIERARCHY)
                        .where(MANAGER_HIERARCHY.SUPERVISOR_UID.in(uids))
                        .fetchStream()))
                .groupingBy(Record2::component2, mapping(Record2::component1, toList()));
    }

    /**
     * Получить субклиентов агентства по списку uid представителей агентства
     * Возвращает мапу: uid представителя -> список ClientID его субклиентов
     * Если у представителя нет клиентов, то в результирующей мапе его не будет
     */
    @Nonnull
    public Map<Long, List<Long>> getSubclientsOfAgencyReps(Collection<Long> agencyUids) {
        if (agencyUids.isEmpty()) {
            return emptyMap();
        }
        return StreamEx.of(shardHelper.dbShards().stream().flatMap(shard ->
                dslContextProvider.ppc(shard)
                        .select(CLIENTS.CLIENT_ID, CLIENTS.AGENCY_UID)
                        .from(CLIENTS)
                        .where(CLIENTS.AGENCY_UID.in(agencyUids))
                        .union(
                                dslContextProvider.ppc(shard)
                                        .select(AGENCY_LIM_REP_CLIENTS.CLIENT_ID, AGENCY_LIM_REP_CLIENTS.AGENCY_UID)
                                        .from(AGENCY_LIM_REP_CLIENTS)
                                        .where(AGENCY_LIM_REP_CLIENTS.AGENCY_UID.in(agencyUids))
                        )
                        .fetchStream()))
                .groupingBy(Record2::component2, mapping(Record2::component1, toList()));
    }

    /**
     * Возвращает список UID всех подчиненных тимлида/супертимлида.
     * - тимлидов и менеджеров для супертимлида
     * - менеджеров для тимлида
     */
    @Nonnull
    public List<Long> getSupervisorSubordinates(Long uid) {
        if (uid == null) {
            return emptyList();
        }

        return getSupervisorsSubordinates(singleton(uid)).getOrDefault(uid, emptyList());
    }

    /**
     * Для каждого тимлида/супертимлида возвращает список его подчиненных.
     * - тимлидов и менеджеров для супертимлида
     * - менеджеров для тимлида
     */
    @Nonnull
    public Map<Long, List<Long>> getSupervisorsSubordinates(Collection<Long> uids) {
        Collection<Long> filteredUids = filterList(uids, Objects::nonNull);
        Map<Long, List<Long>> result = new HashMap<>();
        TypeReference<List<Long>> resultType = new TypeReference<List<Long>>() {
        };

        Map<Long, Optional<UserPerminfo>> info = getUsersPerminfo(filteredUids);
        List<Long> clientIds = EntryStream.of(info)
                .peekKeys(uid -> result.put(uid, new ArrayList<Long>()))
                .values()
                .flatMap(StreamEx::of)
                .map(permInfo -> permInfo.clientId().asLong())
                .toList();

        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).forEach((shard, shardClientIds) ->
                dslContextProvider.ppc(shard)
                        .select(MANAGER_HIERARCHY.MANAGER_UID, MANAGER_HIERARCHY.SUBORDINATES_UID)
                        .from(MANAGER_HIERARCHY)
                        .where(MANAGER_HIERARCHY.MANAGER_CLIENT_ID.in(shardClientIds))
                        .and(MANAGER_HIERARCHY.SUBORDINATES_UID.isNotNull())
                        .forEach(r -> {
                            try {
                                List<Long> subordinates = objectMapper.readerFor(resultType).readValue(r.component2());
                                result.get(r.component1()).addAll(subordinates);
                            } catch (IOException e) {
                                logger.warn("Unable to read manager subordinates", e);
                            }
                        }));
        return result;
    }

    /**
     * Получить представителей одного клиента
     *
     * @param clientId клиент, представителей которого нужно получить
     * @return коллекция представителей
     */
    public Collection<Representative> getClientRepresentatives(@Nullable ClientId clientId) {
        return clientId != null ? massGetClientRepresentatives(singleton(clientId)) : emptyList();
    }

    /**
     * Получить представителей клиентов
     *
     * @param clientIds коллекция id клиентов, представителей которых нужно получить
     * @return коллекция представителей
     */
    public Collection<Representative> massGetClientRepresentatives(Collection<ClientId> clientIds) {
        if (clientIds.isEmpty()) {
            return emptyList();
        }
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong)
                .stream().flatMapKeyValue(
                        (shard, chunk) -> dslContextProvider.ppc(shard)
                                .select(USERS.CLIENT_ID, USERS.UID, USERS.REP_TYPE)
                                .from(USERS)
                                .where(USERS.CLIENT_ID.in(mapList(chunk, ClientId::asLong)))
                                .stream()
                                .map(r -> Representative
                                        .create(ClientId.fromLong(r.value1()),
                                                r.value2(),
                                                RbacRepType.fromSource(r.value3()))))
                .toList();
    }

    /**
     * Вернуть множество субклиентов, которые являются супер-субклиентами по версии PPC
     */
    public Set<ClientId> getSubclientsAllowedToCreateCamps(Collection<ClientId> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).stream()
                .flatMapKeyValue((shard, chunk) -> dslContextProvider.ppc(shard)
                        .select(CLIENTS.CLIENT_ID)
                        .from(CLIENTS)
                        .where(CLIENTS.CLIENT_ID.in(chunk)
                                .and(CLIENTS.ALLOW_CREATE_SCAMP_BY_SUBCLIENT.eq(ClientsAllowCreateScampBySubclient.Yes))
                        )
                        .fetch()
                        .map(r -> r.get(CLIENTS.CLIENT_ID))
                        .stream())
                .map(ClientId::fromLong)
                .toSet();
    }

    /**
     * Является ли субклиент супер-субклиентом по версии PPC
     */
    public boolean isSubclientAllowedToCreateCamps(ClientId clientId) {
        return getSubclientsAllowedToCreateCamps(Collections.singletonList(clientId))
                .contains(clientId);
    }

    /**
     * Вычисляет тип доступа {@link RbacAccessType} оператора к объектам пользователя.
     *
     * @param operatorUid uid оператора
     * @param clientUid   uid пользователя
     */
    @Nonnull
    public RbacAccessType getAccessType(long operatorUid, long clientUid) {
        return getAccessTypes(operatorUid, Set.of(clientUid)).get(clientUid);
    }

    /**
     * Вычисляет тип доступа {@link RbacAccessType} оператора к объектам пользователей.
     *
     * @param operatorUid uid оператора
     * @param uids        uid'ы пользователей
     * @return мапа uid пользователя -> тип доступа {@link RbacAccessType} к объектам пользователя
     */
    @Nonnull
    public Map<Long, RbacAccessType> getAccessTypes(long operatorUid, Collection<Long> uids) {
        if (uids.isEmpty()) {
            return emptyMap();
        }
        Set<Long> distinctUids = new HashSet<>(uids);
        if (distinctUids.equals(singleton(operatorUid))) {
            return singletonMap(operatorUid, RbacAccessType.READ_WRITE);
        }

        Map<Long, Optional<UserPerminfo>> allPerminfo =
                getUsersPerminfo(StreamEx.of(distinctUids).append(operatorUid).toSet());
        Optional<UserPerminfo> operatorPerm = allPerminfo.getOrDefault(operatorUid, Optional.empty());

        Map<Long, RbacAccessType> ret = StreamEx.of(distinctUids).toMap(uid -> RbacAccessType.NONE);
        List<UserPerminfo> perms = StreamEx.of(distinctUids)
                .map(uid -> allPerminfo.getOrDefault(uid, Optional.empty()))
                .flatMap(StreamEx::of)
                .collect(toList());

        operatorPerm.ifPresent(p -> checkAccessTypeByRole(p, ret, perms));
        return ret;
    }

    private void checkAccessTypeByRole(@Nonnull UserPerminfo operatorPerm, Map<Long, RbacAccessType> ret,
                                       List<UserPerminfo> perms) {
        switch (operatorPerm.role()) {
            case SUPER:
                // суперы 'владеют' всеми
                perms.forEach(p -> ret.put(p.userId(), RbacAccessType.READ_WRITE));
                break;
            case SUPERREADER:
                // суперы 'владеют' всеми в режиме readonly
                perms.forEach(p -> ret.put(p.userId(), RbacAccessType.READONLY));
                break;
            case SUPPORT:
                // саппорт 'владеет' клиентами, агентствами и менеджерами
                perms.stream().filter(p -> p.hasRole(CLIENT, AGENCY, MANAGER))
                        .forEach(p -> ret.put(p.userId(), RbacAccessType.READ_WRITE));
                break;
            case LIMITED_SUPPORT:
                // ограниченный саппорт имеет readonly доступ к своим клиентам, агентствам,
                // сабклиентам своих агенств, клиентам своих фрилансеров и управляемых аккаунтов управляющего МСС
                filterByLimitedSupportOwnership(operatorPerm, perms)
                        .forEach(p -> ret.put(p.userId(), RbacAccessType.READONLY));
                break;
            case PLACER:
                // вешальщик 'владеет' клиентами, агентствами в режиме readonly
                perms.stream().filter(p -> p.hasRole(CLIENT, AGENCY))
                        .forEach(p -> ret.put(p.userId(), RbacAccessType.READONLY));
                break;
            case MEDIA:
                // медиапланнер 'владеет' клиентами, агентствами
                perms.stream().filter(p -> p.hasRole(CLIENT, AGENCY))
                        .forEach(p -> ret.put(p.userId(), RbacAccessType.READ_WRITE));
                break;
            case MANAGER:
                filterByManagerOwnership(operatorPerm, perms)
                        .forEach(p -> ret.put(p.userId(), RbacAccessType.READ_WRITE));
                break;
            case AGENCY:
                filterByAgencyOwnership(operatorPerm, perms)
                        .forEach(p -> ret.put(p.userId(), RbacAccessType.READ_WRITE));
                break;
            case INTERNAL_AD_ADMIN:
                filterInternalAdProducts(perms)
                        .forEach(p -> ret.put(p.userId(), RbacAccessType.READ_WRITE));
                break;
            case INTERNAL_AD_MANAGER:
                List<UserPerminfo> internalAdProducts = filterInternalAdProducts(perms);

                List<ClientsRelation> relations = rbacClientsRelationsStorage.getRelationsToSpecificClients(
                        Set.of(operatorPerm.clientId()), mapList(internalAdProducts, UserPerminfo::clientId),
                        Set.of(ClientsRelationType.INTERNAL_AD_PUBLISHER, ClientsRelationType.INTERNAL_AD_READER));

                Map<Long, ClientsRelation> relationsByClientId = StreamEx.of(relations)
                        .mapToEntry(ClientsRelation::getClientIdTo, Functions.identity())
                        .toMap();

                internalAdProducts.forEach(p -> {
                    if (!relationsByClientId.containsKey(p.clientId().asLong())) {
                        return;
                    }

                    ClientsRelation relation = relationsByClientId.get(p.clientId().asLong());
                    ret.put(p.userId(),
                            relation.getRelationType() == ClientsRelationType.INTERNAL_AD_PUBLISHER ?
                                    RbacAccessType.READ_WRITE :
                                    RbacAccessType.READONLY);
                });

                break;
            case INTERNAL_AD_SUPERREADER:
                filterInternalAdProducts(perms)
                        .forEach(p -> ret.put(p.userId(), RbacAccessType.READONLY));
                break;
            case CLIENT:
                perms.stream().filter(p -> p.clientId().equals(operatorPerm.clientId()))
                        .forEach(p -> ret.put(p.userId(),
                                operatorPerm.hasPerm(SUPER_SUBCLIENT) || operatorPerm.hasRepType(CHIEF, MAIN)
                                        ? RbacAccessType.READ_WRITE
                                        : RbacAccessType.READONLY));
                if (operatorPerm.canHaveRelationship()) {
                    // freelancer
                    filterByFreelancerOwnership(operatorPerm, perms)
                            .forEach(p -> ret.put(p.userId(), RbacAccessType.READ_WRITE));
                }

                if (operatorPerm.isMcc()) { // клиентский mcc
                    perms.stream().filter(p -> operatorPerm.mccClientIds().contains(p.clientId().asLong()))
                            .forEach(p -> {
                                var isReadOnly = operatorPerm.isReadonlyRep()
                                        || (p.isAgencySubclient() && !p.isAgencySuperSubclient());
                                        ret.put(p.userId(), isReadOnly ? RbacAccessType.READONLY : RbacAccessType.READ_WRITE);
                            });
                }
                break;
            default:
                break;
        }
    }

    @Nonnull
    private List<UserPerminfo> filterByManagerOwnership(@Nonnull UserPerminfo operatorPerm, List<UserPerminfo> perms) {
        // получаем агентства и агентства клиентов
        List<ClientId> agencyIds = perms.stream()
                .map(p -> p.hasRole(CLIENT) ? p.agencyClientId()
                        : null)
                .filter(Objects::nonNull)
                .collect(toList());
        Map<ClientId, Optional<ClientPerminfo>> agencyPerms = getClientsPerminfo(agencyIds);

        List<UserPerminfo> ret = new ArrayList<>();
        boolean groupCheckRequired = !operatorPerm.clientGroupIds().isEmpty();
        Set<Long> managers = getManagerTeammates(operatorPerm);
        for (UserPerminfo perm : perms) {
            if (perm.hasRole(CLIENT)) {
                if (groupCheckRequired &&
                        (hasGroupAccess(operatorPerm, perm) ||
                                hasGroupAccessToAgency(operatorPerm, perm, agencyPerms))) {
                    // есть доступ через групповую роль к клиенту или его агентству
                    ret.add(perm);
                } else if (perm.managerUids().stream().anyMatch(managers::contains)) {
                    // сервисируемый клиент
                    ret.add(perm);
                } else if (perm.agencyClientId() != null) {
                    if (StreamEx.of(agencyPerms.get(perm.agencyClientId()))
                            .map(ClientPerminfo::managerUids)
                            .flatMap(StreamEx::of)
                            .anyMatch(managers::contains)) {
                        ret.add(perm);
                    }
                }
            } else if (perm.hasRole(AGENCY)) {
                if (groupCheckRequired && hasGroupAccess(operatorPerm, perm)) {
                    // доступ к агентсву через групповую роль
                    ret.add(perm);
                } else if (perm.managerUids().stream().anyMatch(managers::contains)) {
                    // менеджер "владеет" своими агентствами
                    ret.add(perm);
                }
            } else if (perm.hasRole(MANAGER)) {
                if (groupCheckRequired && !perm.clientGroupIds().isEmpty() &&
                        !Collections.disjoint(operatorPerm.clientGroupIds(), perm.clientGroupIds())) {
                    // 'владеет' всеми менеджерами из своей группы
                    ret.add(perm);
                } else if (managers.contains(perm.userId())) {
                    // 'владеет' всеми менеджерами из своей команды
                    ret.add(perm);
                }
            }
        }

        return ret;
    }

    /**
     * @return {@code true}, если {@code operatorPerm} состоит в группе, которая имеет доступ к {@code perm}
     */
    private boolean hasGroupAccess(UserPerminfo operatorPerm, ClientPerminfo perm) {
        return !operatorPerm.clientGroupIds().isEmpty() && !perm.managerGroupIds().isEmpty() &&
                !Collections.disjoint(operatorPerm.clientGroupIds(), perm.managerGroupIds());
    }

    /**
     * @return {@code true}, если у {@code perm} есть агентство, и {@code operatorPerm} имеет групповой доступ
     * к этому агентству
     * @see #hasGroupAccess(UserPerminfo, ClientPerminfo)
     */
    private boolean hasGroupAccessToAgency(
            UserPerminfo operatorPerm, ClientPerminfo perm,
            Map<ClientId, Optional<ClientPerminfo>> agencyPerms) {
        if (perm.agencyClientId() == null) {
            return false;
        }
        Optional<ClientPerminfo> agencyPerm = agencyPerms.getOrDefault(perm.agencyClientId(), Optional.empty());
        if (agencyPerm.isEmpty()) {
            return false;
        }
        return hasGroupAccess(operatorPerm, agencyPerm.get());
    }

    private List<UserPerminfo> filterByAgencyOwnership(UserPerminfo operatorPerm, List<UserPerminfo> perms) {
        // представитель агентства может 'владеть' только представителями своего агентства
        List<UserPerminfo> ret = new ArrayList<>();
        for (UserPerminfo perm : perms) {
            if (perm.hasRole(AGENCY) && operatorPerm.hasRepType(LIMITED)) {
                // ограниченный представитель агентства может 'владеть' только собой
                if (perm.userId().equals(operatorPerm.userId())) {
                    ret.add(perm);
                }
            } else if (perm.hasRole(AGENCY)) {
                // представитель агентства может 'владеть' только представителями своего агентства
                if (perm.clientId().equals(operatorPerm.clientId())) {
                    ret.add(perm);
                }
            } else if (perm.hasRole(CLIENT)) {
                if (Objects.equals(perm.agencyClientId(), operatorPerm.clientId())
                        && (
                        operatorPerm.hasRepType(CHIEF, MAIN)
                                || perm.agencyUids().contains(operatorPerm.userId()))
                ) {
                    ret.add(perm);
                }
            }
        }
        return ret;
    }

    private List<UserPerminfo> filterInternalAdProducts(List<UserPerminfo> perms) {
        return perms.stream()
                .filter(perminfo -> perminfo.hasPerm(ClientPerm.INTERNAL_AD_PRODUCT))
                .collect(Collectors.toList());
    }

    private List<UserPerminfo> filterByFreelancerOwnership(UserPerminfo operatorPerm, List<UserPerminfo> perms) {
        // фрилансер владеет своими представителями и своими клиентами
        Set<ClientId> freelancerClients = filterFreelancerClients(
                operatorPerm.clientId(),
                mapList(perms, ClientPerminfo::clientId));
        List<UserPerminfo> ret = new ArrayList<>();
        for (UserPerminfo perm : perms) {
            if (perm.clientId().equals(operatorPerm.clientId())) {
                ret.add(perm);
            } else if (freelancerClients.contains(perm.clientId())) {
                ret.add(perm);
            }
        }
        return ret;
    }

    /**
     * Для каждого из заданных клиентов ищет фрилансера или управляющий аккаунт МСС, которым тот управляется.
     * @param relatedClientIds - список ClientId клиентов, для которых нужно найти связанных фрилансеров и управляющих МСС
     * @return - Map<ClientIdКлиента, ClientIdФрилансераИлиУправляющегоМСС>
     **/
    private Map<ClientId, Set<ClientId>> findFreelancerOrControlMccClientIdsByRelatedClients(Collection<ClientId> relatedClientIds) {
        List<ClientsRelation> relationByClientIdTo = rbacClientsRelationsStorage
                .getRelationsByClientIdsTo(relatedClientIds, Set.of(ClientsRelationType.FREELANCER, ClientsRelationType.MCC));

        return StreamEx.of(relationByClientIdTo)
                .mapToEntry(ClientsRelation::getClientIdTo, ClientsRelation::getClientIdFrom)
                .mapKeys(ClientId::fromLong)
                .mapValues(ClientId::fromLong)
                .collect(Collectors.groupingBy(Map.Entry::getKey,
                        Collectors.mapping(Map.Entry::getValue, Collectors.toSet())));
    }

    private List<UserPerminfo> filterByLimitedSupportOwnership(UserPerminfo operatorPerm, List<UserPerminfo> perms) {
        boolean canReadAllClients = canLimitedSupportReadAllClients();
        if (canReadAllClients) {
            // fallback-режим. в этом случае limited_support будут иметь readonly доступ к материалам всех клиентов
            return StreamEx.of(perms)
                    .filter(perm -> perm.clientId().equals(operatorPerm.clientId()) || perm.hasRole(CLIENT, AGENCY))
                    .toList();
        }

        Set<ClientId> clientIds = StreamEx.of(perms)
                .map(ClientPerminfo::clientId)
                .toSet();
        Set<ClientId> agencyIds = StreamEx.of(perms)
                .map(ClientPerminfo::agencyClientId)
                .nonNull()
                .toSet();

        Map<ClientId, Set<ClientId>> freelancersOrControlMccByRelatedClients =
                findFreelancerOrControlMccClientIdsByRelatedClients(clientIds);

        Set<ClientId> expandedClientIds = StreamEx.of(clientIds)
                .append(freelancersOrControlMccByRelatedClients.values().stream().flatMap(Set::stream).collect(toSet()))
                .append(agencyIds)
                .toSet();

        Set<ClientId> supportedClients = filterSupportClients(
                operatorPerm.clientId(),
                expandedClientIds);

        List<UserPerminfo> ret = new ArrayList<>();
        for (UserPerminfo perm : perms) {
            if (perm.clientId().equals(operatorPerm.clientId())) {
                ret.add(perm);
            } else if (supportedClients.contains(perm.clientId())) {
                ret.add(perm);
            } else if (freelancersOrControlMccByRelatedClients.containsKey(perm.clientId())
                            && supportedClients.stream().anyMatch(
                                    freelancersOrControlMccByRelatedClients.get(perm.clientId())::contains)) {
                //Клиент моего фрилансера или управляющего МСС - мой клиент
                ret.add(perm);
            } else if ((perm.agencyClientId() != null) && supportedClients.contains(perm.agencyClientId())) {
                ret.add(perm);
            }
        }
        return ret;
    }

    @Nonnull
    private Set<Long> getManagerTeammates(UserPerminfo operatorPerm) {
        checkArgument(operatorPerm.hasRole(MANAGER));
        List<Long> team;
        if (operatorPerm.subrole() == null) {
            team = getManagerSupervisor(operatorPerm.userId())
                    .map(supervisorUserId -> new ImmutableList.Builder<Long>()
                            .add(supervisorUserId)
                            .addAll(getSupervisorSubordinates(supervisorUserId))
                            .build()
                    )
                    .orElse(ImmutableList.of());
        } else {
            team = getSupervisorSubordinates(operatorPerm.userId());
        }
        Set<Long> managers = new HashSet<>(team);
        managers.add(operatorPerm.userId());
        return managers;
    }

    private Set<ClientId> filterFreelancerClients(ClientId freelancerId, Collection<ClientId> clients) {
        return filterClientsByRelation(freelancerId, clients, ClientsRelationsType.freelancer);
    }

    private Set<ClientId> filterSupportClients(ClientId supportId, Collection<ClientId> clients) {
        return filterClientsByRelation(supportId, clients, ClientsRelationsType.support_for_client);
    }

    private Set<ClientId> filterClientsByRelation(
            ClientId operatorClientId,
            Collection<ClientId> clients,
            ClientsRelationsType relationType) {
        Set<ClientId> ret = new HashSet<>();
        shardHelper.groupByShard(clients, ShardKey.CLIENT_ID, ClientId::asLong)
                .forEach((shard, chunk) -> {
                    List<Long> chunkClientIds = chunk.stream().map(ClientId::asLong).collect(toList());
                    dslContextProvider.ppc(shard)
                            .select(CLIENTS_RELATIONS.CLIENT_ID_TO)
                            .from(CLIENTS_RELATIONS)
                            .where(CLIENTS_RELATIONS.CLIENT_ID_FROM.eq(operatorClientId.asLong()))
                            .and(CLIENTS_RELATIONS.TYPE.eq(relationType))
                            .and(CLIENTS_RELATIONS.CLIENT_ID_TO.in(chunkClientIds))
                            .forEach(r -> ret.add(ClientId.fromLong(r.value1())));
                });
        return ret;
    }


    public Map<Long, RbacCampPerms> getCampPerms(long operatorUid, Collection<Long> cids) {
        List<Long> uniqueCids = uniqueList(cids);
        List<OperatorCampIds> campIds = mapList(uniqueCids, cid -> new OperatorCampIds(operatorUid, cid));
        Map<OperatorCampIds, RbacCampPerms> map = campPermsCache.computeIfAbsent(campIds,
                keys -> this.getCampPermsInternal(operatorUid, mapList(keys, OperatorCampIds::getCid)));
        return EntryStream.of(map)
                .mapKeys(OperatorCampIds::getCid)
                .toMap();
    }

    private Map<OperatorCampIds, RbacCampPerms> getCampPermsInternal(long operatorUid, Collection<Long> cids) {
        Map<Long, RbacCampPerms> ret = new HashMap<>();
        cids.forEach(cid -> ret.put(cid, RbacCampPerms.EMPTY));

        UserPerminfo operator = getUserPerminfo(operatorUid).orElse(null);
        if (operator == null || operator.hasRole(EMPTY)) {
            return EntryStream.of(ret).mapKeys(k -> new OperatorCampIds(operatorUid, k)).toMap();
        }

        Map<Long, CampData> campData = getCampData(cids);

        if (operator.hasRole(SUPER)) {
            campData.keySet().forEach(cid -> ret.put(cid, RbacCampPerms.ALL));
        } else if (operator.hasRole(SUPERREADER)) {
            campData.keySet().forEach(cid -> ret.put(cid, RbacCampPerms.READONLY));
        } else if (operator.hasRole(SUPPORT, PLACER)) {
            RbacCampPerms perm = RbacCampPerms.builder()
                    .withCanRead(true)
                    .withCanWrite(true)
                    .withCanDrop(false)
                    .withCanTransferMoney(true)
                    .withCanExportInExcel(true)
                    .build();
            campData.keySet().forEach(cid -> ret.put(cid, perm));
        } else if (operator.hasRole(MEDIA)) {
            RbacCampPerms perm = RbacCampPerms.builder()
                    .withCanRead(true).withCanWrite(true)
                    .withCanDrop(false)
                    .withCanTransferMoney(false)
                    .withCanExportInExcel(false)
                    .build();
            campData.keySet().forEach(cid -> ret.put(cid, perm));
        } else if (operator.hasRole(MANAGER)) {
            ret.putAll(getManagerCampPerms(operator, campData.values()));
        } else if (operator.hasRole(AGENCY)) {
            ret.putAll(getAgencyCampPerms(operator, campData.values()));
        } else if (operator.hasRole(RbacRole.INTERNAL_AD_ADMIN)) {
            ret.putAll(getInternalAdAdminCampPerms(campData.values()));
        } else if (operator.hasRole(RbacRole.INTERNAL_AD_MANAGER)) {
            ret.putAll(getInternalAdManagerCampPerms(operator, campData.values()));
        } else if (operator.hasRole(RbacRole.INTERNAL_AD_SUPERREADER)) {
            ret.putAll(getInternalAdSuperreaderCampPerms(campData.values()));
        } else if (operator.hasRole(LIMITED_SUPPORT)) {
            ret.putAll(getLimitedSupportCampPerms(operator, campData.values()));
        } else if (operator.hasRole(CLIENT)) {
            final Map<ClientId, Optional<ClientPerminfo>> clientsPerm;
            if (operator.isMcc()) {
                List<ClientId> clientIds = campData.values().stream()
                        .map(c -> c.clientId)
                        .distinct()
                        .collect(toList());
                clientsPerm = getClientsPerminfo(clientIds);
            } else {
                clientsPerm = emptyMap();
            }

            campData.forEach((cid, camp) -> {
                if (Objects.equals(camp.clientId, operator.clientId())) {
                    RbacCampPerms.Builder perms = RbacCampPerms.builder().withCanRead(true);
                    if (!operator.isReadonlyRep()) {
                        if (camp.agencyId == null) {
                            perms.withCanWrite(true).withCanDrop(true).withCanTransferMoney(true)
                                    .withCanExportInExcel(true);
                        } else {
                            if (operator.hasPerm(SUPER_SUBCLIENT)) {
                                perms.withCanWrite(true).withCanDrop(true);
                            }
                            if (operator.hasPerm(MONEY_TRANSFER)) {
                                perms.withCanTransferMoney(true);
                            }
                            if(operator.hasPerm(XLS_IMPORT)){
                                perms.withCanExportInExcel(true);
                            }
                        }
                    }
                    ret.put(cid, perms.build());
                } else if (operator.mccClientIds().contains(camp.clientId.asLong())) {
                    RbacCampPerms.Builder perms = RbacCampPerms.builder().withCanRead(true);
                    if (!operator.isReadonlyRep()) {
                        if (camp.agencyId == null) {
                            perms.withCanWrite(true).withCanDrop(true).withCanTransferMoney(true)
                                    .withCanExportInExcel(true);
                        } else if (clientsPerm.containsKey(camp.clientId)) {
                            var clientPerm = clientsPerm.get(camp.clientId).orElse(null);
                            if (clientPerm != null) {
                                if (clientPerm.hasPerm(SUPER_SUBCLIENT)) {
                                    perms.withCanWrite(true).withCanDrop(true);
                                }
                                if (clientPerm.hasPerm(XLS_IMPORT)) {
                                    perms.withCanExportInExcel(true);
                                }
                            }
                        }
                    }
                    ret.put(cid, perms.build());
                }
            });
            if (operator.canHaveRelationship()) {
                // freelancer
                List<ClientId> clientIds = campData.values().stream().map(c -> c.clientId).distinct().collect(toList());
                Set<ClientId> freelancerClients = filterFreelancerClients(operator.clientId(), clientIds);
                campData.forEach((cid, camp) -> {
                    if (camp.agencyId == null && freelancerClients.contains(camp.clientId)) {
                        ret.put(cid, RbacCampPerms.ALL);
                    }
                });
            }
        }
        return EntryStream.of(ret).mapKeys(k -> new OperatorCampIds(operatorUid, k)).toMap();
    }

    private Map<Long, RbacCampPerms> getAgencyCampPerms(UserPerminfo operator, Collection<CampData> camps) {
        List<ClientId> agencyClients = camps.stream()
                .filter(c -> Objects.equals(c.agencyId, operator.clientId()))
                .map(c -> c.clientId)
                .distinct()
                .collect(toList());
        Map<ClientId, Optional<ClientPerminfo>> clientsPerminfo = getClientsPerminfo(agencyClients);

        int shard = shardHelper.getShardByClientIdStrictly(ClientId.fromLong(operator.clientId().asLong()));
        boolean userAgencyAccessForPay = !getUserAgencyIsNoPay(shard, operator.userId());

        Map<Long, RbacCampPerms> ret = new HashMap<>();
        for (CampData camp : camps) {
            if (!clientsPerminfo.containsKey(camp.clientId)) {
                continue;
            }
            ClientPerminfo client = clientsPerminfo.get(camp.clientId).orElse(null);
            if (client == null
                    || !Objects.equals(client.agencyClientId(), operator.clientId())
                    || !Objects.equals(camp.agencyId, operator.clientId())
            ) {
                continue;
            }
            if (operator.hasRepType(CHIEF, MAIN) || client.agencyUids().contains(operator.userId())) {
                ret.put(camp.cid, userAgencyAccessForPay ? RbacCampPerms.ALL : RbacCampPerms.WITHOUT_TRANSFER_MONEY);
            }
        }
        return ret;
    }

    private Map<Long, RbacCampPerms> getInternalAdAdminCampPerms(Collection<CampData> camps) {
        Set<Long> campaignIdsOfCampaignsInInternalAdProducts = getCampaignIdsOfCampaignsInInternalAdProducts(camps);
        return StreamEx.of(camps)
                .mapToEntry(c -> c.cid,
                        c -> campaignIdsOfCampaignsInInternalAdProducts.contains(c.cid) ?
                                RbacCampPerms.ALL : RbacCampPerms.EMPTY)
                .toMap();
    }

    // TODO переделать на InternalAdsOperatorProductAccessService.getAccessToMultipleProducts
    // после https://st.yandex-team.ru/DIRECT-112375
    // getRelationPlaces, getCampaignPlaces после этого будут не нужны
    private Map<Long, RbacCampPerms> getInternalAdManagerCampPerms(UserPerminfo operator, Collection<CampData> camps) {
        Collection<ClientId> campaignClients = listToSet(camps, c -> c.clientId);
        List<ClientsRelation> relations = rbacClientsRelationsStorage.getRelationsToSpecificClients(
                Set.of(operator.clientId()), campaignClients,
                Set.of(ClientsRelationType.INTERNAL_AD_PUBLISHER, ClientsRelationType.INTERNAL_AD_READER));

        Map<ClientId, ClientsRelation> relationsToCampaignClients = StreamEx.of(relations)
                .mapToEntry(relation -> ClientId.fromLong(relation.getClientIdTo()), Function.identity())
                .toImmutableMap();

        Map<ClientId, Optional<ClientPerminfo>> clientsPerminfo = getClientsPerminfo(campaignClients);
        Map<Long, Long> campaignPlaces = getCampaignPlaces(mapList(camps, c -> c.cid));
        Map<Long, Set<Long>> productPlaces = getRelationPlaces(operator.clientId(),
                filterList(relationsToCampaignClients.values(),
                        r -> r.getRelationType() == ClientsRelationType.INTERNAL_AD_PUBLISHER));

        return StreamEx.of(camps)
                .mapToEntry(c -> c.cid, campData -> {
                    ClientId clientId = campData.clientId;
                    if (!clientsPerminfo.containsKey(clientId)) {
                        return RbacCampPerms.EMPTY;
                    }

                    Boolean clientHasPerm = clientsPerminfo.get(clientId)
                            .map(perminfo -> perminfo.hasPerm(ClientPerm.INTERNAL_AD_PRODUCT))
                            .orElse(false);

                    if (!clientHasPerm) {
                        return RbacCampPerms.EMPTY;
                    }

                    if (!relationsToCampaignClients.containsKey(clientId)) {
                        return RbacCampPerms.EMPTY;
                    }

                    if (!campaignPlaces.containsKey(campData.cid)) {
                        return RbacCampPerms.EMPTY;
                    }

                    ClientsRelation productRelation = relationsToCampaignClients.get(campData.clientId);
                    ClientsRelationType productRelationType = productRelation.getRelationType();

                    if (productRelationType == ClientsRelationType.INTERNAL_AD_PUBLISHER) {
                        Long campaignPlaceId = campaignPlaces.get(campData.cid);

                        if (!productPlaces.containsKey(productRelation.getRelationId())) {
                            return RbacCampPerms.ALL;
                        }

                        Set<Long> operatorAccessiblePlaceIds = productPlaces.get(productRelation.getRelationId());
                        if (!operatorAccessiblePlaceIds.isEmpty() &&
                                !operatorAccessiblePlaceIds.contains(campaignPlaceId)) {
                            return RbacCampPerms.READONLY;
                        }

                        return RbacCampPerms.ALL;
                    }

                    // productRelationType == ClientsRelationType.INTERNAL_AD_READER
                    return RbacCampPerms.READONLY;
                })
                .toMap();
    }

    private Map<Long, RbacCampPerms> getInternalAdSuperreaderCampPerms(Collection<CampData> camps) {
        Set<Long> campaignIdsOfCampaignsInInternalAdProducts = getCampaignIdsOfCampaignsInInternalAdProducts(camps);
        return StreamEx.of(camps)
                .mapToEntry(c -> c.cid,
                        c -> campaignIdsOfCampaignsInInternalAdProducts.contains(c.cid) ?
                                RbacCampPerms.READONLY : RbacCampPerms.EMPTY)
                .toMap();
    }

    @Nonnull
    private Set<Long> getCampaignIdsOfCampaignsInInternalAdProducts(Collection<CampData> camps) {
        Collection<ClientId> clientIds = camps.stream().map(c -> c.clientId).collect(Collectors.toSet());
        Map<ClientId, Optional<ClientPerminfo>> clientsPerminfo = getClientsPerminfo(clientIds);
        return StreamEx.of(camps)
                .filter(campData -> {
                    ClientId clientId = campData.clientId;
                    if (!clientsPerminfo.containsKey(clientId)) {
                        return false;
                    }

                    return clientsPerminfo.get(clientId)
                            .map(perminfo -> perminfo.hasPerm(ClientPerm.INTERNAL_AD_PRODUCT))
                            .orElse(false);
                })
                .map(campData -> campData.cid)
                .toSet();
    }

    @Nonnull
    private Map<Long, Set<Long>> getRelationPlaces(ClientId operatorClientId, List<ClientsRelation> relations) {
        return shardHelper.groupByShard(relations, ShardKey.CLIENT_ID, ClientsRelation::getClientIdTo).stream()
                .flatMapKeyValue((shard, relationsOnShard) -> dslContextProvider.ppc(shard)
                        .select(INTERNAL_AD_MANAGER_PLACE_ACCESS.RELATION_ID,
                                INTERNAL_AD_MANAGER_PLACE_ACCESS.PLACE_ID)
                        .from(INTERNAL_AD_MANAGER_PLACE_ACCESS)
                        .where(INTERNAL_AD_MANAGER_PLACE_ACCESS.RELATION_ID.in(
                                mapList(relationsOnShard, ClientsRelation::getRelationId)))
                        .fetchInto(INTERNAL_AD_MANAGER_PLACE_ACCESS)
                        .intoGroups(INTERNAL_AD_MANAGER_PLACE_ACCESS.RELATION_ID,
                                INTERNAL_AD_MANAGER_PLACE_ACCESS.PLACE_ID)
                        .entrySet().stream())
                .toMap(Map.Entry::getKey, entry -> listToSet(entry.getValue(), Function.identity()));
    }

    @Nonnull
    private Map<Long, Long> getCampaignPlaces(List<Long> cids) {
        return shardHelper.groupByShard(cids, ShardKey.CID).stream()
                .flatMapKeyValue((shard, cidsOnShard) -> dslContextProvider.ppc(shard)
                        .select(CAMPAIGNS_INTERNAL.CID, CAMPAIGNS_INTERNAL.PLACE_ID)
                        .from(CAMPAIGNS_INTERNAL)
                        .where(CAMPAIGNS_INTERNAL.CID.in(cids))
                        .fetchMap(CAMPAIGNS_INTERNAL.CID, CAMPAIGNS_INTERNAL.PLACE_ID)
                        .entrySet().stream())
                .toMap(Map.Entry::getKey, Map.Entry::getValue);
    }

    private Map<Long, RbacCampPerms> getManagerCampPerms(UserPerminfo operator, Collection<CampData> camps) {
        Set<Long> managerTeammates = getManagerTeammates(operator);
        List<ClientId> agencies = camps.stream()
                .map(d -> d.agencyId).filter(Objects::nonNull).collect(toList());
        Map<ClientId, ClientPerminfo> agencyPerminfo =
                EntryStream.of(getClientsPerminfo(agencies))
                        .filterValues(Optional::isPresent)
                        .mapValues(Optional::get)
                        .toMap();
        List<Long> agencyChiefs = agencyPerminfo.values().stream().map(ClientPerminfo::chiefUid).collect(toList());
        Map<Long, Boolean> agencyControl = EntryStream.of(getAccessTypes(operator.userId(), agencyChiefs))
                .mapValues(accessType -> accessType != RbacAccessType.NONE)
                .toMap();

        boolean groupCheckRequired = !operator.clientGroupIds().isEmpty();
        Map<ClientId, Optional<ClientPerminfo>> clientsPerms;
        if (groupCheckRequired) {
            clientsPerms = getClientsPerminfo(mapList(camps, c -> c.clientId));
        } else {
            clientsPerms = emptyMap();
        }

        Map<Long, RbacCampPerms> ret = new HashMap<>();
        for (CampData camp : camps) {
            Optional<ClientPerminfo> clientPerm = clientsPerms.getOrDefault(camp.clientId, Optional.empty());
            if (clientPerm.isPresent() && hasGroupAccess(operator, clientPerm.get())) {
                // Менеджер имеет доступ через групповую роль (aka Тирная схема)
                ret.put(camp.cid, RbacCampPerms.ALL);
            } else if (camp.agencyId != null) {
                boolean isOwner = Optional.ofNullable(agencyPerminfo.get(camp.agencyId))
                        .map(ClientPerminfo::chiefUid)
                        .map(agencyControl::get)
                        .orElse(false);
                if (isOwner) {
                    ret.put(camp.cid, RbacCampPerms.ALL);
                }
            } else if (camp.managerUid != null) {
                if (managerTeammates.contains(camp.managerUid)) {
                    ret.put(camp.cid, RbacCampPerms.ALL);
                }
            }
        }
        return ret;
    }

    private Map<Long, RbacCampPerms> getLimitedSupportCampPerms(UserPerminfo operator, Collection<CampData> camps) {
        boolean canReadAllClients = canLimitedSupportReadAllClients();
        if (canReadAllClients) {
            Map<Long, RbacCampPerms> ret = new HashMap<>();
            camps.forEach(camp -> ret.put(camp.cid, RbacCampPerms.READONLY));
            return ret;
        }

        Set<ClientId> clientIds = listToSet(camps, c -> c.clientId);
        Set<ClientId> agencyIds = StreamEx.of(camps)
                .map(c -> c.agencyId)
                .nonNull()
                .toSet();

        Set<ClientId> supportedClients = filterSupportClients(operator.clientId(),
                StreamEx.of(clientIds).append(agencyIds).toSet());

        Map<Long, RbacCampPerms> ret = new HashMap<>();
        for (CampData camp : camps) {
            if (supportedClients.contains(camp.clientId)) {
                ret.put(camp.cid, RbacCampPerms.READONLY);

            } else if ((camp.agencyId != null) && supportedClients.contains(camp.agencyId)) {
                ret.put(camp.cid, RbacCampPerms.READONLY);
            }
        }
        return ret;
    }

    private Map<Long, CampData> getCampData(Collection<Long> cids) {
        Map<Long, CampData> ret = new HashMap<>();
        shardHelper.groupByShard(cids, ShardKey.CID)
                .forEach((shard, chunk) -> {
                    Campaigns c = CAMPAIGNS.as("c");
                    dslContextProvider.ppc(shard)
                            .select(c.CID, c.CLIENT_ID, c.AGENCY_ID, c.MANAGER_UID)
                            .from(c)
                            .where(c.CID.in(chunk))
                            .and(c.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                            .forEach(rec -> ret.put(rec.value1(),
                                    new CampData(
                                            rec.value1(),
                                            ClientId.fromLong(rec.value2()),
                                            isValidId(rec.value3()) ? ClientId.fromLong(rec.value3()) : null,
                                            isValidId(rec.value4()) ? rec.value4() : null)
                            ));
                });
        return ret;
    }

    public boolean canLimitedSupportReadAllClients() {
        // если вызывать через ppcPropertiesSupport, то циклическая зависимость
        String propertyName = "enable_limited_support_read_all_clients";
        String propertyValue = dslContextProvider.ppcdict()
                .select(PPC_PROPERTIES.VALUE)
                .from(PPC_PROPERTIES)
                .where(PPC_PROPERTIES.NAME.eq(propertyName))
                .fetchOne(PPC_PROPERTIES.VALUE);
        return "1".equals(propertyValue);
    }

    /**
     * Получить доступных представителю агентства представителей субклиентов среди uid из переданной коллекции
     *
     * @param agencyRepUid  uid представителя агентства
     * @param subclientUids коллекция uid проверяемых представителей клиентов
     * @return список uid представителей субклиентов, доступных представителю агентства
     */
    public List<Long> getAccessibleAgencySubclients(Long agencyRepUid, Collection<Long> subclientUids) {
        UserPerminfo agency = getUserPerminfo(agencyRepUid).orElse(null);
        if (agency == null || agency.role() != AGENCY) {
            return Collections.emptyList();
        }

        Map<Long, Optional<UserPerminfo>> usersPerminfo = getUsersPerminfo(subclientUids);
        return StreamEx.of(usersPerminfo.values())
                .flatMap(StreamEx::of)
                .filter(r -> userBelongsToAgency(r, agency))
                .map(UserPerminfo::userId)
                .toList();
    }

    /**
     * Фильтрует переданные идентификаторы представителей.
     * Остаются только те, с клиентами которых у заданного фрилансера есть есть отношения предоставляющие доступ к их
     * материалам.
     */
    List<Long> getAccessibleRelatedClients(Long freelancerRepUid, Collection<Long> clientUids) {
        UserPerminfo freelancer = getUserPerminfo(freelancerRepUid).orElse(null);
        if (freelancer == null || !freelancer.canHaveRelationship()) {
            return Collections.emptyList();
        }

        Map<Long, Optional<UserPerminfo>> usersPerminfo = getUsersPerminfo(clientUids);
        List<UserPerminfo> perms = usersPerminfo.values().stream()
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(toList());
        List<UserPerminfo> userPerminfos = filterByFreelancerOwnership(freelancer, perms);

        return mapList(userPerminfos, UserPerminfo::userId);
    }

    /**
     * Возвращает uid'ы всех клиентов с которыми у фрилансера есть отношения предоставляющие доступ к их материалам.
     */
    List<Long> getRelatedClientsChiefs(ClientId freelancerId) {
        return shardHelper.dbShards().stream().map(shard ->
                dslContextProvider.ppc(shard)
                        .select(USERS.UID)
                        .from(CLIENTS_RELATIONS)
                        .join(CLIENTS).on(CLIENTS.CLIENT_ID.eq(CLIENTS_RELATIONS.CLIENT_ID_TO))
                        .join(USERS).on(USERS.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                        .where(CLIENTS_RELATIONS.CLIENT_ID_FROM.eq(freelancerId.asLong()))
                        .fetch(USERS.UID))
                .flatMap(Collection::stream).collect(Collectors.toList());
    }

    /**
     * Возвращаются client_id агентств, привязанных к менеджеру.
     */
    public List<Long> getAgencyIdsByManagerUid(Long managerUid) {
        return shardHelper.dbShards().stream().map(shard ->
                dslContextProvider.ppc(shard)
                        .select(CLIENTS.CLIENT_ID)
                        .from(AGENCY_MANAGERS)
                        .join(CLIENTS).on(CLIENTS.CLIENT_ID.eq(AGENCY_MANAGERS.AGENCY_CLIENT_ID))
                        .where(AGENCY_MANAGERS.MANAGER_UID.eq(managerUid))
                        .fetch(CLIENTS.CLIENT_ID))
                .flatMap(Collection::stream).collect(Collectors.toList());
    }

    /**
     * Возвращает флаг запрета оплаты для пользователей представляющих агентства
     */
    private boolean getUserAgencyIsNoPay(int shard, Long uid) {
        Boolean userAgencyIsNoPay = dslContextProvider.ppc(shard)
                .select(USERS_AGENCY.UID, USERS_AGENCY.IS_NO_PAY)
                .from(USERS_AGENCY)
                .where(USERS_AGENCY.UID.eq(uid))
                .fetchOne(record -> record.value2() > 0);
        return userAgencyIsNoPay != null && userAgencyIsNoPay;
    }

    /**
     * Возвращаются cid-ы кампаний, привязанных к менеджеру.
     */
    public List<Long> getCampaignIdsByManagerUid(Long managerUid) {
        return shardHelper.dbShards().stream().map(shard ->
                dslContextProvider.ppc(shard)
                        .select(CAMPAIGNS.CID)
                        .from(CAMPAIGNS)
                        .where(CAMPAIGNS.MANAGER_UID.eq(managerUid))
                        .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                        .fetch(CAMPAIGNS.CID))
                .flatMap(Collection::stream).collect(Collectors.toList());
    }

    /**
     * Возвращаются client_id прямых клиентов (не агентств), где менеджер указан главным.
     * Такое бывает только в случаях задания "главного менеджера" через роль в Idm
     */
    public List<Long> getClientIdsByPrimaryManagerUid(Long primaryManagerUid) {
        return shardHelper.dbShards().stream().map(shard ->
                dslContextProvider.ppc(shard)
                        .select(CLIENTS.CLIENT_ID)
                        .from(CLIENTS)
                        .where(CLIENTS.PRIMARY_MANAGER_UID.eq(primaryManagerUid))
                        .and(CLIENTS.PRIMARY_MANAGER_SET_BY_IDM.eq(TRUE))
                        .fetch(CLIENTS.CLIENT_ID))
                .flatMap(Collection::stream).collect(Collectors.toList());
    }

    /**
     * Проверяет, принадлежит ли субклиент агенству
     **/

    private boolean userBelongsToAgency(UserPerminfo user, UserPerminfo agency) {
        if (agency.hasRepType(CHIEF, MAIN)) {
            return Objects.equals(user.agencyClientId(), agency.clientId());
        } else {
            return user.agencyUids().contains(agency.userId());
        }
    }

    /**
     * Получить всех доступных представителю агентства субклиентов агентства.
     *
     * @param agencyRepUid uid представителя агентства
     * @return список uid всех субклиентов агентства (включая представителей субклиентов)
     */

    public List<Long> getAgencySubclients(Long agencyRepUid) {
        UserPerminfo agency = getUserPerminfo(agencyRepUid).orElse(null);
        if (agency == null || agency.role() != AGENCY) {
            return Collections.emptyList();
        }

        Condition condition = getConditionForAgencyFilter(agency);

        return shardHelper.dbShards().stream().map(shard -> {
            var query = dslContextProvider.ppc(shard)
                    .select(USERS.UID)
                    .from(CLIENTS)
                    .join(USERS)
                    .on(CLIENTS.CLIENT_ID.eq(USERS.CLIENT_ID))
                    .where(condition);
            if (agency.repType() == LIMITED) {
                query.union(
                    dslContextProvider.ppc(shard)
                            .select(USERS.UID)
                            .from(AGENCY_LIM_REP_CLIENTS)
                            .join(USERS)
                            .on(USERS.CLIENT_ID.eq(AGENCY_LIM_REP_CLIENTS.CLIENT_ID))
                            .where(AGENCY_LIM_REP_CLIENTS.AGENCY_UID.in(agencyRepUid))
                );
            }
            return query.fetch(USERS.UID);
        }).flatMap(Collection::stream).collect(Collectors.toList());
    }

    /**
     * Возвращает условие поиска по агенству в зависимости от RbacRepType
     *
     * @param agency Агенство
     * @return
     */

    private Condition getConditionForAgencyFilter(UserPerminfo agency) {
        Condition condition = DSL.trueCondition();
        if (agency.hasRepType(CHIEF, MAIN)) {
            condition = condition.and(CLIENTS.AGENCY_CLIENT_ID.eq(agency.clientId().asLong()));
        } else {
            condition = condition.and(CLIENTS.AGENCY_UID.eq(agency.userId()));
        }
        return condition;
    }

    /**
     * Очистка кешей, стоит вызыветь при изменениях ролей/...
     */
    public void clearCaches() {
        clientsCache.clear();
        usersCache.clear();
        campPermsCache.clear();
    }

    private static class CampData {
        long cid;
        ClientId clientId;
        ClientId agencyId;
        Long managerUid;

        CampData(long cid, ClientId clientId, ClientId agencyId, Long managerUid) {
            this.cid = cid;
            this.clientId = clientId;
            this.agencyId = agencyId;
            this.managerUid = managerUid;
        }

        boolean hasManager() {
            return managerUid != null;
        }

        boolean hasAgency() {
            return agencyId != null;
        }
    }

    @SuppressWarnings("checkstyle:visibilitymodifier")
    public static class OperatorCampIds {
        public Long operatorUid;
        public Long cid;

        public OperatorCampIds(Long operatorUid, Long cid) {
            this.operatorUid = operatorUid;
            this.cid = cid;
        }

        public Long getCid() {
            return cid;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            OperatorCampIds that = (OperatorCampIds) o;
            return com.google.common.base.Objects.equal(operatorUid, that.operatorUid) &&
                    com.google.common.base.Objects.equal(cid, that.cid);
        }

        @Override
        public int hashCode() {
            return com.google.common.base.Objects.hashCode(operatorUid, cid);
        }
    }
}
