package ru.yandex.direct.rbac;

import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.rbac.model.ClientsRelation;
import ru.yandex.direct.rbac.model.ClientsRelationType;
import ru.yandex.direct.rbac.model.SupportClientRelation;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.rbac.RbacClientsRelationsStorage.EMPTY_CLIENTS_RELATION;
import static ru.yandex.direct.rbac.model.ClientsRelationType.MOBILE_GOALS_ACCESS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Работа с отношениями клиент-клиент
 */
@Service
public class RbacClientsRelations {
    public static final String RELATIONS_CACHE_BEAN_NAME = "rbacRelationsCache";

    private final ShardHelper shardHelper;
    private final PpcRbac ppcRbac;
    private final RbacClientsRelationsStorage rbacClientsRelationsStorage;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public RbacClientsRelations(
            DslContextProvider dslContextProvider,
            ShardHelper shardHelper,
            PpcRbac ppcRbac,
            RbacClientsRelationsStorage rbacClientsRelationsStorage) {
        this.shardHelper = shardHelper;
        this.ppcRbac = ppcRbac;
        this.rbacClientsRelationsStorage = rbacClientsRelationsStorage;
        this.dslContextProvider = dslContextProvider;
    }


    /**
     * Используется для того, чтобы оператор (фрилансер, MCC) имел те же права на кампании, что и сопровождаемый клиент.
     * Если operatorUid принадлежит фрилансеру или MCC и он сопровождает клиента с заданным clientUid -
     * возвращает clientUId, в противном случае - исходный operatorUid
     */
    long replaceOperatorUidByClientUid(long operatorUid, long clientUid) {
        Optional<UserPerminfo> operator = ppcRbac.getUserPerminfo(operatorUid);
        Optional<UserPerminfo> client = ppcRbac.getUserPerminfo(clientUid);

        return operator.isPresent() && client.isPresent()
                && isRelatedClient(operator.get().clientId(), client.get().clientId())
                ? clientUid : operatorUid;
    }

    /**
     * Если оператор -- клиент с возможностью иметь relation'ы, вернётся chiefUid клиента-владельца переданных кампаний.
     * Иначе возвращается {@code null}
     */
    @Nullable
    Long defineUidOfRelatedClientByCids(long operatorUid, Collection<Long> cids) {
        Optional<UserPerminfo> operator = ppcRbac.getUserPerminfo(operatorUid);
        Long clientUid = null;

        if (operator.isPresent() && operator.get().canHaveRelationship()) {
            Set<Long> clientIds = new HashSet<>(shardHelper.getClientIdsByCampaignIds(cids).values());
            if (clientIds.size() > 1) {
                throw new IllegalStateException("Multiple campaign owners: " + cids);
            }

            ClientId freelancerClientId = operator.get().clientId();
            if (!clientIds.isEmpty() && !clientIds.contains(freelancerClientId.asLong())) {
                Optional<ClientPerminfo> client =
                        ppcRbac.getClientPerminfo(ClientId.fromLong(clientIds.iterator().next()));
                if (client.isPresent() && isRelatedClient(freelancerClientId, client.get().clientId())) {
                    clientUid = client.get().chiefUid();
                }
            }
        }

        return clientUid;
    }

    /**
     * Возвращает признак того, что клиент с clientUid сопровождается фрилансером или управляется MCC
     */
    boolean isRelatedClient(ClientId operatorClientId, ClientId clientId) {
        ClientsRelation clientsRelation = getClientRelation(clientId, operatorClientId);

        if (EMPTY_CLIENTS_RELATION.equals(clientsRelation)) {
            return false;
        }

        return Objects.equals(operatorClientId.asLong(), clientsRelation.getClientIdFrom());
    }

    /**
     * Возвращает {@link ClientsRelation} описывающий отношение фрилансера и клиента с {@param clientId}.
     * Если такого отношения нет, возвращается {@link RbacClientsRelationsStorage#EMPTY_CLIENTS_RELATION}
     */
    @Nonnull
    public ClientsRelation getFreelancerRelation(ClientId clientId, ClientId freelancerClientId) {
        return rbacClientsRelationsStorage.getRelationCached(freelancerClientId, clientId,
                Set.of(ClientsRelationType.FREELANCER));
    }

    /**
     * Возвращает набор {@link ClientsRelation} управляющих МСС-аккаунтов для заданного клиента с {@param clientId}.
     * Если связей нет, возвращается пустое множество
     */
    @Nonnull
    public Set<ClientsRelation> getControlMccRelations(ClientId managedClientId) {
        return Set.copyOf(rbacClientsRelationsStorage.getRelationsByClientIdsTo(Set.of(managedClientId),
                Set.of(ClientsRelationType.MCC)));
    }

    /**
     * Возвращает набор {@link ClientId} управляемых клиентов для заданного управляющего МСС-аккаунта с {@param controlClientId}.
     * Если управляемых клиентов нет, возвращается пустое множество
     */
    @Nonnull
    public Set<ClientId> getMangedMccClientIds(ClientId controlClientId) {
        var shard = shardHelper.getShardByClientId(controlClientId);
        var managedClientIds = rbacClientsRelationsStorage.getRelatedClientIdsToByReverseClientsRelations(
                shard, Set.of(controlClientId), ClientsRelationType.MCC).get(controlClientId);
        return managedClientIds != null ? Set.copyOf(managedClientIds) : Set.of();
    }

    /**
     * Возвращает {@link ClientsRelation}, описывающий отношение фрилансера и клиента с {@param clientId}.
     * Если такого отношения нет, возвращается {@link RbacClientsRelationsStorage#EMPTY_CLIENTS_RELATION}
     */
    @Nonnull
    public ClientsRelation getClientRelation(ClientId clientId, ClientId operatorClientId) {
        return rbacClientsRelationsStorage.getRelationCached(operatorClientId, clientId,
                Set.of(ClientsRelationType.FREELANCER, ClientsRelationType.MCC));
    }

    /**
     * Добавить связь клиента и фрилансера. Связь кладём в шард клиента
     *
     * @param clientId     id клиента, к которому будет доступ у фрилансера
     * @param freelancerId id фрилансера
     */
    public void addFreelancerRelation(ClientId clientId, ClientId freelancerId) {
        int shard = shardHelper.getShardByClientId(clientId);
        rbacClientsRelationsStorage.addRelation(shard, freelancerId, clientId, ClientsRelationType.FREELANCER);
    }

    /**
     * Удалить связь клиента и фрилансера. Связь ищем в шарде клиента
     *
     * @param clientId     id клиента, который предоставил доступ фрилансеру
     * @param freelancerId id фрилансера
     */
    public void removeFreelancerRelation(ClientId clientId, ClientId freelancerId) {
        int shard = shardHelper.getShardByClientId(clientId);
        rbacClientsRelationsStorage.removeRelation(shard, freelancerId, clientId, ClientsRelationType.FREELANCER);
    }

    /**
     * Связать клиента с МСС. Связь кладём в шарды обоих клиентов (в разные таблицы)
     *
     * @param controlMccClientId id управляющего аккаунта МСС
     * @param managedMccClientId id клиента, к которому будет доступ у управляющего аккаунта
     */
    @ParametersAreNonnullByDefault
    public void addClientMccRelation(ClientId controlMccClientId, ClientId managedMccClientId) {
        addGenericRelation(controlMccClientId, managedMccClientId, ClientsRelationType.MCC);
    }

    /**
     * Удалить связь клиента с МСС
     *
     * @param controlMccClientId id управляющего аккаунта МСС
     * @param managedMccClientId id клиента, для которого удаляем доступ управляющего аккаунта
     */
    public boolean removeClientMccRelation(ClientId controlMccClientId, ClientId managedMccClientId) {
        return removeGenericRelation(controlMccClientId, managedMccClientId, ClientsRelationType.MCC);
    }

    public void addSupportRelation(ClientId subjectClientId, ClientId operatorClientId) {
        ClientsRelationType relationType = ClientsRelationType.SUPPORT_FOR_CLIENT;
        int shard = shardHelper.getShardByClientIdStrictly(subjectClientId);

        Long relationId =
                rbacClientsRelationsStorage.getRelationId(shard, operatorClientId, subjectClientId, relationType);

        if (relationId == null) {
            rbacClientsRelationsStorage.addRelation(shard, operatorClientId, subjectClientId, relationType);
        }
    }

    public void removeSupportRelation(ClientId subjectClientId, ClientId operatorClientId) {
        ClientsRelationType relationType = ClientsRelationType.SUPPORT_FOR_CLIENT;
        int shard = shardHelper.getShardByClientIdStrictly(subjectClientId);
        rbacClientsRelationsStorage.removeRelation(shard, operatorClientId, subjectClientId, relationType);
    }

    public List<SupportClientRelation> getAllSupportRelations() {
        return rbacClientsRelationsStorage.getAllSupportRelations();
    }

    public List<SupportClientRelation> getNextPageSupportRelations(
            @Nullable SupportClientRelation lastReturnedRole, int pageSize) {
        return rbacClientsRelationsStorage.getNextPageSupportRelations(lastReturnedRole, pageSize);
    }

    /**
     * @param clientId  id менеджера внутренней рекламы
     * @param productId clientId продукта внутренней рекламы
     * @return
     */
    public boolean isInternalAdRelation(ClientId clientId, ClientId productId) {
        ClientsRelation relation = rbacClientsRelationsStorage.getRelationCached(clientId, productId,
                Set.of(ClientsRelationType.INTERNAL_AD_PUBLISHER, ClientsRelationType.INTERNAL_AD_READER));
        return !EMPTY_CLIENTS_RELATION.equals(relation);
    }


    /**
     * Получить отношения продуктов внутреннией рекламы к которым имеет отношение заданный clientId c разбивкой по
     * шардам
     *
     * @param clientIds - идентификаторы клиентов - маркетологов внутренней рекламы (clients.role =
     *                  "internal_ad_manager")
     * @return Map с непустыми списками отношений для каждого шарда
     */
    public Map<Integer, List<ClientsRelation>> getRelatedProductsForInternalAdManager(Set<ClientId> clientIds) {
        return rbacClientsRelationsStorage.getRelations(clientIds, Set.of(ClientsRelationType.INTERNAL_AD_PUBLISHER,
                ClientsRelationType.INTERNAL_AD_READER));
    }


    /**
     * Получить отношения продуктов внутреннией рекламы к которым имеет отношение заданный clientId c разбивкой по
     * clientId
     *
     * @param clientIds
     * @return возвращает мап со списком отношений для каждого clientId
     */
    public Map<Long, List<ClientsRelation>> getRelatedProductsForInternalAdManagers(Collection<ClientId> clientIds) {
        Map<Integer, List<ClientsRelation>> relationsByShard = rbacClientsRelationsStorage.getRelations(clientIds,
                Set.of(ClientsRelationType.INTERNAL_AD_PUBLISHER, ClientsRelationType.INTERNAL_AD_READER));

        return StreamEx.of(relationsByShard.values())
                .flatMap(Collection::stream)
                .mapToEntry(ClientsRelation::getClientIdFrom, Function.identity())
                .grouping();
    }

    public boolean isSupportForClientRelation(ClientId supportClientId, ClientId clientId) {
        var relation = rbacClientsRelationsStorage
                .getRelationCached(supportClientId, clientId, Set.of(ClientsRelationType.SUPPORT_FOR_CLIENT));
        return !EMPTY_CLIENTS_RELATION.equals(relation);
    }

    public List<ClientsRelation> getSupportRelationsByOperatorIds(Collection<ClientId> operatorIds) {
        if (operatorIds.isEmpty()) {
            return emptyList();
        }
        Map<Integer, List<ClientsRelation>> relationsByShard = rbacClientsRelationsStorage.getRelations(operatorIds,
                Set.of(ClientsRelationType.SUPPORT_FOR_CLIENT));

        return StreamEx.of(relationsByShard.values())
                .flatMap(Collection::stream)
                .toList();
    }

    /**
     * Получить отношение менеджера и продукта
     *
     * @param clientId  id клиента - маркетолога внутрнней рекламы
     * @param productId clientId продукта внутренней рекламы
     * @return
     */
    public Optional<ClientsRelation> getInternalAdProductRelation(ClientId clientId, ClientId productId) {
        int shard = shardHelper.getShardByClientId(productId);
        return rbacClientsRelationsStorage.getRelations(shard, Set.of(clientId),
                        Set.of(ClientsRelationType.INTERNAL_AD_PUBLISHER, ClientsRelationType.INTERNAL_AD_READER))
                .stream()
                .filter(relation -> relation.getClientIdTo().equals(productId.asLong()))
                .findFirst();
    }

    /**
     * Получить отношения менеджера и нескольких продуктов
     *
     * @param clientId   id клиента - маркетолога внутренней рекламы
     * @param productIds clientId продуктов внутренней рекламы
     */
    public List<ClientsRelation> getInternalAdProductRelationsForMultipleProducts(
            ClientId clientId, Collection<ClientId> productIds) {
        return rbacClientsRelationsStorage.getRelationsToSpecificClients(
                Set.of(clientId),
                productIds,
                Set.of(ClientsRelationType.INTERNAL_AD_PUBLISHER, ClientsRelationType.INTERNAL_AD_READER));
    }

    /**
     * Добавить доступ к продукту внутренней рекламы для маркетолога. Кладем в шард продукта.
     *
     * @param clientId  id клиента, который получает доступ к продукту
     * @param productId clientId продукта
     */
    public int addInternalAdProductRelation(ClientId clientId, ClientId productId, ClientsRelationType relationType) {
        int shard = shardHelper.getShardByClientId(productId);
        return addInternalAdProductRelation(dslContextProvider.ppc(shard), clientId, productId, relationType);
    }

    /**
     * Добавить доступ к продукту внутренней рекламы для маркетолога. Кладем в шард продукта.
     *
     * @param dslContext
     * @param clientId   id клиента, который получает доступ к продукту
     * @param productId  clientId продукта
     */
    public int addInternalAdProductRelation(DSLContext dslContext, ClientId clientId, ClientId productId,
                                            ClientsRelationType relationType) {
        checkArgument(Set.of(ClientsRelationType.INTERNAL_AD_PUBLISHER, ClientsRelationType.INTERNAL_AD_READER)
                        .contains(relationType),
                "unexpected relation for internal ad products: productId=%s; relationType=%s", productId,
                relationType.toString());
        return rbacClientsRelationsStorage.addRelation(dslContext, clientId, productId, relationType);
    }

    /**
     * Обновить доступ к продукту внутренней рекламы для маркетолога.
     *
     * @param clientId  id клиента с доступом к продукту
     * @param productId clientId продукта
     */
    public int updateInternalAdProductRelation(ClientId clientId, ClientId productId,
                                               ClientsRelationType relationType) {
        int shard = shardHelper.getShardByClientId(productId);
        return updateInternalAdProductRelation(dslContextProvider.ppc(shard), clientId, productId, relationType);
    }

    /**
     * Обновить доступ к продукту внутренней рекламы для маркетолога.
     *
     * @param clientId  id клиента с доступом к продукту
     * @param productId clientId продукта
     */
    public int updateInternalAdProductRelation(DSLContext dslContext,
                                               ClientId clientId, ClientId productId,
                                               ClientsRelationType relationType) {
        checkArgument(Set.of(ClientsRelationType.INTERNAL_AD_PUBLISHER, ClientsRelationType.INTERNAL_AD_READER)
                        .contains(relationType),
                "unexpected relation for internal ad products: productId=%s; relationType=%s", productId,
                relationType.toString());
        return rbacClientsRelationsStorage.updateRelation(dslContext, clientId, relationType, productId);
    }


    /**
     * Удалить связь менеджера с продуктами
     *
     * @param clientId   id менеджера внутренней рекламы
     * @param productIds список продуктов внутренней рекламы
     * @return возвращает количество удаленных записей
     */
    public void removeInternalAdProductsRelations(DSLContext dslContext, ClientId clientId,
                                                  Collection<ClientId> productIds) {
        rbacClientsRelationsStorage.removeRelations(dslContext, clientId, productIds,
                ClientsRelationType.INTERNAL_AD_PUBLISHER,
                ClientsRelationType.INTERNAL_AD_READER);
    }

    /**
     * Получить всех потребителей мобильных целей для владельца мобильных целей (приложений) ownerOfMobileGoals
     */
    @ParametersAreNonnullByDefault
    public List<ClientId> getConsumersOfMobileGoals(ClientId ownerOfMobileGoals) {
        return rbacClientsRelationsStorage.getRelationsByClientIdsTo(
                        shardHelper.getShardByClientId(ownerOfMobileGoals),
                        ownerOfMobileGoals, Set.of(MOBILE_GOALS_ACCESS)
                ).stream()
                .map(ClientsRelation::getClientIdFrom)
                .map(ClientId::fromLong)
                .collect(Collectors.toList());
    }

    /**
     * Получить всех владельцев мобильных целей, к целям которым имеет доступ клиент consumerOfMobileGoals
     */
    @ParametersAreNonnullByDefault
    public List<ClientId> getOwnersOfMobileGoals(ClientId consumerOfMobileGoals) {
        return rbacClientsRelationsStorage.getRelatedClientIdsToByReverseClientsRelations(
                        shardHelper.getShardByClientId(consumerOfMobileGoals), List.of(consumerOfMobileGoals),
                        MOBILE_GOALS_ACCESS)
                .getOrDefault(consumerOfMobileGoals, emptyList());
    }

    /**
     * Получить всех владельцев мобильных целей, к целям которым имеет доступ клиенты consumerOfMobileGoals
     */
    @ParametersAreNonnullByDefault
    public Map<ClientId, List<ClientId>> getOwnersOfMobileGoals(Collection<ClientId> consumerOfMobileGoals) {
        return shardHelper.groupByShard(consumerOfMobileGoals, ShardKey.CLIENT_ID).stream()
                .mapKeyValue((shard, consumers) ->
                        rbacClientsRelationsStorage.getRelatedClientIdsToByReverseClientsRelations(
                                shard, consumers, MOBILE_GOALS_ACCESS)
                )
                .flatMapToEntry(Function.identity())
                .toMap();
    }

    /**
     * Добавить доступ нескольких потребителей мобильных целей consumersOfMobileGoals, к целям клиента
     * ownerOfMobileGoals
     */
    @ParametersAreNonnullByDefault
    public void addMobileGoalsAccessRelations(Collection<ClientId> consumersOfMobileGoals,
                                              ClientId ownerOfMobileGoals) {
        checkArgument(consumersOfMobileGoals.size() <= 500); // не хотим думать как быть с большим кол-вом
        List<Pair<ClientId, ClientId>> clientPairs = mapList(
                consumersOfMobileGoals, consumer -> Pair.of(consumer, ownerOfMobileGoals));
        dslContextProvider.ppc(shardHelper.getShardByClientId(ownerOfMobileGoals)).transaction(conf -> {
            rbacClientsRelationsStorage.doAddRelation(conf.dsl(), clientPairs, MOBILE_GOALS_ACCESS);

            LinkedList<Map.Entry<Integer, List<Pair<ClientId, ClientId>>>> shardData = shardHelper.groupByShard(
                            clientPairs, ShardKey.CLIENT_ID, Pair::getLeft)
                    .stream()
                    .collect(Collectors.toCollection(LinkedList::new));

            innerRecursiveTransactionalShardHandler(
                    (dsl, pairs) -> rbacClientsRelationsStorage.doAddReverseRelation(dsl, pairs, MOBILE_GOALS_ACCESS),
                    shardData
            );
        });
        rbacClientsRelationsStorage.clearCache();
    }

    /**
     * Удалить доступ нескольких потребителей мобильных целей consumersOfMobileGoals к целям клиента
     * ownerOfMobileGoals
     */
    @ParametersAreNonnullByDefault
    public void removeMobileGoalsAccessRelations(Collection<ClientId> consumersOfMobileGoals,
                                                 ClientId ownerOfMobileGoals) {
        checkArgument(consumersOfMobileGoals.size() <= 500); // не хотим думать как быть с большим кол-вом
        List<Pair<ClientId, ClientId>> clientPairs = mapList(
                consumersOfMobileGoals, consumer -> Pair.of(consumer, ownerOfMobileGoals));
        dslContextProvider.ppc(shardHelper.getShardByClientId(ownerOfMobileGoals)).transaction(conf -> {
            rbacClientsRelationsStorage.doRemoveRelations(conf.dsl(), clientPairs, Set.of(MOBILE_GOALS_ACCESS));

            LinkedList<Map.Entry<Integer, List<Pair<ClientId, ClientId>>>> shardData = shardHelper.groupByShard(
                            clientPairs, ShardKey.CLIENT_ID, Pair::getLeft)
                    .stream()
                    .collect(Collectors.toCollection(LinkedList::new));
            innerRecursiveTransactionalShardHandler(
                    (dsl, pairs) -> rbacClientsRelationsStorage.doRemoveReverseRelations(
                            dsl, pairs, Set.of(MOBILE_GOALS_ACCESS)),
                    shardData
            );
        });
        rbacClientsRelationsStorage.clearCache();
    }

    private <T> void innerRecursiveTransactionalShardHandler(
            BiConsumer<DSLContext, T> handler,
            LinkedList<Map.Entry<Integer, T>> shardData
    ) {
        // почему функция рекурсивная: jooq пока не имеет поддержки процедурного
        // управления транзакциями (https://github.com/jOOQ/jOOQ/issues/5376)
        // нужный процесс можно описать вложенными транзакционными блоками, а относительно небольшое кол-во
        // шардов позволят использовать рекурсию
        var current = shardData.poll();
        if (current == null) {
            return;
        }
        int shard = current.getKey();
        T dataOnShard = current.getValue();
        dslContextProvider.ppc(shard).transaction(conf -> {
            handler.accept(conf.dsl(), dataOnShard);
            innerRecursiveTransactionalShardHandler(handler, shardData);
        });
    }

    /**
     * Добавить межклиентскую связь
     * <p>
     * Тип доступа определяет, в том числе и необходимость записи "обратной" связи. "Обратная" связь хранится в
     * reverse_clients_relation и отличается от основной только шардом, на котором она хранится
     *
     * @param clientIdFrom клиент, который будет получать доступ
     * @param clientIdTo   клиент, к которому будет получен доступ
     * @param relationType тип доступа
     */
    @ParametersAreNonnullByDefault
    public void addGenericRelation(ClientId clientIdFrom, ClientId clientIdTo, ClientsRelationType relationType) {
        rbacClientsRelationsStorage.addRelation(
                shardHelper.getShardByClientId(clientIdTo), clientIdFrom, clientIdTo, relationType);

        if (relationType.hasReverseRelation()) {
            rbacClientsRelationsStorage.addReverseRelation(
                    shardHelper.getShardByClientId(clientIdFrom), clientIdFrom, clientIdTo, relationType);
        }
    }

    @ParametersAreNonnullByDefault
    public boolean removeGenericRelation(ClientId clientIdFrom, ClientId clientIdTo, ClientsRelationType relationType) {
        var result = rbacClientsRelationsStorage.removeRelation(
                shardHelper.getShardByClientId(clientIdTo), clientIdFrom, clientIdTo, relationType);

        if (relationType.hasReverseRelation()) {
            var resultReverse = rbacClientsRelationsStorage.removeReverseRelation(
                    shardHelper.getShardByClientId(clientIdFrom), clientIdFrom, clientIdTo, relationType);
            return result > 0 || resultReverse > 0;
        }

        return result > 0;
    }
}
