package ru.yandex.direct.rbac;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.InsertQuery;
import org.jooq.SelectField;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import ru.yandex.direct.dbschema.ppc.enums.ClientsRelationsType;
import ru.yandex.direct.dbschema.ppc.enums.ReverseClientsRelationsType;
import ru.yandex.direct.dbschema.ppc.tables.records.ClientsRelationsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.ReverseClientsRelationsRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.rbac.cache.RbacCache;
import ru.yandex.direct.rbac.model.ClientsRelation;
import ru.yandex.direct.rbac.model.ClientsRelationType;
import ru.yandex.direct.rbac.model.RelatedClients;
import ru.yandex.direct.rbac.model.SupportClientRelation;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.singletonList;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_RELATIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.REVERSE_CLIENTS_RELATIONS;
import static ru.yandex.direct.rbac.RbacClientsRelations.RELATIONS_CACHE_BEAN_NAME;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

/**
 * Работа с отношениями клиент-клиент: низкоуровневые операции с базой, которые
 * не зависят от RbacService (напротив, RbacService зависит от этого компонента).
 * <p>
 * Для использования снаружи rbac не предназначен, все функции package-private.
 */
@ParametersAreNonnullByDefault
@Service
public class RbacClientsRelationsStorage {
    // NB на момент написания List.of здесь нельзя, потому что jOOQ вызывает на нём indexOf(null)
    // и реализация возвращаемого List.of объекта падает с NPE
    private static final List<SelectField<?>> CLIENTS_RELATIONS_FIELDS = Arrays.asList(
            CLIENTS_RELATIONS.RELATION_ID,
            CLIENTS_RELATIONS.CLIENT_ID_FROM,
            CLIENTS_RELATIONS.CLIENT_ID_TO,
            CLIENTS_RELATIONS.TYPE);

    static final ClientsRelation EMPTY_CLIENTS_RELATION = new ClientsRelation();

    private final RbacCache<RelatedClients, ClientsRelation> relationsCache;
    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;

    public RbacClientsRelationsStorage(
            @Qualifier(RELATIONS_CACHE_BEAN_NAME) RbacCache<RelatedClients, ClientsRelation> relationsCache,
            ShardHelper shardHelper, DslContextProvider dslContextProvider) {
        this.relationsCache = relationsCache;
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
    }

    @Nonnull
    ClientsRelation getRelationCached(ClientId clientIdFrom, ClientId clientIdTo,
                                      Set<ClientsRelationType> relationTypes) {
        RelatedClients key = new RelatedClients(clientIdFrom, clientIdTo);
        var relations = relationsCache.computeIfAbsent(singletonList(key),
                l -> ImmutableMap.of(key, getRelation(clientIdFrom, clientIdTo)));
        ClientsRelation relation = relations.get(key);
        if (!EMPTY_CLIENTS_RELATION.equals(relation) &&
                relationTypes.contains(relation.getRelationType())) {
            return relation;
        }
        return EMPTY_CLIENTS_RELATION;
    }

    /**
     * Получить связь двух клиентов. Связь ищем в шарде целевого клиента
     *
     * @param clientIdFrom id клиента, установившего отношения с целевым клиентом
     * @param clientIdTo   id целевого клиента
     */
    @Nonnull
    private ClientsRelation getRelation(ClientId clientIdFrom, ClientId clientIdTo) {
        int shard = shardHelper.getShardByClientId(clientIdTo);
        ClientsRelation result = getRelation(shard, clientIdFrom, clientIdTo);
        return result == null ? EMPTY_CLIENTS_RELATION : result;
    }

    private ClientsRelation getRelation(int shard, ClientId clientIdFrom, ClientId clientIdTo) {
        ClientsRelationsRecord dbRecord = dslContextProvider.ppc(shard)
                .select(CLIENTS_RELATIONS.RELATION_ID, CLIENTS_RELATIONS.CLIENT_ID_FROM,
                        CLIENTS_RELATIONS.CLIENT_ID_TO, CLIENTS_RELATIONS.TYPE)
                .from(CLIENTS_RELATIONS)
                .where(CLIENTS_RELATIONS.CLIENT_ID_FROM.eq(clientIdFrom.asLong())
                        .and(CLIENTS_RELATIONS.CLIENT_ID_TO.eq(clientIdTo.asLong())))
                .fetchOneInto(CLIENTS_RELATIONS);

        if (dbRecord == null) {
            return null;
        }

        return new ClientsRelation()
                .withRelationId(dbRecord.getRelationId())
                .withClientIdFrom(dbRecord.getClientIdFrom())
                .withClientIdTo(dbRecord.getClientIdTo())
                .withRelationType(ClientsRelationType.fromSource(dbRecord.getType()));
    }

    public Long getRelationId(int shard, ClientId clientIdFrom, ClientId clientIdTo,
                              ClientsRelationType relationType) {
        return getRelationId(dslContextProvider.ppc(shard), clientIdFrom, clientIdTo, relationType);
    }

    public Long getRelationId(DSLContext dslContext, ClientId clientIdFrom, ClientId clientIdTo,
                              ClientsRelationType relationType) {
        return dslContext
                .select(CLIENTS_RELATIONS.RELATION_ID)
                .from(CLIENTS_RELATIONS)
                .where(CLIENTS_RELATIONS.CLIENT_ID_FROM.eq(clientIdFrom.asLong())
                        .and(CLIENTS_RELATIONS.CLIENT_ID_TO.eq(clientIdTo.asLong()))
                        .and(CLIENTS_RELATIONS.TYPE.eq(ClientsRelationType.toSource(relationType))))
                .fetchOne(CLIENTS_RELATIONS.RELATION_ID);
    }

    public List<Long> getRelationIds(int shard, ClientId clientIdFrom, Collection<ClientId> clientIdsTo,
                                     Collection<ClientsRelationType> relationTypes) {
        return dslContextProvider.ppc(shard)
                .select(CLIENTS_RELATIONS.RELATION_ID)
                .from(CLIENTS_RELATIONS)
                .where(CLIENTS_RELATIONS.CLIENT_ID_FROM.eq(clientIdFrom.asLong())
                        .and(CLIENTS_RELATIONS.CLIENT_ID_TO.in(mapList(clientIdsTo, ClientId::asLong)))
                        .and(CLIENTS_RELATIONS.TYPE.in(mapList(relationTypes, ClientsRelationType::toSource))))
                .fetch(CLIENTS_RELATIONS.RELATION_ID);
    }

    /**
     * Получить список всех отношений для указанного clientId
     *
     * @param clientIdsFrom - идентификаторы клиентов для которых нужно получить набор отношений
     * @param relations     - тип отношений по которым фильтруем исходный набор
     * @return Map с непустыми списками отношений для каждого шарда
     */
    Map<Integer, List<ClientsRelation>> getRelations(Collection<ClientId> clientIdsFrom,
                                                     Collection<ClientsRelationType> relations) {
        Map<Integer, List<ClientsRelation>> result = new HashMap<>();

        Set<ClientsRelationsType> dbRelationTypes = listToSet(relations, ClientsRelationType::toSource);
        Set<Long> clientIds = listToSet(clientIdsFrom, ClientId::asLong);

        shardHelper.forEachShard(shard -> {
                    List<ClientsRelation> shardResult = doGetRelations(shard, clientIds, dbRelationTypes);
                    result.put(shard, shardResult);
                }
        );
        return EntryStream.of(result)
                .filterValues(v -> !v.isEmpty())
                .toMap();
    }

    List<ClientsRelation> getRelations(int shard, Set<ClientId> clientIds, Set<ClientsRelationType> relations) {
        return doGetRelations(shard,
                mapSet(clientIds, ClientId::asLong),
                mapSet(relations, ClientsRelationType::toSource));
    }

    private List<ClientsRelation> doGetRelations(int shard, Set<Long> clientIds, Set<ClientsRelationsType> relations) {
        return dslContextProvider.ppc(shard)
                .select(CLIENTS_RELATIONS_FIELDS)
                .from(CLIENTS_RELATIONS)
                .where(CLIENTS_RELATIONS.CLIENT_ID_FROM.in(clientIds))
                .and(CLIENTS_RELATIONS.TYPE.in(relations))
                .fetchStreamInto(CLIENTS_RELATIONS)
                .map(r -> new ClientsRelation()
                        .withRelationId(r.getRelationId())
                        .withClientIdFrom(r.getClientIdFrom())
                        .withClientIdTo(r.getClientIdTo())
                        .withRelationType(ClientsRelationType.fromSource(r.getType())))
                .collect(Collectors.toList());
    }

    List<ClientsRelation> getRelationsToSpecificClients(Collection<ClientId> clientIdsFrom,
                                                        Collection<ClientId> clientIdsTo,
                                                        Set<ClientsRelationType> relations) {
        return shardHelper.groupByShard(clientIdsTo, ShardKey.CLIENT_ID, ClientId::asLong)
                .stream()
                .flatMapKeyValue((shard, clientIdsToChunk) ->
                        dslContextProvider.ppc(shard)
                                .select(CLIENTS_RELATIONS_FIELDS)
                                .from(CLIENTS_RELATIONS)
                                .where(CLIENTS_RELATIONS.CLIENT_ID_FROM.in(mapList(clientIdsFrom, ClientId::asLong)))
                                .and(CLIENTS_RELATIONS.CLIENT_ID_TO.in(mapList(clientIdsToChunk, ClientId::asLong)))
                                .and(CLIENTS_RELATIONS.TYPE.in(mapSet(relations, ClientsRelationType::toSource)))
                                .fetchStreamInto(CLIENTS_RELATIONS)
                                .map(r -> new ClientsRelation()
                                        .withRelationId(r.getRelationId())
                                        .withClientIdFrom(r.getClientIdFrom())
                                        .withClientIdTo(r.getClientIdTo())
                                        .withRelationType(ClientsRelationType.fromSource(r.getType()))))
                .toList();
    }

    List<ClientsRelation> getRelationsByClientIdsTo(Collection<ClientId> clientIdsTo,
                                                    Set<ClientsRelationType> relations) {
        return shardHelper.groupByShard(clientIdsTo, ShardKey.CLIENT_ID, ClientId::asLong)
                .stream()
                .flatMapKeyValue((shard, clientIdsToChunk) ->
                        dslContextProvider.ppc(shard)
                                .select(CLIENTS_RELATIONS_FIELDS)
                                .from(CLIENTS_RELATIONS)
                                .where(CLIENTS_RELATIONS.CLIENT_ID_TO.in(mapList(clientIdsToChunk, ClientId::asLong)))
                                .and(CLIENTS_RELATIONS.TYPE.in(mapSet(relations, ClientsRelationType::toSource)))
                                .fetchStreamInto(CLIENTS_RELATIONS)
                                .map(r -> new ClientsRelation()
                                        .withRelationId(r.getRelationId())
                                        .withClientIdFrom(r.getClientIdFrom())
                                        .withClientIdTo(r.getClientIdTo())
                                        .withRelationType(ClientsRelationType.fromSource(r.getType()))))
                .toList();
    }

    List<ClientsRelation> getRelationsByClientIdsTo(int shard, ClientId clientIdTo,
                                                    Set<ClientsRelationType> relations) {
        return dslContextProvider.ppc(shard)
                .select(CLIENTS_RELATIONS_FIELDS)
                .from(CLIENTS_RELATIONS)
                .where(CLIENTS_RELATIONS.CLIENT_ID_TO.eq(clientIdTo.asLong()))
                .and(CLIENTS_RELATIONS.TYPE.in(mapSet(relations, ClientsRelationType::toSource)))
                .fetchStreamInto(CLIENTS_RELATIONS)
                .map(r -> new ClientsRelation()
                        .withRelationId(r.getRelationId())
                        .withClientIdFrom(r.getClientIdFrom())
                        .withClientIdTo(r.getClientIdTo())
                        .withRelationType(ClientsRelationType.fromSource(r.getType())))
                .collect(Collectors.toList());
    }

    /**
     * Получить всех связанных клиентов с указанным типом связи, которые стоят в отношении в позиции to. Поиск идёт
     * по reverse_clients_relations, поэтому тип связи должен поддерживать обратное отношение.
     * Запрос выполняется по reverse_clients_relations, записи в которой находятся в шарде клиента from, поэтому
     * нет необходимости многошардового запроса.
     *
     * @param shardFrom           шард клиентов (from)
     * @param clientIdsFrom       идентификаторы клиентов (from) для которого нужно найти связанных клиентов
     * @param clientsRelationType тип связи
     * @return отображение клиента from в список клиентов (to)
     */
    Map<ClientId, List<ClientId>> getRelatedClientIdsToByReverseClientsRelations(
            int shardFrom, Collection<ClientId> clientIdsFrom,
            ClientsRelationType clientsRelationType) {
        checkArgument(clientsRelationType.getDbReverseClientRelationType() != null,
                "clientsRelationType=%s has no reverse relation", clientsRelationType);

        return dslContextProvider.ppc(shardFrom)
                .select(REVERSE_CLIENTS_RELATIONS.CLIENT_ID_FROM, REVERSE_CLIENTS_RELATIONS.CLIENT_ID_TO)
                .from(REVERSE_CLIENTS_RELATIONS)
                .where(REVERSE_CLIENTS_RELATIONS.CLIENT_ID_FROM.in(mapList(clientIdsFrom, ClientId::asLong)))
                .and(REVERSE_CLIENTS_RELATIONS.TYPE.eq(clientsRelationType.getDbReverseClientRelationType()))
                .fetchGroups(r -> ClientId.fromLong(r.get(REVERSE_CLIENTS_RELATIONS.CLIENT_ID_FROM)),
                        r -> ClientId.fromLong(r.get(REVERSE_CLIENTS_RELATIONS.CLIENT_ID_TO)));
    }

    List<SupportClientRelation> getAllSupportRelations() {
        return StreamEx.of(shardHelper.dbShards())
                .map(this::getAllSupportRelationsSharded)
                .flatMap(Collection::stream)
                .toList();
    }

    @QueryWithoutIndex("Взятие всех элементов таблицы. Кол. элементов в таблице не существенно для разбития по частям")
    private List<SupportClientRelation> getAllSupportRelationsSharded(int shard) {
        return dslContextProvider.ppc(shard)
                .select(CLIENTS_RELATIONS.CLIENT_ID_FROM, CLIENTS_RELATIONS.CLIENT_ID_TO)
                .from(CLIENTS_RELATIONS)
                .where(CLIENTS_RELATIONS.TYPE.eq(ClientsRelationsType.support_for_client))
                .fetch(r -> new SupportClientRelation()
                        .withSupportClientId(ClientId.fromLong(r.get(CLIENTS_RELATIONS.CLIENT_ID_FROM)))
                        .withSubjectClientId(ClientId.fromLong(r.get(CLIENTS_RELATIONS.CLIENT_ID_TO))));
    }

    /**
     * Возвращает набор ролей support_for_client заданной длины начиная со следующей после последней возвращённой.
     * Набор отсортирован сначала по шарду, потом по subjectClientId, потом по supportClientId.
     * Если последняя возвращённая роль не задана, то возвращаются роли с начала первого шарда.
     */
    List<SupportClientRelation> getNextPageSupportRelations(
            @Nullable SupportClientRelation lastReturnedRole, int pageSize) {
        int continueShard = nvl(ifNotNull(lastReturnedRole,
                m -> shardHelper.getShardByClientId(m.getSubjectClientId())), 0);
        Iterator<Integer> shards = StreamEx.of(shardHelper.dbShards())
                .filter(shard -> continueShard <= shard)
                .sorted()
                .iterator();
        ArrayList<SupportClientRelation> roles = new ArrayList<>(pageSize);
        int limit = pageSize - roles.size();
        while (shards.hasNext() && 0 < limit) {
            List<SupportClientRelation> shardRoles = getNextPageSupportRelationsSharded(
                    shards.next(), lastReturnedRole, limit);
            roles.addAll(shardRoles);
            lastReturnedRole = null;
            limit = pageSize - roles.size();
        }
        return roles;
    }

    private List<SupportClientRelation> getNextPageSupportRelationsSharded(
            int shard, @Nullable SupportClientRelation lastReturnedRole, int limit) {
        Condition condition = CLIENTS_RELATIONS.TYPE.eq(ClientsRelationsType.support_for_client);
        if (lastReturnedRole != null) {
            Long clientId = lastReturnedRole.getSubjectClientId().asLong();
            Long supportId = lastReturnedRole.getSupportClientId().asLong();
            Condition continueLastClient = CLIENTS_RELATIONS.CLIENT_ID_TO.eq(clientId)
                    .and(CLIENTS_RELATIONS.CLIENT_ID_FROM.greaterThan(supportId));
            Condition continueNextClients = CLIENTS_RELATIONS.CLIENT_ID_TO.greaterThan(clientId);
            condition = condition.and(continueLastClient.or(continueNextClients));
        }
        return dslContextProvider.ppc(shard)
                .select(CLIENTS_RELATIONS.CLIENT_ID_FROM, CLIENTS_RELATIONS.CLIENT_ID_TO)
                .from(CLIENTS_RELATIONS)
                .where(condition)
                .orderBy(CLIENTS_RELATIONS.CLIENT_ID_TO, CLIENTS_RELATIONS.CLIENT_ID_FROM)
                .limit(limit)
                .fetch(r -> new SupportClientRelation()
                        .withSupportClientId(ClientId.fromLong(r.get(CLIENTS_RELATIONS.CLIENT_ID_FROM)))
                        .withSubjectClientId(ClientId.fromLong(r.get(CLIENTS_RELATIONS.CLIENT_ID_TO)))
                );
    }

    int addRelation(int shardTo, ClientId clientIdFrom, ClientId clientIdTo, ClientsRelationType relationType) {
        return addRelation(dslContextProvider.ppc(shardTo), clientIdFrom, clientIdTo, relationType);
    }

    int addReverseRelation(int shardFrom, ClientId clientIdFrom, ClientId clientIdTo,
                           ClientsRelationType relationType) {
        return addReverseRelation(dslContextProvider.ppc(shardFrom), clientIdFrom, clientIdTo, relationType);
    }

    int addRelation(DSLContext dslContext, ClientId clientIdFrom, ClientId clientIdTo,
                    ClientsRelationType relationType) {
        int res = doAddRelation(dslContext, List.of(Pair.of(clientIdFrom, clientIdTo)), relationType);
        clearCache();
        return res;
    }

    /**
     * @param dslContext       должен соответствовать шарду клиентов to
     * @param clientsFromAndTo список пар клиентов from и to, все клиенты to должны быть из одного шарда
     */
    int doAddRelation(DSLContext dslContext, List<Pair<ClientId, ClientId>> clientsFromAndTo,
                      ClientsRelationType clientsRelationType) {
        ClientsRelationsType dbClientRelationType = clientsRelationType.getDbClientRelationType();
        InsertQuery<ClientsRelationsRecord> insertQuery = dslContext.insertQuery(CLIENTS_RELATIONS);
        EntryStream.zip(clientsFromAndTo, shardHelper.generateClientRelationIds(clientsFromAndTo.size()))
                .forKeyValue((clientFromAndTo, relationId) -> {
                    insertQuery.addValues(Map.of(
                            CLIENTS_RELATIONS.RELATION_ID, relationId,
                            CLIENTS_RELATIONS.CLIENT_ID_FROM, clientFromAndTo.getLeft().asLong(),
                            CLIENTS_RELATIONS.CLIENT_ID_TO, clientFromAndTo.getRight().asLong(),
                            CLIENTS_RELATIONS.TYPE, dbClientRelationType
                    ));
                    insertQuery.newRecord();
                });
        return insertQuery.execute();
    }

    int updateRelation(
            DSLContext dslContext, ClientId relatedClientId, ClientsRelationType relationType, ClientId clientId) {
        int res = doUpdateRelation(dslContext, clientId, relationType, relatedClientId);
        clearCache();
        return res;
    }

    private int doUpdateRelation(
            DSLContext dslContext, ClientId to, ClientsRelationType clientsRelationType, ClientId from) {
        return dslContext.update(CLIENTS_RELATIONS)
                .set(CLIENTS_RELATIONS.TYPE, ClientsRelationType.toSource(clientsRelationType))
                .where(CLIENTS_RELATIONS.CLIENT_ID_FROM.eq(from.asLong()).and(
                        CLIENTS_RELATIONS.CLIENT_ID_TO.eq(to.asLong())))
                .execute();
    }

    int removeRelation(int shard, ClientId clientIdFrom, ClientId clientIdTo, ClientsRelationType... relationType) {
        int res = doRemoveRelations(shard, clientIdFrom, Set.of(clientIdTo), Set.of(relationType));
        clearCache();
        return res;
    }

    /**
     * Удалить связи с неcколькими clientIdTo в текущем шарде.
     *
     * @param clientIdFrom клиент которому нужно разорвать связь
     * @param clientIdsTo  множество клиентов в текущем шарде с которыми нужно разорвать связь.
     * @param relationType тип связи
     */
    int removeRelations(DSLContext dslContext, ClientId clientIdFrom, Collection<ClientId> clientIdsTo,
                        ClientsRelationType... relationType) {
        int res = doRemoveRelations(dslContext, clientIdFrom, clientIdsTo, Set.of(relationType));
        clearCache();
        return res;
    }

    private int doRemoveRelations(int shard, ClientId from, Collection<ClientId> to,
                                  Set<ClientsRelationType> clientsRelationType) {
        return doRemoveRelations(dslContextProvider.ppc(shard), from, to, clientsRelationType);
    }

    private int doRemoveRelations(DSLContext dslContext, ClientId from, Collection<ClientId> to,
                                  Set<ClientsRelationType> clientsRelationType) {
        return doRemoveRelations(dslContext, mapList(to, toClient -> Pair.of(from, toClient)), clientsRelationType);
    }

    int doRemoveRelations(DSLContext dslContext, List<Pair<ClientId, ClientId>> clientsFromAndTo,
                          Set<ClientsRelationType> clientsRelationType) {
        return dslContext.deleteFrom(CLIENTS_RELATIONS)
                .where(CLIENTS_RELATIONS.TYPE.in(mapSet(clientsRelationType, ClientsRelationType::toSource)))
                .and(row(CLIENTS_RELATIONS.CLIENT_ID_FROM, CLIENTS_RELATIONS.CLIENT_ID_TO).in(
                        mapList(clientsFromAndTo, pair -> row(pair.getLeft().asLong(), pair.getRight().asLong()))
                ))
                .execute();
    }

    int addReverseRelation(DSLContext dslContext, ClientId clientIdFrom, ClientId clientIdTo,
                           ClientsRelationType relationType) {
        int res = doAddReverseRelation(dslContext, List.of(Pair.of(clientIdFrom, clientIdTo)), relationType);
        clearCache();
        return res;
    }

    /**
     * @param dslContext       должен соответствовать шарду, на котором клиенты from
     * @param clientsFromAndTo список пар клиентов from и to
     */
    int doAddReverseRelation(DSLContext dslContext, List<Pair<ClientId, ClientId>> clientsFromAndTo,
                             ClientsRelationType clientsRelationType) {
        ReverseClientsRelationsType dbReverseClientRelationType = clientsRelationType.getDbReverseClientRelationType();
        checkArgument(dbReverseClientRelationType != null,
                "clientsRelationType=%s has no reverse relation", clientsRelationType);
        InsertQuery<ReverseClientsRelationsRecord> insertQuery = dslContext.insertQuery(REVERSE_CLIENTS_RELATIONS);
        EntryStream.zip(clientsFromAndTo, shardHelper.generateReverseClientsRelationsIds(clientsFromAndTo.size()))
                .forKeyValue((clientFromAndTo, reverseRelationId) -> {
                    insertQuery.addValues(Map.of(
                            REVERSE_CLIENTS_RELATIONS.REVERSE_RELATION_ID, reverseRelationId,
                            REVERSE_CLIENTS_RELATIONS.CLIENT_ID_FROM, clientFromAndTo.getLeft().asLong(),
                            REVERSE_CLIENTS_RELATIONS.CLIENT_ID_TO, clientFromAndTo.getRight().asLong(),
                            REVERSE_CLIENTS_RELATIONS.TYPE, dbReverseClientRelationType
                    ));
                    insertQuery.newRecord();
                });
        return insertQuery.execute();
    }

    /**
     * Удалить связи в reverse_clients_relations с clientIdTo
     *
     * @param clientIdFrom клиент которому нужно разорвать связь
     * @param clientIdTo   клиент с которым нужно разорвать связь
     * @param relationType тип связи
     */
    int removeReverseRelation(int shardFrom, ClientId clientIdFrom, ClientId clientIdTo,
                              ClientsRelationType... relationType) {
        return removeReverseRelation(shardFrom, clientIdFrom, Set.of(clientIdTo), Set.of(relationType));
    }

    /**
     * Удалить связи в reverse_clients_relations с clientIdTo
     *
     * @param clientIdFrom  клиент которому нужно разорвать связь
     * @param clientIdsTo   клиенты с которыми нужно разорвать связь
     * @param relationTypes типы связи
     */
    private int removeReverseRelation(int shardFrom, ClientId clientIdFrom, Collection<ClientId> clientIdsTo,
                                      Set<ClientsRelationType> relationTypes) {
        int res = doRemoveReverseRelations(dslContextProvider.ppc(shardFrom),
                mapList(clientIdsTo, to -> Pair.of(clientIdFrom, to)), relationTypes);
        clearCache();
        return res;
    }

    int doRemoveReverseRelations(DSLContext dslContext, List<Pair<ClientId, ClientId>> clientsFromAndTo,
                                 Set<ClientsRelationType> clientsRelationType) {
        return dslContext.deleteFrom(REVERSE_CLIENTS_RELATIONS)
                .where(REVERSE_CLIENTS_RELATIONS.TYPE.in(
                        mapSet(clientsRelationType, ClientsRelationType::getDbReverseClientRelationType)))
                .and(row(REVERSE_CLIENTS_RELATIONS.CLIENT_ID_FROM, REVERSE_CLIENTS_RELATIONS.CLIENT_ID_TO).in(
                        mapList(clientsFromAndTo, pair -> row(pair.getLeft().asLong(), pair.getRight().asLong()))
                ))
                .execute();
    }

    /**
     * Очистка кеша, стоит вызывать при изменении отношений клиент-клиент
     */
    void clearCache() {
        relationsCache.clear();
    }
}
