package ru.yandex.direct.rbac;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.dbschema.ppc.enums.ClientsRole;
import ru.yandex.direct.dbschema.ppc.enums.UsersAgencyLimRepType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.rbac.model.RbacAccessType;
import ru.yandex.direct.rbac.model.RbacCampPerms;
import ru.yandex.direct.rbac.model.Representative;
import ru.yandex.direct.rbac.model.SubclientGrants;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.dbschema.ppc.Tables.AGENCY_LIM_REP_CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPS_FOR_SERVICING;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS_AGENCY;
import static ru.yandex.direct.rbac.ClientPerm.SUPER_SUBCLIENT;
import static ru.yandex.direct.rbac.RbacRepType.CHIEF;
import static ru.yandex.direct.rbac.RbacRepType.LIMITED;
import static ru.yandex.direct.rbac.RbacRepType.MAIN;
import static ru.yandex.direct.rbac.RbacRole.AGENCY;
import static ru.yandex.direct.rbac.RbacRole.CLIENT;
import static ru.yandex.direct.rbac.RbacRole.MEDIA;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component(RbacService.RBAC_SERVICE)
@ParametersAreNonnullByDefault
public class RbacService implements RbacUserLookupService {

    /**
     * Фиксированный uid для внутренней рекламы для доступа к счетчикам сервисов Яндекса
     * Это временное решение пока IDM не сделает: https://st.yandex-team.ru/IDM-10208#5fad02ea3f25c34adadd9c10
     * чтобы можно было получить доступ для каждого продукта внутренней рекламы.
     * Это константа так-же есть в perl в MetrikaCounters.pm и менять их надо одновременно
     */
    public static final long INTERNAL_AD_UID_PRODUCTS_FOR_GET_METRIKA_COUNTERS = 1230527186;

    public static final String RBAC_SERVICE = "rbacService";

    private static final Logger logger = LoggerFactory.getLogger(RbacService.class);

    protected final ShardHelper shardHelper;

    protected final DslContextProvider dslContextProvider;

    private final PpcRbac ppcRbac;

    private final RbacClientsRelations rbacClientsRelations;

    @Autowired
    public RbacService(
            ShardHelper shardHelper,
            DslContextProvider dslContextProvider,
            PpcRbac ppcRbac,
            RbacClientsRelations rbacClientsRelations
    ) {
        this.shardHelper = Objects.requireNonNull(shardHelper, "shardHelper");
        this.dslContextProvider = dslContextProvider;
        this.ppcRbac = ppcRbac;
        this.rbacClientsRelations = rbacClientsRelations;
    }

    /**
     * Получить роль по uid
     * TODO: надо прикрутить request-scope кеширование
     *
     * @param uid
     * @return
     */
    @Nonnull
    public RbacRole getUidRole(long uid) {
        return getUidsRoles(singletonList(uid)).get(uid);
    }

    /**
     * Получить роли пользователей по переданной коллекции uid
     *
     * @return отбражение uid в роль пользователя с этим uid
     */
    @Override
    @Nonnull
    public Map<Long, RbacRole> getUidsRoles(Collection<Long> uids) {
        return ppcRbac.getUidsRoles(uids);
    }

    /**
     * Получить uid шеф-представителя
     */
    public long getChief(long uid) {
        RbacRole role = getUidRole(uid);
        return getChief(uid, role);
    }

    /**
     * Получить uid шеф-представителя, если известна роль.
     * <p>
     * Следует использовать, только если требуется минимизировать количество запросов к БД RBAC.
     * Для большинства случаев должен подойти {@link #getChief(long)}
     *
     * @param uid  UID
     * @param role Роль, соответсвующая {@code uid}
     * @return chiefUid
     * @see #getChief(long)
     */
    public long getChief(long uid, RbacRole role) {
        return ppcRbac.getUserPerminfo(uid)
                .filter(pi -> pi.role() == role)
                .map(UserPerminfo::chiefUid)
                .orElse(uid);
    }

    /**
     * Получить список представителей клиента по ClientID для похода за счетчиками
     * Для внутренней рекламы вернет фиксированный uid - временный костыль. В будущем этот метод будет удален в пользу
     * {@link #getClientRepresentativesUids}
     * Нужно использовать только в тех местах где надо получить список доступных счетчиков как для внутренней рекламы
     * так и для остального Директа. Если не уверен подходит ли метод, то можно уточнить у @xy6er
     *
     * @param clientId ID клиента
     * @return Коллекция uid всех представителей клиента
     */
    @Nonnull
    public List<Long> getClientRepresentativesUidsForGetMetrikaCounters(ClientId clientId) {
        return isInternalAdProduct(clientId)
                ? List.of(INTERNAL_AD_UID_PRODUCTS_FOR_GET_METRIKA_COUNTERS)
                : List.copyOf(getClientRepresentativesUids(clientId));
    }

    /**
     * Получить список представителей клиента по ClientID
     *
     * @param clientId ID клиента
     * @return Коллекция uid всех представителей клиента
     */
    @Nonnull
    public Collection<Long> getClientRepresentativesUids(ClientId clientId) {
        return getClientRepresentatives(clientId)
                .stream()
                .map(Representative::getUserId)
                .collect(toList());
    }

    /**
     * Получить представителей клиента
     *
     * @param clientId id клиента, представителей которого нужно получить
     */
    @Nonnull
    public Collection<Representative> getClientRepresentatives(ClientId clientId) {
        return massGetClientRepresentatives(Collections.singletonList(clientId));
    }

    /**
     * Получить представителей клиентов
     *
     * @param clientIds коллекция id клиентов, представителей которых нужно получить
     * @return коллекция представителей
     */
    @Nonnull
    public Collection<Representative> massGetClientRepresentatives(Collection<ClientId> clientIds) {
        return ppcRbac.massGetClientRepresentatives(clientIds);
    }

    /**
     * Получить uid шеф-представителя для клиента
     *
     * @param clientId ID клиента
     * @return UID шеф-представителя для клиента
     * @throws IllegalStateException если для переданного clientId нет chief'а
     * @see ShardHelper#isExistentClientId(long)
     */
    public long getChiefByClientId(ClientId clientId) throws IllegalStateException {
        return getChiefByClientIdOptional(clientId)
                .orElseThrow(() -> new IllegalStateException("Can't find chief for clientId " + clientId));
    }

    /**
     * Получить uid шеф-представителя для клиента
     *
     * @param clientId ID клиента
     * @return UID шеф-представителя для клиента
     */
    public Optional<Long> getChiefByClientIdOptional(ClientId clientId) {
        return ppcRbac.getClientPerminfo(clientId).map(ClientPerminfo::chiefUid);
    }

    /**
     * Получить шефов по коллекции ClientID
     *
     * @param clientIds коллекция id клиентов, представителей которых нужно получить
     * @return мапа ClientID -> chiefUid
     */
    @Override
    public Map<ClientId, Long> getChiefsByClientIds(Collection<ClientId> clientIds) {
        return EntryStream.of(ppcRbac.getClientsPerminfo(clientIds))
                .flatMapValues(StreamEx::of)
                .mapValues(ClientPerminfo::chiefUid)
                .nonNullValues()
                .toMap();
    }

    public long getEffectiveOperatorUid(long operatorUid, @Nullable Long clientUid) {
        long effectiveUid = ppcRbac.replaceManagerByTeamLead(operatorUid);
        if (effectiveUid == operatorUid && clientUid != null) {
            effectiveUid = rbacClientsRelations.replaceOperatorUidByClientUid(operatorUid, clientUid);
        }

        return effectiveUid;
    }

    /**
     * Проверить, имеет ли оператор права на клиента
     *
     * @param operatorUid uid оператора
     * @param clientUid   uid клиента
     */
    public boolean isOwner(long operatorUid, long clientUid) {
        return ppcRbac.getAccessTypes(operatorUid, singletonList(clientUid))
                .getOrDefault(clientUid, RbacAccessType.NONE)
                .isOwner();
    }

    /**
     * Возвращает клиентов, на которые есть права у оператора, из предложенного списка
     *
     * @param operatorUid uid оператора
     * @param clientUids  uid клиентов
     */
    public Set<Long> getOwnedClients(long operatorUid, Collection<Long> clientUids) {
        return EntryStream.of(ppcRbac.getAccessTypes(operatorUid, clientUids))
                .filterValues(RbacAccessType::isOwner)
                .keys()
                .collect(Collectors.toSet());
    }

    /**
     * Проверить, имеет ли оператор права на модификацию данных клиента
     *
     * @param operatorUid uid оператора
     * @param clientUid   uid клиента
     */
    public boolean canWrite(long operatorUid, long clientUid) {
        UserPerminfo userPermInfo = getUserPermInfo(operatorUid);

        if (userPermInfo.isReadonlyRep()) {
            return false;
        }

        boolean notSubClientOrSuperSubClient =
                !isUnderAgency(userPermInfo.chiefUid()) || isUserSuperSubclient(userPermInfo);
        boolean hasWriteAccess = ppcRbac.getAccessTypes(operatorUid, singletonList(clientUid))
                .getOrDefault(clientUid, RbacAccessType.NONE).canWrite();
        return notSubClientOrSuperSubClient && hasWriteAccess;
    }

    /**
     * Проверить, имеет ли оператор права на просмотр данных клиента
     *
     * @param operatorUid uid оператора
     * @param clientUid   uid клиента
     */
    public boolean canRead(long operatorUid, long clientUid) {
        return ppcRbac.getAccessTypes(operatorUid, singletonList(clientUid))
                .getOrDefault(clientUid, RbacAccessType.NONE).canRead();
    }

    public boolean canAgencyCreateCampaign(long agencyUid, long clientUid) {
        UserPerminfo agency = ppcRbac.getUserPerminfo(agencyUid).orElse(null);
        UserPerminfo client = ppcRbac.getUserPerminfo(clientUid).orElse(null);
        if (agency == null || client == null || !agency.hasRole(AGENCY) || !client.hasRole(CLIENT)) {
            return false;
        }
        if (agency.hasRepType(CHIEF, MAIN)) {
            return Objects.equals(agency.clientId(), client.agencyClientId());
        } else {
            return client.agencyUids().contains(agency.userId());
        }
    }

    public boolean canOperatorEditInternationalisation(long operatorUid) {
        return ppcRbac.getUserPerminfo(operatorUid)
                .map(p -> p.role() == RbacRole.SUPER)
                .orElse(false);
    }

    /**
     * Проверить, принадлежит ли клиент какому-нибудь агентству
     */
    public boolean isUnderAgency(long uid) {
        return ppcRbac.getUserPerminfo(uid).map(u -> u.agencyClientId() != null)
                .orElseThrow(() -> new RuntimeException("Unable to get UserPerminfo for uid " + uid));
    }

    /**
     * Вернуть коллекцию uid всех представителей агенств, имеющих права на субклиента
     * <p>
     * Если клиент не является субклиентом ни одного агенства вернётся пустая коллекция
     * <p>
     * Если у агентства есть limited-представитель для этого клиента, то вернется UID этого представителя,
     * всех main-представителей и шефа, иначе вернутся только uid всех main-представителей и шефа
     *
     * @param uid Uid клиента
     */
    @Nonnull
    public Collection<Long> getAgencyRepsOfSubclient(long uid) {
        Optional<UserPerminfo> userInfo = ppcRbac.getUserPerminfo(uid);
        Collection<Representative> reps = userInfo
                .map(info -> filterList(ppcRbac.getClientRepresentatives(info.agencyClientId()),
                        rep -> !rep.isLimited() || info.agencyUids().contains(rep.getUserId())))
                .orElseThrow(() -> new RuntimeException("Unable to get UserPerminfo for uid " + uid));
        return mapList(reps, Representative::getUserId);
    }

    /**
     * Вернуть коллекцию uid chief-представителей агенств субклиента
     * <p>
     * Если клиент не является субклиентом ни одного агенства вернётся пустая коллекция
     *
     * @param uid Uid клиента
     */
    @Nonnull
    public Collection<Long> getAgencyChiefsOfSubclient(long uid) {
        Collection<Long> agencyReps = getAgencyRepsOfSubclient(uid);
        Set<Long> agencyClientIds = new HashSet<>(shardHelper.getClientIdsByUids(agencyReps).values());
        return getChiefsByClientIds(mapList(agencyClientIds, ClientId::fromLong)).values();
    }

    /**
     * Возвращает список UID менеджеров, доступных данному пользователю.
     * - Для клиентов и агентств это ответственных за них менеджер
     * - Для менеджера это сам менеджер
     * - Для тимлида это сам тимлид и подчиненные ему менеджеры,
     * - Для супертимлида это сам супертимлид, подчиненные ему тимлиды и подчиненные им менеджеры
     * - Для супера и суперридера это вообще все менеджеры
     *
     * @param uid UID пользователя (клиента/агентства/менеджера/etc), для которого требуется получить список менеджеров
     */
    @Nonnull
    public List<Long> getManagersOfUser(long uid) {
        Optional<UserPerminfo> userInfo = ppcRbac.getUserPerminfo(uid);
        return userInfo.map(info -> {
            RbacRole role = info.role();
            if (role == RbacRole.CLIENT || role == RbacRole.AGENCY) {
                return new ArrayList<>(info.managerUids());
            }
            if (role == RbacRole.SUPERREADER || role == RbacRole.SUPER) {
                return ppcRbac.getAllManagers();
            }
            if (role == RbacRole.MANAGER) {
                List<Long> result = ppcRbac.getSupervisorSubordinates(uid);
                result.add(uid);
                return result;
            }
            return Collections.<Long>emptyList();
        }).orElse(Collections.emptyList());
    }

    /**
     * Имеет ли субклиент права на редактирование своих кампаний по данным RBAC
     * Это право может переопределяться в таблице clients, для прикладных целей лучше использовать
     * ClientService.isSuperSubclient
     *
     * @param clientId ID клиента
     * @return true если у клиента есть права на редактирование своих кампаний
     */
    public boolean isSuperSubclient(@Nonnull ClientId clientId) {
        return ppcRbac.getClientPerminfo(clientId).map(p -> p.perms().contains(ClientPerm.SUPER_SUBCLIENT))
                .orElseThrow(() -> new RuntimeException("Unable to get ClientPerminfo for clientId " + clientId));
    }

    /**
     * Имеет ли субклиент права на редактирование своих кампаний по данным RBAC
     *
     * @param userPerminfo набор прав пользователя
     * @return true если у клиента есть права на редактирование своих кампаний
     */
    public boolean isUserSuperSubclient(UserPerminfo userPerminfo) {
        return userPerminfo.hasPerm(SUPER_SUBCLIENT);
    }

    /**
     * Определяет какие кампании из переданного списка видимы оператору
     *
     * @param operatorUid оператор
     * @param cids        список id кампаний
     * @return множество кампаний видимых оператору
     */
    @Nonnull
    public Set<Long> getVisibleCampaigns(long operatorUid, Collection<Long> cids) {
        return EntryStream.of(ppcRbac.getCampPerms(operatorUid, cids))
                .filterValues(RbacCampPerms::canRead)
                .keys()
                .toSet();
    }

    /**
     * Определяет, для каких кампаний у оператора имеются write права
     *
     * @param operatorUid оператор
     * @param cids        список id кампаний
     * @return множество кампаний, доступных для редактирования оператору
     */
    @Nonnull
    public Set<Long> getWritableCampaigns(long operatorUid, Collection<Long> cids) {
        // нет прав на запись, если operator_role == 'media' (Perl: base.pm#_no_rights_if_mediaplanner)
        UserPerminfo operator = ppcRbac.getUserPerminfo(operatorUid).orElse(null);
        if (operator == null || operator.hasRole(MEDIA)) {
            return Collections.emptySet();
        }

        return EntryStream.of(ppcRbac.getCampPerms(operatorUid, cids))
                .filterValues(RbacCampPerms::canWrite)
                .keys()
                .toSet();
    }

    @Nonnull
    public Map<Long, RbacCampPerms> getCampaignsRights(long operatorUid, Collection<Long> cids) {
        return ppcRbac.getCampPerms(operatorUid, cids);
    }

    @Nonnull
    public Map<Long, Boolean> getCampaignsWaitForServicing(Collection<Long> campaignIds) {
        Set<Long> campaignsIdsWhichWaitForServicing = shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .stream()
                .mapKeyValue((shard, cids) ->
                        dslContextProvider.ppc(shard)
                                .select(CAMPS_FOR_SERVICING.CID)
                                .from(CAMPS_FOR_SERVICING)
                                .where(CAMPS_FOR_SERVICING.CID.in(cids))
                                .fetch(CAMPS_FOR_SERVICING.CID)
                )
                .flatMap(StreamEx::of)
                .toSet();

        return StreamEx.of(campaignIds)
                .distinct()
                .toMap(campaignsIdsWhichWaitForServicing::contains);
    }

    /**
     * Получить всех доступных представителю агентства субклиентов агентства. Для того чтобы получить
     * всех субклиентов агентства, нужно вызывать от главного представителя агентства
     *
     * @param agencyRepUid uid представителя агентства
     * @return список uid всех субклиентов агентства (включая представителей субклиентов)
     */
    public List<Long> getAgencySubclients(long agencyRepUid) {
        return ppcRbac.getAgencySubclients(agencyRepUid);
    }

    /**
     * Возвращает uid'ы всех клиентов с которыми у фрилансера есть отношения предоставляющие доступ к их материалам.
     */
    public List<Long> getRelatedClientsChiefs(ClientId freelancerId) {
        return ppcRbac.getRelatedClientsChiefs(freelancerId);
    }

    /**
     * Получить главных преставителей, для переданной коллекции преставителей. Если во входных данных будут
     * несколько представителей одного клиента, в выходном списке будет только один главный представитель этого
     * клиента.
     *
     * @param uids коллекция uid
     * @return список uid главных преставителей, соответствующий клиентам, представители которых переданы на вход
     */
    public List<Long> getChiefSubclients(Collection<Long> uids) {
        return StreamEx.of(ppcRbac.getUsersPerminfo(uids).values())
                .flatMap(StreamEx::of)
                .map(UserPerminfo::chiefUid)
                .distinct()
                .collect(toList());
    }

    /**
     * Возвращает мап (uid -> uid главного представителя).
     */
    public Map<Long, Long> getChiefUidByUids(Collection<Long> uids) {
        return EntryStream.of(ppcRbac.getUsersPerminfo(uids))
                .filterValues(Optional::isPresent)
                .mapValues(userInfoOptional -> userInfoOptional.get().chiefUid())
                .toMap();
    }

    /**
     * Получить ограниченных представителей агентств, по переданной коллекции uid представителей
     *
     * @param uids коллекция uid
     * @return список ограниченных представителей агентств
     */
    public List<Representative> getAgencyLimitedRepresentatives(Collection<Long> uids) {
        if (uids.isEmpty()) {
            return emptyList();
        }
        Map<Long, Optional<UserPerminfo>> info = ppcRbac.getUsersPerminfo(uids);
        return StreamEx.of(info.values())
                .flatMap(StreamEx::of)
                .filter(r -> RbacRole.AGENCY.equals(r.role()))
                .map(r -> Representative.create(r.clientId(), r.userId(), r.repType()))
                .filter(Representative::isLimited)
                .toList();
    }

    /**
     * Получить субклиентов агентства по списку uid представителей агентства
     * Возвращает мапу: uid представителя -> список ClientID его субклиентов
     * Если у представителя нет клиентов, то в результирующей мапе его не будет
     */
    @Nonnull
    public Map<Long, List<Long>> getSubclientsOfAgencyReps(Collection<Long> agencyUids) {
        return ppcRbac.getSubclientsOfAgencyReps(agencyUids);
    }

    /**
     * Получить доступных представителю агентства представителей субклиентов среди uid из переданной коллекции
     *
     * @param agencyRepUid  uid представителя агентства
     * @param subclientUids коллекция uid проверяемых представителей клиентов
     * @return список uid представителей субклиентов, доступных представителю агентства
     */
    @Nonnull
    public List<Long> getAccessibleAgencySubclients(Long agencyRepUid, Collection<Long> subclientUids) {
        return ppcRbac.getAccessibleAgencySubclients(agencyRepUid, subclientUids);
    }

    /**
     * Фильтрует переданные идентификаторы представителей.
     * Остаются только те, с клиентами которых у заданного фрилансера есть есть отношения предоставляющие доступ к их
     * материалам.
     */
    @Nonnull
    public List<Long> getAccessibleRelatedClients(Long freelancerUid, Collection<Long> clientUids) {
        return ppcRbac.getAccessibleRelatedClients(freelancerUid, clientUids);
    }

    /**
     * Получить разрешения, которые выдали агенства указанному клиенту
     *
     * @param agenciesUids коллекция uid агентств клиента
     * @param clientId     id клиента
     * @return коллекция разрешений, выданных клиенту от всех его агентств
     */
    @Nonnull
    public Collection<SubclientGrants> getSubclientGrants(Collection<Long> agenciesUids, ClientId clientId) {
        return getSubclientsGrants(agenciesUids, Collections.singletonList(clientId));
    }

    /**
     * Получить разрешения, которые выдало агенства на своих клиентов
     *
     * @param agencyUid  агентство
     * @param clientsIds коллекци id клиентов, под этим агенством
     * @return коллекция разрешений, выданных клиентам от агентства
     */
    public Collection<SubclientGrants> getSubclientsGrants(Long agencyUid, Collection<ClientId> clientsIds) {
        return getSubclientsGrants(Collections.singletonList(agencyUid), clientsIds);
    }

    /**
     * Получить разрешения клиентов из переданной коллекции, выданные каждому клиенту от всех его агентств
     * <p>
     * Внимание! Все клиенты должны быть под указанными агенствами. Если это условие не выполняется, появятся
     * лишние элементы SubclientGrants, в которых всё запрещено клиенту, который не под этим агенством.
     *
     * @param agenciesUids коллекция uid агентств клиента
     * @param clientIds    id клиента
     * @return коллекция разрешений выданных разрешение от всех агентств клиентов
     */
    private Collection<SubclientGrants> getSubclientsGrants(Collection<Long> agenciesUids,
                                                            Collection<ClientId> clientIds) {
        Map<ClientId, Optional<ClientPerminfo>> clients = ppcRbac.getClientsPerminfo(clientIds);
        Map<Long, Optional<UserPerminfo>> agencies = ppcRbac.getUsersPerminfo(agenciesUids);

        return StreamEx.of(agenciesUids)
                .cross(clientIds)
                .mapKeyValue(
                        (agencyUid, clientId) -> {
                            UserPerminfo agency = agencies.getOrDefault(agencyUid, Optional.empty()).orElse(null);
                            ClientPerminfo client = clients.getOrDefault(clientId, Optional.empty()).orElse(null);
                            return SubclientGrants
                                    .buildWithPermissions(agencyUid, clientId,
                                            getAgencyClientPermissions(agency, client)
                                    );
                        }
                ).toList();
    }

    private Set<ClientPerm> getAgencyClientPermissions(
            @Nullable UserPerminfo agency,
            @Nullable ClientPerminfo client) {
        if (client == null || agency == null) {
            return emptySet();
        }
        if (!Objects.equals(client.agencyClientId(), agency.clientId())) {
            return emptySet();
        }
        if (agency.hasRepType(CHIEF, MAIN) || client.agencyUids().contains(agency.userId())) {
            return client.perms();
        } else {
            return emptySet();
        }
    }

    /**
     * Для клиента, который не под агенством выдаёт все разрешения
     */
    public Collection<SubclientGrants> getClientGrantsWithoutAgency(ClientId clientId) {
        return Collections.singletonList(SubclientGrants.buildWithAllGrants(null, clientId));
    }

    public Map<UidAndClientId, UidAndClientId> getChiefOfGroupForLimitedAgencyReps(Collection<UidAndClientId> limiterReps) {
        Map<Long, Long> mainToChief = new HashMap<>(limiterReps.size());
        Map<Long, UidAndClientId> uids = listToMap(limiterReps, UidAndClientId::getUid);
        shardHelper.groupByShard(uids.keySet(), ShardKey.UID)
                .forEach((shard, chunk) -> mainToChief.putAll(getChiefOfGroupForLimitedAgencyReps(shard, chunk)));
        return EntryStream.of(mainToChief)
                .mapToValue((main, chief) -> UidAndClientId.of(chief, uids.get(main).getClientId()))
                .mapKeys(uids::get)
                .toMap();
    }

    private Map<Long, Long> getChiefOfGroupForLimitedAgencyReps(Integer shard, Collection<Long> uids) {
        if (uids.isEmpty()) {
            return emptyMap();
        }
        String mainTableAlias = "tmain";
        String chiefTableAlias = "tchief";
        return dslContextProvider.ppc(shard)
                .select(USERS_AGENCY.as(mainTableAlias).UID, USERS_AGENCY.as(chiefTableAlias).UID)
                .from(USERS_AGENCY.as(mainTableAlias))
                .join(USERS_AGENCY.as(chiefTableAlias))
                .on(USERS_AGENCY.as(chiefTableAlias).GROUP_ID.eq(USERS_AGENCY.as(mainTableAlias).GROUP_ID)
                        .and(USERS_AGENCY.as(chiefTableAlias).LIM_REP_TYPE.eq(UsersAgencyLimRepType.chief)))
                .where(USERS_AGENCY.as(mainTableAlias).UID.in(uids))
                .and(USERS_AGENCY.as(mainTableAlias).GROUP_ID.gt(0L))
                .fetchMap(USERS_AGENCY.as(mainTableAlias).UID, USERS_AGENCY.as(chiefTableAlias).UID);
    }

    public Set<ClientId> bindClientsToAgencyAndReps(long operatorUid, Collection<UidAndClientId> agencyReps, Set<ClientId> clientIds) {
        if (agencyReps.isEmpty()) {
            return emptySet();
        }
        ClientId agencyClientId = agencyReps.stream().findAny().map(UidAndClientId::getClientId).get();
        Map<Long, UserPerminfo> limRepsPerms = EntryStream.of(ppcRbac.getUsersPerminfo(mapList(agencyReps, UidAndClientId::getUid)))
                .filterValues(Optional::isPresent)
                .mapValues(Optional::get)
                .filterValues(p -> p.hasRepType(LIMITED))
                .toMap();
        if (limRepsPerms.isEmpty()) {
            return bindClientsToAgency(operatorUid, agencyReps, clientIds);
        }

        Collection<UidAndClientId> agencyChiefs = StreamEx.of(limRepsPerms.values())
                .map(perm -> UidAndClientId.of(perm.chiefUid(), agencyClientId))
                .collect(Collectors.toSet());
        Set<ClientId> clients = bindClientsToAgency(operatorUid, agencyChiefs, clientIds);
        Set<Long> limRepUids = limRepsPerms.keySet();
        if (!clients.isEmpty()) {
            for (ClientId clientId : clientIds) {
                try {
                    int shard = shardHelper.getShardByClientIdStrictly(clientId);
                    var insert = dslContextProvider.ppc(shard)
                            .insertInto(AGENCY_LIM_REP_CLIENTS)
                            .columns(AGENCY_LIM_REP_CLIENTS.CLIENT_ID, AGENCY_LIM_REP_CLIENTS.AGENCY_UID);
                    limRepUids.forEach(agencyUid -> insert.values(clientId.asLong(), agencyUid));
                    insert.onDuplicateKeyIgnore().execute();
                } catch (RuntimeException e) {
                    logger.error("Can't bind client (ClientId: %s) to agency reps (%s)", clientId, limRepsPerms, e);
                }
            }
        }
        ppcRbac.clearCaches();
        return clients;
    }

    /**
     * Привязать заданных клиентов к заданному агентству в RBAC-ке
     *
     * @return Множество Id-ков успешно привязанных клиентов
     */
    public Set<ClientId> bindClientsToAgency(long operatorUid, Collection<UidAndClientId> agencyReps, Set<ClientId> clientIds) {
        if (clientIds.isEmpty()) {
            return emptySet();
        }
        if (agencyReps.size() > 1) {
            logger.error("Привязка к нескольких представителям без фичи NEW_LIM_REP_SCHEMA");
            return emptySet();
        }
        UidAndClientId agency = agencyReps.stream().findAny().orElse(null);
        Set<ClientId> result = new HashSet<>(clientIds.size());

        for (ClientId clientId : clientIds) {
            try {
                int shard = shardHelper.getShardByClientIdStrictly(clientId);
                dslContextProvider.ppc(shard)
                        .update(CLIENTS)
                        .set(CLIENTS.AGENCY_CLIENT_ID, agency.getClientId().asLong())
                        .set(CLIENTS.AGENCY_UID, agency.getUid())
                        .set(CLIENTS.ROLE, ClientsRole.client)
                        .where(CLIENTS.CLIENT_ID.eq(clientId.asLong()))
                        .execute();
                result.add(clientId);
            } catch (RuntimeException e) {
                logger.error("Can't bind client (ClientId: %s) to agency (%s)", clientId, agency, e);
            }
        }

        ppcRbac.clearCaches();

        return result;
    }

    public boolean isOperatorManagerOfAgency(long operatorUid, long agencyUserId, long agencyClientId) {
        Set<Long> managerUids = ImmutableSet.of(operatorUid, ppcRbac.replaceManagerByTeamLead(operatorUid));
        int shard = shardHelper.getShardByClientIdStrictly(ClientId.fromLong(agencyClientId));
        int rows = dslContextProvider.ppc(shard)
                .selectOne()
                .from(CLIENTS)
                .where(CLIENTS.CLIENT_ID.eq(agencyClientId)
                        .and(CLIENTS.PRIMARY_MANAGER_UID.in(managerUids)))
                .execute();
        return rows > 0 || managerUids.stream().anyMatch(uid -> isOwner(uid, agencyUserId));
    }

    /**
     * Возвращает идентификаторы кампаний в которые можно импортировать xls
     * <p>
     * Аналог RBACDirect::rbac_get_allow_xls_import_camp_list, но список кампаний передается методу и нет проверки
     * типа кампаний
     */
    public List<Long> getCampaignsAllowImportXLS(Long operatorUid, Collection<Long> campaignIds) {
        Map<Long, RbacCampPerms> campaignsRights = getCampaignsRights(operatorUid, campaignIds);
        UserPerminfo operator = getUserPermInfo(operatorUid);

        return EntryStream.of(campaignsRights)
                .filterValues(RbacCampPerms::canWrite)
                .filter(x -> operator.role() != CLIENT
                        || operator.agencyClientId() == null
                        || operator.hasPerm(ClientPerm.XLS_IMPORT))
                .keys()
                .toList();
    }

    /**
     * Проверяет, может ли оператор импортировать из XLS новую кампанию клиенту
     * <p>
     * Аналог RBACDirect::rbac_can_import_xls_into_new_camp, но без параметра $agency_clientid
     */
    public boolean canImportXLSIntoNewCampaign(long operatorUid, long clientUid, ClientId clientId) {
        if (!isOwner(operatorUid, clientUid)) {
            return false;
        }

        return ppcRbac.getClientPerminfo(clientId)
                .map(p -> p.agencyClientId() == null || !getOwnedClients(operatorUid, p.agencyUids()).isEmpty() && p
                        .hasPerm(ClientPerm.XLS_IMPORT))
                .orElse(false);
    }

    /**
     * Проверяет, является ли этот клиент продуктом внутренней рекламы
     *
     * @return true, если есть и является
     * @throws IllegalArgumentException если нет такого клиента
     */
    public boolean isInternalAdProduct(ClientId clientId) {
        return ppcRbac.getClientPerminfo(clientId)
                .orElseThrow(() -> new IllegalArgumentException("no such client: " + clientId))
                .hasPerm(ClientPerm.INTERNAL_AD_PRODUCT);
    }

    /**
     * Из переданных clientId возвращает только те, соответствующие клиенты
     * которых являются продуктами внутренней рекламы
     *
     * @throws IllegalArgumentException если какого-то из этих клиентов не существует
     */
    public List<ClientId> filterInternalAdProducts(Collection<ClientId> clientIds) {
        Map<ClientId, Optional<ClientPerminfo>> clientsPerminfo = ppcRbac.getClientsPerminfo(clientIds);
        return clientIds.stream()
                .filter(clientId -> {
                    ClientPerminfo perminfo = clientsPerminfo.get(clientId)
                            .orElseThrow(() -> new IllegalArgumentException("no such client: " + clientId));
                    return perminfo.hasPerm(ClientPerm.INTERNAL_AD_PRODUCT);
                })
                .collect(toList());
    }

    /**
     * Возвращает базовую информацию о правах пользователя по uid
     */
    @Nonnull
    public UserPerminfo getUserPermInfo(long uid) throws IllegalArgumentException {
        return ppcRbac.getUserPerminfo(uid).orElseThrow(() -> new IllegalArgumentException("UserPerminfo not found"));
    }

    /**
     * Возвращает флаг, показывающий, что оператор является MCC для заданного клиента
     */
    @Nonnull
    public Boolean isOperatorMccForClient(Long uid, Long clientId) throws IllegalArgumentException {
        var operatorPerminfo = ppcRbac.getUserPerminfo(uid)
                .orElseThrow(() -> new IllegalArgumentException("UserPerminfo not found"));
        return isOperatorMccForClient(operatorPerminfo, clientId);
    }

    /**
     * Возвращает флаг, показывающий, что оператор является MCC для заданного клиента
     */
    @Nonnull
    public Boolean isOperatorMccForClient(UserPerminfo operatorPerminfo, Long clientId) {
        return operatorPerminfo.isMccForClient(clientId);
    }

    /**
     * Возвращает флаг, показывающий, что пользователь является управляющим аккаунтом MCC
     */
    @Nonnull
    public Boolean isUserMccControlClient(Long uid) {
        return getUserPermInfo(uid).isMcc();
    }

    /**
     * Очистка кеша прав, имеет смысл вызывать после изменения прав/ролей
     */
    public void clearCaches() {
        ppcRbac.clearCaches();
    }

    /**
     * Возвращает признак того, что клиент с clientId сопровождается фрилансером freelancerClientId
     *
     * @param freelancerClientId - id фрилансера
     * @param clientId           - id клиента
     * @return - true, если клиента сопровождается фрилансером
     */
    public boolean isRelatedClient(ClientId freelancerClientId, ClientId clientId) {
        return rbacClientsRelations.isRelatedClient(freelancerClientId, clientId);
    }

}
