package ru.yandex.direct.rbac.cache;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import com.google.common.cache.Cache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationListener;
import org.springframework.web.context.support.ServletRequestHandledEvent;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;

/**
 * Реализация кеша для rbac:
 * - кеш независимый в рамках каждого потока
 * - кеш сбрасывается по окончанию обработки http-запроса
 *
 * @param <K>
 * @param <V>
 */
public class RbacGuavaCache<K, V> implements RbacCache<K, V>, ApplicationListener<ServletRequestHandledEvent> {
    private static final Logger logger = LoggerFactory.getLogger(RbacGuavaCache.class);

    private final ThreadLocal<Cache<K, V>> tls;

    public RbacGuavaCache(Supplier<Cache<K, V>> cacheSupplier) {
        this.tls = ThreadLocal.withInitial(cacheSupplier);
    }

    @Override
    public Map<K, V> computeIfAbsent(Collection<K> keys, CacheMassLoader<K, V> loader) {
        Cache<K, V> cache = tls.get();
        Map<K, V> result = new HashMap<>(cache.getAllPresent(keys));

        List<K> absentKeys = filterList(keys, k -> !result.containsKey(k));
        if (!absentKeys.isEmpty()) {
            Map<K, V> newValues = loader.get(absentKeys);

            cache.putAll(newValues);
            result.putAll(newValues);
        }

        return result;
    }

    @Override
    public void clear() {
        tls.remove();
    }

    @Override
    public void onApplicationEvent(ServletRequestHandledEvent event) {
        logger.trace("clear cache on request end");
        clear();
    }
}
