package ru.yandex.direct.rbac.model;

import java.util.Arrays;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.dbschema.ppc.enums.ClientsRelationsType;
import ru.yandex.direct.dbschema.ppc.enums.ReverseClientsRelationsType;

import static ru.yandex.direct.dbschema.ppc.enums.ClientsRelationsType.freelancer;
import static ru.yandex.direct.dbschema.ppc.enums.ClientsRelationsType.internal_ad_publisher;
import static ru.yandex.direct.dbschema.ppc.enums.ClientsRelationsType.internal_ad_reader;
import static ru.yandex.direct.dbschema.ppc.enums.ClientsRelationsType.mcc;
import static ru.yandex.direct.dbschema.ppc.enums.ClientsRelationsType.mobile_goals_access;
import static ru.yandex.direct.dbschema.ppc.enums.ClientsRelationsType.support_for_client;

@ParametersAreNonnullByDefault
public enum ClientsRelationType {
    FREELANCER(freelancer),
    INTERNAL_AD_PUBLISHER(internal_ad_publisher),
    INTERNAL_AD_READER(internal_ad_reader),
    SUPPORT_FOR_CLIENT(support_for_client),
    MCC(mcc, ReverseClientsRelationsType.mcc),
    MOBILE_GOALS_ACCESS(mobile_goals_access, ReverseClientsRelationsType.mobile_goals_access),
    ;

    private static final Map<ClientsRelationsType, ClientsRelationType> DB_TYPE_MAPPING =
            Arrays.stream(ClientsRelationType.class.getEnumConstants())
                    .collect(Collectors.toMap(ClientsRelationType::getDbClientRelationType, Function.identity()));


    private final ClientsRelationsType dbClientRelationType;

    @Nullable
    private final ReverseClientsRelationsType dbReverseClientRelationType;

    ClientsRelationType(ClientsRelationsType dbClientRelationType) {
        this(dbClientRelationType, null);
    }

    ClientsRelationType(ClientsRelationsType dbClientRelationType,
                        @Nullable ReverseClientsRelationsType dbReverseClientRelationType) {
        this.dbClientRelationType = dbClientRelationType;
        this.dbReverseClientRelationType = dbReverseClientRelationType;
    }

    public ClientsRelationsType getDbClientRelationType() {
        return dbClientRelationType;
    }

    @Nullable
    public ReverseClientsRelationsType getDbReverseClientRelationType() {
        return dbReverseClientRelationType;
    }

    public boolean hasReverseRelation() {
        return dbReverseClientRelationType != null;
    }

    @Nullable
    public static ClientsRelationType fromSource(@Nullable ClientsRelationsType value) {
        if (value == null) {
            return null;
        }

        ClientsRelationType clientsRelationType = DB_TYPE_MAPPING.get(value);
        if (clientsRelationType == null) {
            throw new IllegalStateException("No such value: " + value);
        }
        return clientsRelationType;
    }

    @Nullable
    public static ClientsRelationsType toSource(@Nullable ClientsRelationType value) {
        if (value == null) {
            return null;
        }
        return value.dbClientRelationType;
    }
}
