package ru.yandex.direct.screenshooter.client.service;

import java.util.Map;
import java.util.UUID;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.avatars.client.AvatarsClient;
import ru.yandex.direct.avatars.client.model.AvatarInfo;
import ru.yandex.direct.avatars.client.model.answer.ImageSize;
import ru.yandex.direct.rotor.client.RotorClient;
import ru.yandex.direct.rotor.client.model.RotorImageResponse;
import ru.yandex.direct.screenshooter.client.model.ScreenShooterScreenshot;
import ru.yandex.direct.screenshooter.client.model.ScreenShooterSizeInfo;
import ru.yandex.direct.screenshooter.client.model.ScreenShooterSizesInfo;

@Service
@ParametersAreNonnullByDefault
public class ScreenShooterClient {
    private static final Logger logger = LoggerFactory.getLogger(ScreenShooterClient.class);

    private final RotorClient rotorClient;
    private final AvatarsClient avatarsClient;

    public ScreenShooterClient(RotorClient rotorClient, AvatarsClient avatarsClient) {
        this.rotorClient = rotorClient;
        this.avatarsClient = avatarsClient;
    }

    public ScreenShooterScreenshot getScreenshotFromUrl(final String url, final long width, final long height) {
        logger.info("Start making screenshot from url \"{}\"", url);
        RotorImageResponse image = rotorClient.getScreenshotFromUrl(url, width, height);
        return uploadImage(image);
    }

    public ScreenShooterScreenshot getScreenshotFromHtml(final String html, final long width, final long height) {
        logger.info("Start making screenshot from html \"{}\"", html);
        RotorImageResponse image = rotorClient.getScreenshotFromHtml(html, width, height);
        return uploadImage(image);
    }

    private ScreenShooterScreenshot uploadImage(@Nullable RotorImageResponse image) {
        ScreenShooterScreenshot screenshot = new ScreenShooterScreenshot();
        if (image != null && image.getImage() != null) {
            logger.info("Uploading screenshot to avatars");
            AvatarInfo avatarsImageInfo = avatarsClient.upload(UUID.randomUUID().toString(), image.getImage());

            ScreenShooterSizesInfo sizes = new ScreenShooterSizesInfo()
                    .withOrig(convertSizeToScreenshot(avatarsImageInfo.getSizes(), "orig"))
                    .withScreenshot(convertSizeToScreenshot(avatarsImageInfo.getSizes(), "screenshot"))
                    .withOptimize(convertSizeToScreenshot(avatarsImageInfo.getSizes(), "optimize"));

            screenshot
                    .withUrl(sizes.getOrig().getUrl())
                    .withSizes(sizes)
                    .withIsDone(true)
                    .withImageData(image.getImage());

            logger.info("Screenshot is done");
        } else {
            logger.warn("Failed to make screenshot, can't get screenshot from rotor");
            screenshot.withIsDone(false);
        }
        return screenshot;
    }

    @Nullable
    private ScreenShooterSizeInfo convertSizeToScreenshot(Map<String, ImageSize> sizes, String sizeName) {
        if (!sizes.containsKey(sizeName)) {
            return null;
        }
        ImageSize size = sizes.get(sizeName);
        return new ScreenShooterSizeInfo()
                .withUrl(avatarsClient.getReadUrl(size.getPath()))
                .withWidth(size.getWidth())
                .withHeight(size.getHeight());
    }
}
