package ru.yandex.direct.http.smart.converter;

import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.ArrayUtils;

import ru.yandex.direct.http.smart.annotations.Json;
import ru.yandex.direct.http.smart.annotations.Proto;
import ru.yandex.direct.http.smart.annotations.ProtoAsJson;
import ru.yandex.direct.http.smart.annotations.ResponseHandler;
import ru.yandex.direct.http.smart.annotations.Xml;
import ru.yandex.direct.http.smart.reflection.ProtoReflectionUtils;
import ru.yandex.direct.http.smart.reflection.Utils;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class ResponseConverterFactory {
    private final Map<Class<? extends ResponseConverter>, ResponseConverter<?>> converterMap;

    private final Map<Class<?>, ResponseConverter> typeSpecificConverterMap;
    private final Map<Class<? extends ResponseChecker>, ResponseChecker> checkerMap;
    private static final ResponseChecker DEFAULT_RESPONSE_CHECKER = new DefaultResponseChecker();

    private ResponseConverterFactory(Map<Class<? extends ResponseConverter>, ResponseConverter<?>> converterMap,
                                     Map<Class<?>, ResponseConverter> typeSpecificConverterMap,
                                     Map<Class<? extends ResponseChecker>, ResponseChecker> checkerMap) {
        this.converterMap = converterMap;
        this.typeSpecificConverterMap = typeSpecificConverterMap;
        this.checkerMap = checkerMap;
    }

    @Nonnull
    public <R> ResponseConverter<R> getConverter(Annotation[] methodAnnotations, Type returnType) {
        Class<?> rawReturnType = Utils.getRawType(returnType);
        ResponseConverter<R> typeConverter = getTypeSpecificConverter(rawReturnType);
        if (typeConverter != null) {
            return typeConverter;
        }
        for (Annotation annotation : methodAnnotations) {
            if (annotation instanceof ResponseHandler) {
                Class<?> parserClass = ((ResponseHandler) annotation).parserClass();
                return getConverter(parserClass);
            }
            if (annotation instanceof Json) {
                return getConverter(JsonResponseConverter.class);
            }
            if (annotation instanceof Xml) {
                return getConverter(XmlResponseConverter.class);
            }
            if (annotation instanceof ProtoAsJson) {
                return getConverter(ProtoAsJsonResponseConverter.class);
            }
            if (annotation instanceof Proto) {
                var protoClass = ProtoReflectionUtils.getProtoClass(returnType);
                if (protoClass == null) {
                    throw new IllegalArgumentException("Cannot find proto class for type " + returnType);
                }
                //noinspection unchecked
                return (ResponseConverter<R>)
                        new ProtoResponseConverter(protoClass, Iterable.class.isAssignableFrom(rawReturnType));
            }
        }
        if (Void.class.isAssignableFrom(rawReturnType)) {
            return getConverter(VoidResponseConverter.class);
        }
        return getConverter(StringResponseConverter.class);
    }

    @Nonnull
    public ResponseChecker getChecker(Annotation[] methodAnnotations) {
        for (Annotation parameterAnnotation : methodAnnotations) {
            if (parameterAnnotation instanceof ResponseHandler) {
                ResponseHandler responseHandler = (ResponseHandler) parameterAnnotation;
                var checkerClass = responseHandler.checkerClass();
                var checker = checkerMap.get(checkerClass);
                if (checker == null) {
                    var registered = String.join(",", mapList(checkerMap.keySet(), Class::toString));
                    throw new IllegalArgumentException(String.format("Cannot find checker with class %s. " +
                            "Registered: %s", checkerClass, registered));
                }
                List<Integer> expectedCodes =
                        StreamEx.of(ArrayUtils.toObject(responseHandler.expectedCodes())).toList();
                //если указаны кастомные коды, то считаем что чекер дефолтный
                if (expectedCodes.size() == 1 && expectedCodes.get(0).equals(200)) {
                    return checker;
                } else {
                    return r -> r.hasResponseStatus() && expectedCodes.contains(r.getStatusCode());
                }
            }
        }
        return checkerMap.get(DefaultResponseChecker.class);
    }

    @Nullable
    private <R> ResponseConverter<R> getTypeSpecificConverter(Class<?> type) {
        //noinspection unchecked
        return typeSpecificConverterMap.get(type);
    }

    private <R> ResponseConverter<R> getConverter(Class<?> converterClass) {
        //noinspection unchecked
        return (ResponseConverter<R>) converterMap.get(converterClass);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private final Map<Class<? extends ResponseConverter>, ResponseConverter<?>> converterMap = new HashMap<>();
        private final Map<Class<?>, ResponseConverter> typeSpecificConverterMap = new HashMap<>();
        private final Map<Class<? extends ResponseChecker>, ResponseChecker> checkerMap = new HashMap<>();

        /**
         * Добавить новый конвертер. Он будет хранится в мапе по ключу clazz
         *
         * @param clazz     класс конвертера, по этому ключу можно его можно указывать в аннотации
         *                  {@link ResponseHandler}
         * @param converter конвертер
         * @param <R>       тип результата конвертации
         * @param <T>       тип конвертера
         */
        public <R, T extends ResponseConverter<R>> Builder addConverter(Class<T> clazz, T converter) {
            converterMap.put(clazz, converter);
            return this;
        }

        /**
         * Добавить новый конвертер. Он будет хранится в мапе по ключу {@link ResponseConverter#getClass()}
         * По этому же ключу можно его можно указывать в аннотации {@link ResponseHandler}
         *
         * @param converters Список конвертеров
         */
        public Builder addConverters(ResponseConverter<?>... converters) {
            Preconditions.checkArgument(
                    Arrays.stream(converters).map(t -> t.getClass()).distinct().count() == converters.length,
                    "Contervers must have unique classes");
            for (var converter : converters) {
                converterMap.put(converter.getClass(), converter);
            }
            return this;
        }

        @SafeVarargs
        public final <T extends ResponseChecker> Builder addCheckers(T... checkers) {
            Preconditions.checkArgument(
                    Arrays.stream(checkers).map(t -> t.getClass()).distinct().count() == checkers.length,
                    "Checkers must have unique classes");
            for (T checker : checkers) {
                checkerMap.put(checker.getClass(), checker);
            }
            return this;
        }

        public <T extends ResponseChecker> Builder addChecker(Class<T> clazz, T checker) {
            checkerMap.put(clazz, checker);
            return this;
        }

        /**
         * Добавить новый конвертер для конкретного типа ответа.
         * Если задан, имеет приоритет над общими конвертерами из
         * {@link Builder#addConverter(Class, ResponseConverter)} и {@link Builder#addConverters(ResponseConverter[])}
         *
         * @param clazz     класс, для которого необходимо использовать конвертер
         * @param converter конвертер для указанного типа
         * @param <T>       тип результата конвертации
         */
        public <T> Builder addTypeSpecificConverter(Class<T> clazz, ResponseConverter<T> converter) {
            typeSpecificConverterMap.put(clazz, converter);
            return this;
        }

        public ResponseConverterFactory build() {
            addConverters(
                    new JsonResponseConverter(),
                    new ProtoAsJsonResponseConverter(),
                    new XmlResponseConverter(),
                    new StringResponseConverter(),
                    new VoidResponseConverter()
            );
            addCheckers(DEFAULT_RESPONSE_CHECKER);

            return new ResponseConverterFactory(converterMap, typeSpecificConverterMap, checkerMap);
        }
    }
}
