package ru.yandex.direct.http.smart.core;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.asynchttpclient.Param;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableRequest;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceChild;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.InterruptedRuntimeException;

import static com.google.common.collect.Lists.partition;
import static ru.yandex.direct.solomon.SolomonUtils.getParallelFetcherMetricRegistry;
import static ru.yandex.direct.tracing.util.TraceUtil.X_YANDEX_TRACE;
import static ru.yandex.direct.tracing.util.TraceUtil.traceToHeader;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

/**
 * Аналог Calls. Позволяет распаралелить запрос по одному из query-параметров.
 * Такой параметр нужно помечать аннотацией @PartibleQuery вместо @Query
 */
public class CallsPack<R> extends AbstractCall<R> {
    protected List<ParsableRequest<R>> requests; //Кешируем запросы сгенерированные последним вызовом getRequests
    protected int lastChunkSize; //Значение chunkSize, которое передавалось при последнем вызове getRequests

    public CallsPack(ServiceMethod<R> serviceMethod, ParallelFetcherFactory parallelFetcherFactory,
                     SmartRequestBuilder requestBuilder) {
        super(serviceMethod, parallelFetcherFactory, requestBuilder);
    }

    public List<? extends ParsableRequest<R>> getRequests() {
        return getRequests(lastChunkSize);
    }

    /**
     * @param chunkSize ограничение на количество значений параметра в одном запросе
     * @return список запросов
     */
    public List<? extends ParsableRequest<R>> getRequests(int chunkSize) {
        if (requests == null || chunkSize != lastChunkSize) {
            lastChunkSize = chunkSize;

            if (chunkSize > 0 && !requestBuilder.partibleQueryParams.isEmpty()) {
                requests = new ArrayList<>();
                Long savedRequestId = requestBuilder.getRequestId();
                for (List<Param> params : partition(requestBuilder.partibleQueryParams, chunkSize)) {
                    requests.add(getRequest(params));
                    //Для того чтобы результаты не перетирались в Map с результатами
                    requestBuilder.setRequestId(requestBuilder.getRequestId() + 1);
                }
                requestBuilder.setRequestId(savedRequestId);
            } else if (!requestBuilder.partibleBodyParams.isEmpty()) {
                requests = new ArrayList<>();
                Long savedRequestId = requestBuilder.getRequestId();
                requestBuilder.getPartibleBodyParams().forEach(body -> {
                    requests.add(getRequest(body));
                    requestBuilder.setRequestId(requestBuilder.getRequestId() + 1);
                });
                requestBuilder.setRequestId(savedRequestId);
            } else {
                requests = List.of(getRequest(requestBuilder.partibleQueryParams));
            }
        }
        return requests;
    }

    private ParsableRequest<R> getRequest(byte[] body) {
        requestBuilder.setBody(body);
        return getRequest();
    }

    private ParsableRequest<R> getRequest(List<Param> additionalParams) {
        //Сохраняем значение исходных параметров
        List<Param> permanentParams = ifNotNull(requestBuilder.getQueryParams(), ArrayList::new);
        try {
            requestBuilder.addQueryParams(additionalParams);
            return getRequest();
        } finally {
            //Восстанавливаем исходные параметры
            requestBuilder.setQueryParams(permanentParams);
        }
    }

    /**
     * @param chunkSize ограничение на количество значений query параметра в одном запросе
     * @return результаты запросов
     */
    public Map<Long, Result<R>> execute(int chunkSize) {
        return execute(chunkSize, parallelFetcherFactory.defaultSettingsCopy());
    }

    /**
     * @param chunkSize        ограничение на количество значений query параметра в одном запросе
     * @param overrideParallel переопределяющее число параллельно выполняемых запросов
     * @return результаты запросов
     */
    public Map<Long, Result<R>> execute(int chunkSize, int overrideParallel) {
        FetcherSettings fetcherSettings = parallelFetcherFactory.defaultSettingsCopy()
                .withParallel(overrideParallel);
        return execute(chunkSize, fetcherSettings);
    }

    /**
     * @param chunkSize ограничение на количество значений query параметра в одном запросе
     * @return результаты запросов
     */
    private Map<Long, Result<R>> execute(int chunkSize, FetcherSettings fetcherSettings) {
        Map<Long, Result<R>> result;
        try (TraceProfile profile = Trace.current().profile(traceFunc, "smart_client");
             TraceChild child = Trace.current().child(serviceMethod.smart.profileName, serviceMethod.methodName)) {
            if (fetcherSettings.getMetricRegistry() == null) {
                fetcherSettings.withMetricRegistry(getParallelFetcherMetricRegistry(profile.getFunc()));
            }
            ParallelFetcher<R> fetcher = parallelFetcherFactory.getParallelFetcher(fetcherSettings);

            requestBuilder.addHeader(X_YANDEX_TRACE, traceToHeader(child));
            result = fetcher.execute(getRequests(chunkSize));
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
        return result;
    }
}
