package ru.yandex.direct.http.smart.core;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.net.URI;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.common.net.MediaType;
import io.netty.handler.codec.http.DefaultHttpHeaders;
import io.netty.handler.codec.http.HttpHeaders;
import org.apache.commons.lang3.ClassUtils;
import org.asynchttpclient.Response;
import org.asynchttpclient.request.body.multipart.Part;
import org.asynchttpclient.uri.Uri;

import ru.yandex.direct.http.smart.annotations.Id;
import ru.yandex.direct.http.smart.converter.PartConverter;
import ru.yandex.direct.http.smart.converter.RequestBodyConverter;
import ru.yandex.direct.http.smart.converter.ResponseChecker;
import ru.yandex.direct.http.smart.converter.ResponseConverter;
import ru.yandex.direct.http.smart.http.Body;
import ru.yandex.direct.http.smart.http.Cookie;
import ru.yandex.direct.http.smart.http.DELETE;
import ru.yandex.direct.http.smart.http.Field;
import ru.yandex.direct.http.smart.http.FieldMap;
import ru.yandex.direct.http.smart.http.FormUrlEncoded;
import ru.yandex.direct.http.smart.http.GET;
import ru.yandex.direct.http.smart.http.HEAD;
import ru.yandex.direct.http.smart.http.HTTP;
import ru.yandex.direct.http.smart.http.Header;
import ru.yandex.direct.http.smart.http.HeaderMap;
import ru.yandex.direct.http.smart.http.Headers;
import ru.yandex.direct.http.smart.http.Multipart;
import ru.yandex.direct.http.smart.http.OPTIONS;
import ru.yandex.direct.http.smart.http.PATCH;
import ru.yandex.direct.http.smart.http.POST;
import ru.yandex.direct.http.smart.http.PUT;
import ru.yandex.direct.http.smart.http.PartMap;
import ru.yandex.direct.http.smart.http.PartibleBody;
import ru.yandex.direct.http.smart.http.PartibleQuery;
import ru.yandex.direct.http.smart.http.Path;
import ru.yandex.direct.http.smart.http.Query;
import ru.yandex.direct.http.smart.http.QueryMap;
import ru.yandex.direct.http.smart.http.QueryName;
import ru.yandex.direct.http.smart.http.Url;
import ru.yandex.direct.http.smart.parameters.ArrayParameterHandler;
import ru.yandex.direct.http.smart.parameters.BodyParameterHandler;
import ru.yandex.direct.http.smart.parameters.CookieParameterHandler;
import ru.yandex.direct.http.smart.parameters.FieldMapParameterHandler;
import ru.yandex.direct.http.smart.parameters.FieldParameterHandler;
import ru.yandex.direct.http.smart.parameters.HeaderMapParameterHandler;
import ru.yandex.direct.http.smart.parameters.HeaderParameterHandler;
import ru.yandex.direct.http.smart.parameters.IdParameterHandler;
import ru.yandex.direct.http.smart.parameters.IterableParameterHandler;
import ru.yandex.direct.http.smart.parameters.ParameterHandler;
import ru.yandex.direct.http.smart.parameters.PartMapParameterHandler;
import ru.yandex.direct.http.smart.parameters.PartParameterHandler;
import ru.yandex.direct.http.smart.parameters.PartRawParameterHandler;
import ru.yandex.direct.http.smart.parameters.PartibleBodyParameterHandler;
import ru.yandex.direct.http.smart.parameters.PartibleQueryParameterHandler;
import ru.yandex.direct.http.smart.parameters.PathParameterHandler;
import ru.yandex.direct.http.smart.parameters.QueryMapParameterHandler;
import ru.yandex.direct.http.smart.parameters.QueryNameParameterHandler;
import ru.yandex.direct.http.smart.parameters.QueryParameterHandler;
import ru.yandex.direct.http.smart.parameters.RelativeUrlParameterHandler;
import ru.yandex.direct.http.smart.reflection.Utils;


final class ServiceMethodBuilder<R> {
    private static final String PARAM = "[a-zA-Z][a-zA-Z0-9_-]*";
    private static final Pattern PARAM_URL_REGEX = Pattern.compile("\\{(" + PARAM + ")\\}");
    private static final Pattern PARAM_NAME_REGEX = Pattern.compile(PARAM);

    final Smart smart;

    final Method method;

    private final Annotation[] methodAnnotations;
    private final Annotation[][] parameterAnnotationsArray;
    private final Type[] parameterTypes;

    private boolean gotField;
    private boolean gotPart;
    private boolean gotBody;
    private boolean gotPath;
    private boolean gotQuery;
    private boolean gotUrl;
    private boolean hasBody;
    private boolean isFormEncoded;
    private boolean isMultipart;
    private boolean hasId;
    private boolean hasPartibleParam;

    private Set<String> relativeUrlParamNames;

    Type responseType;
    String httpMethod;
    String relativeUrl;
    HttpHeaders headers = new DefaultHttpHeaders();
    ResponseConverter<R> responseConverter;
    ResponseChecker responseChecker;

    ParameterHandler<?>[] parameterHandlers;

    List<ParserPredicate> parserPredicates = new ArrayList<>();

    ServiceMethodBuilder(Smart smart, Method method) {
        this.smart = smart;

        this.method = method;
        this.methodAnnotations = method.getAnnotations();
        this.parameterTypes = method.getGenericParameterTypes();
        this.parameterAnnotationsArray = method.getParameterAnnotations();
        initParserList();
    }

    public ServiceMethod build() {
        responseType = Utils.getCallResponseType(method.getGenericReturnType());
        if (responseType.equals(Response.class)) {
            throw methodError("'"
                    + Utils.getRawType(responseType).getName()
                    + "' is not a valid response body type.");
        }

        for (Annotation annotation : methodAnnotations) {
            parseMethodAnnotation(annotation);
        }

        if (httpMethod == null) {
            throw methodError("HTTP method annotation is required (e.g., @GET, @POST, etc.).");
        }

        if (!hasBody) {
            if (isMultipart) {
                throw methodError(
                        "Multipart can only be specified on HTTP methods with request body (e.g., @POST).");
            }
            if (isFormEncoded) {
                throw methodError("FormUrlEncoded can only be specified on HTTP methods with "
                        + "request body (e.g., @POST).");
            }
        }

        int parameterCount = parameterAnnotationsArray.length;
        parameterHandlers = new ParameterHandler<?>[parameterCount];
        for (int p = 0; p < parameterCount; p++) {
            Type parameterType = parameterTypes[p];
            if (Utils.hasUnresolvableType(parameterType)) {
                throw parameterError(p, "Parameter type must not include a type variable or wildcard: %s",
                        parameterType);
            }

            Annotation[] parameterAnnotations = parameterAnnotationsArray[p];

            parameterHandlers[p] = parseParameter(p, parameterType, parameterAnnotations);
        }

        if (relativeUrl == null && !gotUrl) {
            throw methodError("Missing either @%s URL or @Url parameter.", httpMethod);
        }
        if (!isFormEncoded && !isMultipart && !hasBody && gotBody) {
            throw methodError("Non-body HTTP method cannot contain @Body.");
        }
        if (isFormEncoded && !gotField) {
            throw methodError("Form-encoded method must contain at least one @Field.");
        }
        if (isMultipart && !gotPart) {
            throw methodError("Multipart method must contain at least one @Part.");
        }
        responseConverter = smart.responseConverterFactory.getConverter(methodAnnotations, responseType);
        responseChecker = smart.responseConverterFactory.getChecker(methodAnnotations);

        return new ServiceMethod<>(this);
    }

    private void parseMethodAnnotation(Annotation annotation) {
        if (annotation instanceof DELETE) {
            parseHttpMethodAndPath("DELETE", ((DELETE) annotation).value(), false);
        } else if (annotation instanceof GET) {
            parseHttpMethodAndPath("GET", ((GET) annotation).value(), false);
        } else if (annotation instanceof HEAD) {
            parseHttpMethodAndPath("HEAD", ((HEAD) annotation).value(), false);
            if (!Void.class.equals(responseType)) {
                throw methodError("HEAD method must use Void as response type.");
            }
        } else if (annotation instanceof PATCH) {
            parseHttpMethodAndPath("PATCH", ((PATCH) annotation).value(), true);
        } else if (annotation instanceof POST) {
            parseHttpMethodAndPath("POST", ((POST) annotation).value(), true);
        } else if (annotation instanceof PUT) {
            parseHttpMethodAndPath("PUT", ((PUT) annotation).value(), true);
        } else if (annotation instanceof OPTIONS) {
            parseHttpMethodAndPath("OPTIONS", ((OPTIONS) annotation).value(), false);
        } else if (annotation instanceof HTTP) {
            HTTP http = (HTTP) annotation;
            parseHttpMethodAndPath(http.method(), http.path(), http.hasBody());
        } else if (annotation instanceof Headers) {
            String[] headersToParse = ((Headers) annotation).value();
            if (headersToParse.length == 0) {
                throw methodError("@Headers annotation is empty.");
            }
            headers = parseHeaders(headersToParse);
        } else if (annotation instanceof Multipart) {
            if (isFormEncoded) {
                throw methodError("Only one encoding annotation is allowed.");
            }
            isMultipart = true;
        } else if (annotation instanceof FormUrlEncoded) {
            if (isMultipart) {
                throw methodError("Only one encoding annotation is allowed.");
            }
            isFormEncoded = true;
        }
    }

    private void parseHttpMethodAndPath(String httpMethod, String value, boolean hasBody) {
        if (this.httpMethod != null) {
            throw methodError("Only one HTTP method is allowed. Found: %s and %s.",
                    this.httpMethod, httpMethod);
        }
        this.httpMethod = httpMethod;
        this.hasBody = hasBody;

        if (value.isEmpty()) {
            return;
        }

        int question = value.indexOf('?');
        if (question != -1 && question < value.length() - 1) {
            String queryParams = value.substring(question + 1);
            Matcher queryParamMatcher = PARAM_URL_REGEX.matcher(queryParams);
            if (queryParamMatcher.find()) {
                throw methodError("URL query string \"%s\" must not have replace block. "
                        + "For dynamic query parameters use @Query.", queryParams);
            }
        }

        this.relativeUrl = value;
        this.relativeUrlParamNames = parsePathParameters(value);
    }

    private HttpHeaders parseHeaders(String[] headersArray) {
        DefaultHttpHeaders headers = new DefaultHttpHeaders();

        for (String header : headersArray) {
            int colon = header.indexOf(':');
            if (colon == -1 || colon == 0 || colon == header.length() - 1) {
                throw methodError(
                        "@Headers value must be in the form \"Name: Value\". Found: \"%s\"", header);
            }
            String headerName = header.substring(0, colon);
            String headerValue = header.substring(colon + 1).trim();
            if ("Content-Type".equalsIgnoreCase(headerName)) {
                try {
                    MediaType.parse(headerValue);
                } catch (IllegalArgumentException e) {
                    throw methodError("Malformed content type: %s", headerValue);
                }
            }
            headers.add(headerName, headerValue);
        }
        return headers;
    }

    private ParameterHandler<?> parseParameter(int p, Type parameterType, Annotation[] annotations) {
        ParameterHandler<?> result = null;
        for (Annotation annotation : annotations) {
            AnnotationContainer container = new AnnotationContainer(p, parameterType, annotations, annotation);
            ParameterHandler<?> annotationAction = parserPredicates.stream()
                    .filter(l -> l.predicate.test(container))
                    .findFirst()
                    .map(l -> l.function.apply(container))
                    .orElse(null);
            if (annotationAction == null) {
                continue;
            }

            if (result != null) {
                throw parameterError(p, "Multiple Smart annotations found, only one allowed.");
            }

            result = annotationAction;
        }

        if (result == null) {
            throw parameterError(p, "No Smart annotation found.");
        }

        return result;
    }

    private void initParserList() {
        addParser(l -> l.annotation instanceof Url, this::parseUrlAnnotation);
        addParser(l -> l.annotation instanceof Path, this::parsePathAnnotation);
        addParser(l -> l.annotation instanceof Query, this::parseQueryAnnotation);
        addParser(l -> l.annotation instanceof PartibleQuery, this::parsePartibleQueryAnnotation);
        addParser(l -> l.annotation instanceof QueryName, this::parseQueryNameAnnotation);
        addParser(l -> l.annotation instanceof QueryMap, this::parseQueryMapAnnotation);
        addParser(l -> l.annotation instanceof Header, this::parseHeaderAnnotation);
        addParser(l -> l.annotation instanceof HeaderMap, this::parseHeaderMapAnnotation);
        addParser(l -> l.annotation instanceof Field, this::parseFieldAnnotation);
        addParser(l -> l.annotation instanceof FieldMap, this::parseFieldMapAnnotation);
        addParser(l -> l.annotation instanceof ru.yandex.direct.http.smart.http.Part, this::parsePartAnnotation);
        addParser(l -> l.annotation instanceof PartMap, this::parsePartMapAnnotation);
        addParser(l -> l.annotation instanceof Body, this::parseBodyAnnotation);
        addParser(l -> l.annotation instanceof PartibleBody, this::parsePartibleBodyAnnotation);
        addParser(l -> l.annotation instanceof Id, this::parseIdAnnotation);
        addParser(l -> l.annotation instanceof Cookie, this::parseCookieAnnotation);
    }

    private void addParser(Predicate<AnnotationContainer> predicate,
                           Function<AnnotationContainer, ParameterHandler<?>> function) {
        parserPredicates.add(new ParserPredicate(predicate, function));
    }

    private ParameterHandler<?> parseUrlAnnotation(AnnotationContainer container) {
        if (gotUrl) {
            throw parameterError(container.parameterIndex, "Multiple @Url method annotations found.");
        }
        if (gotPath) {
            throw parameterError(container.parameterIndex, "@Path parameters may not be used with @Url.");
        }
        if (gotQuery) {
            throw parameterError(container.parameterIndex, "A @Url parameter must not come after a @Query");
        }
        if (relativeUrl != null) {
            throw parameterError(container.parameterIndex, "@Url cannot be used with @%s URL", httpMethod);
        }

        gotUrl = true;

        if (container.parameterType.equals(Uri.class)
                || container.parameterType.equals(String.class)
                || container.parameterType.equals(URI.class)) {
            return new RelativeUrlParameterHandler();
        } else {
            throw parameterError(container.parameterIndex,
                    "@Url must be org.asynchttpclient.uri.Uri, String or java.net.URI type.");
        }
    }

    private ParameterHandler<?> parsePathAnnotation(AnnotationContainer container) {
        if (gotQuery) {
            throw parameterError(container.parameterIndex, "A @Path parameter must not come after a @Query.");
        }
        if (gotUrl) {
            throw parameterError(container.parameterIndex, "@Path parameters may not be used with @Url.");
        }
        if (relativeUrl == null) {
            throw parameterError(container.parameterIndex, "@Path can only be used with relative url on @%s",
                    httpMethod);
        }
        gotPath = true;

        Path path = (Path) container.annotation;
        String name = path.value();
        validatePathName(container.parameterIndex, name);

        RequestBodyConverter<?> converter = requestConverter(container.annotations, smart);
        return new PathParameterHandler<>(name, converter);
    }

    private ParameterHandler<?> parseQueryAnnotation(AnnotationContainer container) {
        Query query = (Query) container.annotation;
        String name = query.value();

        Class<?> rawParameterType = Utils.getRawType(container.parameterType);
        gotQuery = true;
        RequestBodyConverter<?> converter = requestConverter(container.annotations, smart);
        if (Iterable.class.isAssignableFrom(rawParameterType)) {
            if (!(container.parameterType instanceof ParameterizedType)) {
                throw parameterError(container.parameterIndex, rawParameterType.getSimpleName()
                        + " must include generic type (e.g., "
                        + rawParameterType.getSimpleName()
                        + "<String>)");
            }
            return new IterableParameterHandler<>(new QueryParameterHandler<>(name, converter));
        } else if (rawParameterType.isArray()) {
            return new ArrayParameterHandler<>(new QueryParameterHandler<>(name, converter));
        } else {
            return new QueryParameterHandler<>(name, converter);
        }

    }

    private ParameterHandler<?> parsePartibleQueryAnnotation(AnnotationContainer container) {
        if (hasPartibleParam) {
            throw parameterError(container.parameterIndex, "Only one partible parameter allowed. (@PartibleBody " +
                    "or @PartibleQuery)");
        }

        PartibleQuery query = (PartibleQuery) container.annotation;
        String name = query.value();

        Class<?> rawParameterType = Utils.getRawType(container.parameterType);
        gotQuery = true;
        hasPartibleParam = true;
        RequestBodyConverter<?> converter = requestConverter(container.annotations, smart);
        if (Iterable.class.isAssignableFrom(rawParameterType)) {
            if (!(container.parameterType instanceof ParameterizedType)) {
                throw parameterError(container.parameterIndex, rawParameterType.getSimpleName()
                        + " must include generic type (e.g., "
                        + rawParameterType.getSimpleName()
                        + "<String>)");
            }
            return new IterableParameterHandler<>(new PartibleQueryParameterHandler<>(name, converter));
        } else if (rawParameterType.isArray()) {
            return new ArrayParameterHandler<>(new PartibleQueryParameterHandler<>(name, converter));
        } else {
            return new PartibleQueryParameterHandler<>(name, converter);
        }
    }

    private ParameterHandler<?> parseQueryNameAnnotation(AnnotationContainer container) {
        Class<?> rawParameterType = Utils.getRawType(container.parameterType);
        gotQuery = true;
        RequestBodyConverter<?> converter = requestConverter(container.annotations, smart);
        if (Iterable.class.isAssignableFrom(rawParameterType)) {
            if (!(container.parameterType instanceof ParameterizedType)) {
                throw parameterError(container.parameterIndex, rawParameterType.getSimpleName()
                        + " must include generic type (e.g., "
                        + rawParameterType.getSimpleName()
                        + "<String>)");
            }
            return new IterableParameterHandler<>(new QueryNameParameterHandler<>(converter));
        } else if (rawParameterType.isArray()) {

            return new ArrayParameterHandler<>(new QueryNameParameterHandler<>(converter));
        } else {
            return new QueryNameParameterHandler<>(converter);
        }
    }

    private ParameterHandler<?> parseQueryMapAnnotation(AnnotationContainer container) {
        Class<?> rawParameterType = Utils.getRawType(container.parameterType);
        if (!Map.class.isAssignableFrom(rawParameterType)) {
            throw parameterError(container.parameterIndex, "@QueryMap parameter type must be Map.");
        }
        Type mapType = Utils.getSupertype(container.parameterType, rawParameterType, Map.class);
        if (!(mapType instanceof ParameterizedType)) {
            throw parameterError(container.parameterIndex, "Map must include generic types (e.g., Map<String, " +
                    "String>)");
        }
        ParameterizedType parameterizedType = (ParameterizedType) mapType;
        Type keyType = Utils.getParameterUpperBound(0, parameterizedType);
        if (!String.class.equals(keyType)) {
            throw parameterError(container.parameterIndex, "@QueryMap keys must be of type String: " + keyType);
        }
        RequestBodyConverter<?> valueConverter = requestConverter(container.annotations, smart);
        return new QueryMapParameterHandler<>(valueConverter);
    }

    private ParameterHandler<?> parseHeaderAnnotation(AnnotationContainer container) {
        Header header = (Header) container.annotation;
        String name = header.value();

        Class<?> rawParameterType = Utils.getRawType(container.parameterType);
        RequestBodyConverter<?> converter = requestConverter(container.annotations, smart);
        if (Iterable.class.isAssignableFrom(rawParameterType)) {
            if (!(container.parameterType instanceof ParameterizedType)) {
                throw parameterError(container.parameterIndex, rawParameterType.getSimpleName()
                        + " must include generic type (e.g., "
                        + rawParameterType.getSimpleName()
                        + "<String>)");
            }
            return new IterableParameterHandler<>(new HeaderParameterHandler<>(name, converter));
        } else if (rawParameterType.isArray()) {
            return new ArrayParameterHandler<>(new HeaderParameterHandler<>(name, converter));
        } else {
            return new HeaderParameterHandler<>(name, converter);
        }
    }

    private ParameterHandler<?> parseHeaderMapAnnotation(AnnotationContainer container) {
        Class<?> rawParameterType = Utils.getRawType(container.parameterType);
        if (!Map.class.isAssignableFrom(rawParameterType)) {
            throw parameterError(container.parameterIndex, "@HeaderMap parameter type must be Map.");
        }

        Type mapType = Utils.getSupertype(container.parameterType, rawParameterType, Map.class);
        if (!(mapType instanceof ParameterizedType)) {
            throw parameterError(container.parameterIndex, "Map must include generic types (e.g., Map<String, " +
                    "String>)");
        }
        ParameterizedType parameterizedType = (ParameterizedType) mapType;
        Type keyType = Utils.getParameterUpperBound(0, parameterizedType);
        if (!String.class.equals(keyType)) {
            throw parameterError(container.parameterIndex, "@HeaderMap keys must be of type String: " + keyType);
        }
        RequestBodyConverter<?> valueConverter = requestConverter(container.annotations, smart);
        return new HeaderMapParameterHandler<>(valueConverter);
    }

    private ParameterHandler<?> parseFieldAnnotation(AnnotationContainer container) {
        if (!isFormEncoded) {
            throw parameterError(container.parameterIndex, "@Field parameters can only be used with form encoding" +
                    ".");
        }
        Field field = (Field) container.annotation;
        String name = field.value();

        gotField = true;

        Class<?> rawParameterType = Utils.getRawType(container.parameterType);
        RequestBodyConverter<?> converter = requestConverter(container.annotations, smart);
        if (Iterable.class.isAssignableFrom(rawParameterType)) {
            if (!(container.parameterType instanceof ParameterizedType)) {
                throw parameterError(container.parameterIndex, rawParameterType.getSimpleName()
                        + " must include generic type (e.g., "
                        + rawParameterType.getSimpleName()
                        + "<String>)");
            }

            return new IterableParameterHandler<>(new FieldParameterHandler<>(name, converter));
        } else if (rawParameterType.isArray()) {
            return new ArrayParameterHandler<>(new FieldParameterHandler<>(name, converter));
        } else {
            return new FieldParameterHandler<>(name, converter);
        }
    }

    private ParameterHandler<?> parseFieldMapAnnotation(AnnotationContainer container) {
        if (!isFormEncoded) {
            throw parameterError(container.parameterIndex, "@FieldMap parameters can only be used with form " +
                    "encoding.");
        }
        Class<?> rawParameterType = Utils.getRawType(container.parameterType);
        if (!Map.class.isAssignableFrom(rawParameterType)) {
            throw parameterError(container.parameterIndex, "@FieldMap parameter type must be Map.");
        }
        Type mapType = Utils.getSupertype(container.parameterType, rawParameterType, Map.class);
        if (!(mapType instanceof ParameterizedType)) {
            throw parameterError(container.parameterIndex,
                    "Map must include generic types (e.g., Map<String, String>)");
        }
        ParameterizedType parameterizedType = (ParameterizedType) mapType;
        Type keyType = Utils.getParameterUpperBound(0, parameterizedType);
        if (!String.class.equals(keyType)) {
            throw parameterError(container.parameterIndex, "@FieldMap keys must be of type String: " + keyType);
        }
        RequestBodyConverter<?> valueConverter = requestConverter(container.annotations, smart);

        gotField = true;
        return new FieldMapParameterHandler<>(valueConverter);
    }

    private ParameterHandler<?> parsePartAnnotation(AnnotationContainer container) {
        if (!isMultipart) {
            throw parameterError(container.parameterIndex, "@Part parameters can only be used with multipart " +
                    "encoding.");
        }
        ru.yandex.direct.http.smart.http.Part part = (ru.yandex.direct.http.smart.http.Part) container.annotation;
        gotPart = true;

        String partName = part.value();
        Class<?> rawParameterType = Utils.getRawType(container.parameterType);
        if (partName.isEmpty()) {
            if (Iterable.class.isAssignableFrom(rawParameterType)) {
                if (!(container.parameterType instanceof ParameterizedType)) {
                    throw parameterError(container.parameterIndex, rawParameterType.getSimpleName()
                            + " must include generic type (e.g., "
                            + rawParameterType.getSimpleName()
                            + "<String>)");
                }
                ParameterizedType parameterizedType = (ParameterizedType) container.parameterType;
                Type iterableType = Utils.getParameterUpperBound(0, parameterizedType);
                if (!Part.class.isAssignableFrom(Utils.getRawType(iterableType))) {
                    throw parameterError(container.parameterIndex,
                            "@Part annotation must supply a name or use org.asynchttpclient.request.body" +
                                    ".multipart.Part parameter type.");
                }
                return new IterableParameterHandler<>(PartRawParameterHandler.INSTANCE);
            } else if (rawParameterType.isArray()) {
                Class<?> arrayComponentType = rawParameterType.getComponentType();
                if (!Part.class.isAssignableFrom(arrayComponentType)) {
                    throw parameterError(container.parameterIndex,
                            "@Part annotation must supply a name or use org.asynchttpclient.request.body" +
                                    ".multipart.Part parameter type.");
                }
                return new ArrayParameterHandler<>(PartRawParameterHandler.INSTANCE);
            } else if (Part.class.isAssignableFrom(rawParameterType)) {
                return PartRawParameterHandler.INSTANCE;
            } else {
                throw parameterError(container.parameterIndex,
                        "@Part annotation must supply a name or use org.asynchttpclient.request.body.multipart" +
                                ".Part parameter type.");
            }
        } else {
            PartConverter<?> partConverter = partConverter(container.annotations, smart, partName);
            if (Iterable.class.isAssignableFrom(rawParameterType)) {

                if (!(container.parameterType instanceof ParameterizedType)) {
                    throw parameterError(container.parameterIndex, rawParameterType.getSimpleName()
                            + " must include generic type (e.g., "
                            + rawParameterType.getSimpleName()
                            + "<String>)");
                }
                ParameterizedType parameterizedType = (ParameterizedType) container.parameterType;
                Type iterableType = Utils.getParameterUpperBound(0, parameterizedType);
                if (Part.class.isAssignableFrom(Utils.getRawType(iterableType))) {
                    throw parameterError(container.parameterIndex,
                            "@Part parameters using the org.asynchttpclient.request.body.multipart.Part must not "
                                    + "include a part name in the annotation.");
                }

                return new IterableParameterHandler<>(new PartParameterHandler<>(partConverter, part.encoding()));
            } else if (rawParameterType.isArray()) {
                Class<?> arrayComponentType =
                        ClassUtils.primitiveToWrapper(rawParameterType.getComponentType());
                if (Part.class.isAssignableFrom(arrayComponentType)) {
                    throw parameterError(container.parameterIndex,
                            "@Part parameters using the org.asynchttpclient.request.body.multipart.Part must not "
                                    + "include a part name in the annotation.");
                }
                return new ArrayParameterHandler<>(new PartParameterHandler<>(partConverter, part.encoding()));
            } else if (Part.class.isAssignableFrom(rawParameterType)) {
                throw parameterError(container.parameterIndex,
                        "@Part parameters using the org.asynchttpclient.request.body.multipart.Part must not "
                                + "include a part name in the annotation.");
            } else {
                return new PartParameterHandler<>(partConverter, part.encoding());
            }
        }
    }

    private ParameterHandler<?> parsePartMapAnnotation(AnnotationContainer container) {
        if (!isMultipart) {
            throw parameterError(container.parameterIndex, "@PartMap parameters can only be used with multipart " +
                    "encoding.");
        }
        gotPart = true;
        Class<?> rawParameterType = Utils.getRawType(container.parameterType);
        if (!Map.class.isAssignableFrom(rawParameterType)) {
            throw parameterError(container.parameterIndex, "@PartMap parameter type must be Map.");
        }
        Type mapType = Utils.getSupertype(container.parameterType, rawParameterType, Map.class);
        if (!(mapType instanceof ParameterizedType)) {
            throw parameterError(container.parameterIndex, "Map must include generic types (e.g., Map<String, " +
                    "String>)");
        }
        ParameterizedType parameterizedType = (ParameterizedType) mapType;

        Type keyType = Utils.getParameterUpperBound(0, parameterizedType);
        if (!String.class.equals(keyType)) {
            throw parameterError(container.parameterIndex, "@PartMap keys must be of type String: " + keyType);
        }

        Type valueType = Utils.getParameterUpperBound(1, parameterizedType);
        if (Part.class.isAssignableFrom(Utils.getRawType(valueType))) {
            throw parameterError(container.parameterIndex,
                    "@PartMap values cannot be org.asynchttpclient.request.body.multipart.Part. "
                            + "Use @Part List<Part> or a different value type instead.");
        }

        PartConverter<?> partConverter = partConverter(container.annotations, smart, null);

        PartMap partMap = (PartMap) container.annotation;
        return new PartMapParameterHandler<>(partConverter, partMap.encoding());
    }

    private ParameterHandler<?> parseBodyAnnotation(AnnotationContainer container) {
        if (isFormEncoded || isMultipart) {
            throw parameterError(container.parameterIndex,
                    "@Body parameters cannot be used with form or multi-part encoding.");
        }
        if (gotBody) {
            throw parameterError(container.parameterIndex, "Multiple @Body method annotations found.");
        }
        RequestBodyConverter<?> converter = requestConverter(container.annotations, smart);
        gotBody = true;
        return new BodyParameterHandler<>(container.parameterType, converter);
    }

    private ParameterHandler<?> parsePartibleBodyAnnotation(AnnotationContainer container) {
        if (hasPartibleParam) {
            throw parameterError(container.parameterIndex, "Only one partible parameter allowed. (@PartibleBody " +
                    "or @PartibleQuery)");
        }
        if (isFormEncoded || isMultipart) {
            throw parameterError(container.parameterIndex,
                    "@Body parameters cannot be used with form or multi-part encoding.");
        }
        if (gotBody) {
            throw parameterError(container.parameterIndex, "Multiple @Body method annotations found.");
        }

        gotBody = true;
        hasPartibleParam = true;
        Class<?> rawParameterType = Utils.getRawType(container.parameterType);
        if (Iterable.class.isAssignableFrom(rawParameterType)) {
            if (!(container.parameterType instanceof ParameterizedType)) {
                throw parameterError(container.parameterIndex, rawParameterType.getSimpleName()
                        + " must include generic type (e.g., "
                        + rawParameterType.getSimpleName()
                        + "<String>)");
            }
            ParameterizedType parameterizedType = (ParameterizedType) container.parameterType;
            Type iterableType = Utils.getParameterUpperBound(0, parameterizedType);
            return new IterableParameterHandler<>(new PartibleBodyParameterHandler<>(iterableType));
        } else {
            throw parameterError(container.parameterIndex, "@PartibleBody parameter has to extend Iterable.");
        }
    }

    private ParameterHandler<?> parseIdAnnotation(AnnotationContainer container) {
        Class<?> rawParameterType = Utils.getRawType(container.parameterType);
        if (!Long.class.isAssignableFrom(rawParameterType)) {
            throw parameterError(container.parameterIndex, "@Id parameter must be java.lang.Long.");
        }
        if (hasId) {
            throw parameterError(container.parameterIndex, "Multiple @Id method annotations found.");
        }
        hasId = true;
        return IdParameterHandler.INSTANCE;
    }

    private ParameterHandler<?> parseCookieAnnotation(AnnotationContainer container) {
        Cookie cookie = (Cookie) container.annotation;
        String name = cookie.value();

        Class<?> rawParameterType = Utils.getRawType(container.parameterType);
        if (Iterable.class.isAssignableFrom(rawParameterType) || rawParameterType.isArray()) {
            throw parameterError(container.parameterIndex, "@Cookie parameter must be a scalar");
        }
        RequestBodyConverter<?> converter = requestConverter(container.annotations, smart);
        return new CookieParameterHandler<>(name, converter);
    }

    private void validatePathName(int p, String name) {
        if (!PARAM_NAME_REGEX.matcher(name).matches()) {
            throw parameterError(p, "@Path parameter name must match %s. Found: %s",
                    PARAM_URL_REGEX.pattern(), name);
        }
        if (!relativeUrlParamNames.contains(name)) {
            throw parameterError(p, "URL \"%s\" does not contain \"{%s}\".", relativeUrl, name);
        }
    }

    private IllegalArgumentException methodError(String message, Object... args) {
        return methodError(null, message, args);
    }

    private IllegalArgumentException methodError(Throwable cause, String message, Object... args) {
        message = String.format(message, args);
        return new IllegalArgumentException(message
                + "\n    for method "
                + method.getDeclaringClass().getSimpleName()
                + "."
                + method.getName(), cause);
    }

    private IllegalArgumentException parameterError(int p, String message, Object... args) {
        return methodError(message + " (parameter #" + (p + 1) + ")", args);
    }

    private static Set<String> parsePathParameters(String path) {
        Matcher m = PARAM_URL_REGEX.matcher(path);
        Set<String> patterns = new LinkedHashSet<>();
        while (m.find()) {
            patterns.add(m.group(1));
        }
        return patterns;
    }

    private static RequestBodyConverter<?> requestConverter(Annotation[] annotations, Smart smart) {
        return smart.requestConverterFactory.getConverter(annotations);
    }

    private static PartConverter<?> partConverter(Annotation[] parameterAnnotations, Smart smart, String partName) {
        RequestBodyConverter<?> sc = requestConverter(parameterAnnotations, smart);
        return new PartConverter<>(sc, partName);
    }

    private class ParserPredicate {
        ParserPredicate(Predicate<AnnotationContainer> predicate,
                        Function<AnnotationContainer, ParameterHandler<?>> function) {
            this.predicate = predicate;
            this.function = function;
        }

        final Predicate<AnnotationContainer> predicate;
        final Function<AnnotationContainer, ParameterHandler<?>> function;
    }

    private class AnnotationContainer {
        AnnotationContainer(int parameterIndex,
                            Type parameterType,
                            Annotation[] annotations,
                            Annotation annotation) {
            this.parameterIndex = parameterIndex;
            this.parameterType = parameterType;
            this.annotations = annotations;
            this.annotation = annotation;
        }

        final int parameterIndex;
        final Type parameterType;
        final Annotation[] annotations;
        final Annotation annotation;
    }

}
