package ru.yandex.direct.http.smart.core;

import java.lang.invoke.MethodHandles;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.Nullable;

import io.netty.handler.codec.http.HttpHeaders;
import org.asynchttpclient.uri.Uri;

import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.http.smart.converter.RequestConverterFactory;
import ru.yandex.direct.http.smart.converter.ResponseConverterFactory;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.inside.passport.tvm2.TvmHeaders;

import static com.google.common.base.Preconditions.checkState;


public class Smart {
    private final Map<Method, ServiceMethod<?>> serviceMethodCache = new ConcurrentHashMap<>();
    final RequestConverterFactory requestConverterFactory;
    final ResponseConverterFactory responseConverterFactory;
    final ParallelFetcherFactory parallelFetcherFactory;
    final List<? extends Configurator<HttpHeaders>> headersConfigurators;
    final String baseUrl;
    final String profileName;

    public Smart(RequestConverterFactory requestConverterFactory,
                 ResponseConverterFactory responseConverterFactory,
                 ParallelFetcherFactory parallelFetcherFactory,
                 List<? extends Configurator<HttpHeaders>> headersConfigurators, String baseUrl,
                 String profileName) {
        this.requestConverterFactory = requestConverterFactory;
        this.responseConverterFactory = responseConverterFactory;
        this.parallelFetcherFactory = parallelFetcherFactory;
        this.headersConfigurators = headersConfigurators;
        this.baseUrl = baseUrl;
        this.profileName = profileName;
    }

    @SuppressWarnings("unchecked")
    public <T> T create(final Class<T> service) {
        validateServiceInterface(service);
        return (T) Proxy.newProxyInstance(service.getClassLoader(), new Class<?>[]{service},
                new InvocationHandler() {
                    @Override
                    public Object invoke(Object proxy, Method method, @Nullable Object[] args) throws Throwable {
                        if (method.getDeclaringClass().equals(Object.class)) {
                            return method.invoke(this, args);
                        }
                        if (method.isDefault()) {
                            return invokeDefaultMethod(method, service, proxy, args);
                        }
                        ServiceMethod<Object> serviceMethod =
                                (ServiceMethod<Object>) loadServiceMethod(method);
                        return serviceMethod.toCall(args);
                    }
                });
    }

    private ServiceMethod<?> loadServiceMethod(Method method) {
        ServiceMethod<?> result = serviceMethodCache.get(method);
        if (result != null) {
            return result;
        }

        synchronized (serviceMethodCache) {
            result = serviceMethodCache.get(method);
            if (result == null) {
                result = new ServiceMethodBuilder<>(this, method).build();
                serviceMethodCache.put(method, result);
            }
        }
        return result;
    }

    private Object invokeDefaultMethod(Method method, Class<?> declaringClass, Object object,
                                       @Nullable Object... args) throws Throwable {
        Constructor<MethodHandles.Lookup>
                constructor = MethodHandles.Lookup.class.getDeclaredConstructor(Class.class, int.class);
        constructor.setAccessible(true);
        return constructor.newInstance(declaringClass, -1 /* trusted */)
                .unreflectSpecial(method, declaringClass)
                .bindTo(object)
                .invokeWithArguments(args);
    }

    private void validateServiceInterface(Class<?> service) {
        if (!service.isInterface()) {
            throw new IllegalArgumentException("API declarations must be interfaces.");
        }
        if (service.getInterfaces().length > 0) {
            throw new IllegalArgumentException("API interfaces must not extend other interfaces.");
        }
        for (Method method : service.getDeclaredMethods()) {
            if (method.isDefault()) {
                loadServiceMethod(method);
            }
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public static final class Builder {
        private ParallelFetcherFactory parallelFetcherFactory;
        private String baseUrl;
        private String profileName;
        private boolean tvmAdded = false;
        private RequestConverterFactory requestConverterFactory = new RequestConverterFactory();
        private ResponseConverterFactory responseConverterFactory = ResponseConverterFactory.builder().build();
        private List<Configurator<HttpHeaders>> headersConfigurators = new ArrayList<>();

        public Builder() {
        }

        public Builder withParallelFetcherFactory(ParallelFetcherFactory parallelFetcherFactory) {
            this.parallelFetcherFactory = parallelFetcherFactory;
            return this;
        }

        public Builder addHeaderConfigurator(Configurator<HttpHeaders> configurator) {
            headersConfigurators.add(configurator);
            return this;
        }

        public Builder useTvm(TvmIntegration tvmIntegration, TvmService tvmService) {
            checkState(!tvmAdded, "tvm is already used");
            headersConfigurators.add(headers -> {
                if (tvmIntegration.isEnabled() && !headers.contains(TvmHeaders.SERVICE_TICKET)) {
                    String serviceTicket = tvmIntegration.getTicket(tvmService);
                    if (serviceTicket != null) {
                        headers.add(TvmHeaders.SERVICE_TICKET, serviceTicket);
                    }
                }
            });
            tvmAdded = true;
            return this;
        }

        public Builder withBaseUrl(String baseUrl) {
            this.baseUrl = baseUrl;
            return this;
        }

        public Builder withProfileName(String profileName) {
            this.profileName = profileName;
            return this;
        }

        public Builder withRequestConverterFactory(
                RequestConverterFactory requestBodyConverterFactory) {
            this.requestConverterFactory = requestBodyConverterFactory;
            return this;
        }

        public Builder withResponseConverterFactory(
                ResponseConverterFactory responseBodyConverterFactory) {
            this.responseConverterFactory = responseBodyConverterFactory;
            return this;
        }

        public Smart build() {
            checkState(baseUrl != null, "Base URL required.");
            checkState(profileName != null, "profileName required.");
            checkState(parallelFetcherFactory != null, "parallelFetcherFactory required.");

            Uri.create(baseUrl);

            return new Smart(requestConverterFactory, responseConverterFactory, parallelFetcherFactory,
                    headersConfigurators, baseUrl, profileName);
        }
    }
}
