package ru.yandex.direct.http.smart.parameters;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.lang.reflect.Type;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Message;
import org.apache.commons.lang3.reflect.TypeUtils;

import ru.yandex.direct.http.smart.core.SmartRequestBuilder;

@ParametersAreNonnullByDefault
public class PartibleBodyParameterHandler<T> implements ParameterHandler<T> {
    private final Type responseType;

    public PartibleBodyParameterHandler(Type responseType) {
        this.responseType = responseType;
    }

    @Override
    public void apply(SmartRequestBuilder builder, @Nullable T value) {
        if (value == null) {
            throw new IllegalArgumentException("Body parameter value must not be null.");
        }

        if (TypeUtils.isAssignable(responseType, byte[].class)) {
            builder.addPartibleBodyParam((byte[]) value);
        } else if (TypeUtils.isAssignable(responseType, Message.class)) {
            builder.addPartibleBodyParam(protobufMessageToByteArray((Message) value));
        } else {
            throw new UnsupportedOperationException("Partible body parameter of type " +
                    responseType + " is not supported");
        }
    }

    private byte[] protobufMessageToByteArray(Message message) {
        ByteArrayOutputStream stream = new ByteArrayOutputStream();
        try {
            message.writeDelimitedTo(stream);
        } catch (IOException e) {
            throw new RuntimeException("Failed to serialize protobuf body", e);
        }
        return stream.toByteArray();
    }
}
