package ru.yandex.direct.traceinterception.entity.traceinterception.model;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Semaphore;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.traceinterception.model.TraceInterception;
import ru.yandex.direct.traceinterception.model.TraceInterceptionCondition;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricId;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
public class TraceInterceptionsStorage {
    private static final Logger logger = LoggerFactory.getLogger(TraceInterceptionsStorage.class);
    private static final ConcurrentHashMap<Object, ResizeableSemaphore> SEMAPHORES = new ConcurrentHashMap<>();
    private final List<TraceInterception> traceInterceptions;
    private final Map<TraceInterceptionCondition, TraceInterception> traceCondition2Interception;

    public TraceInterceptionsStorage(List<TraceInterception> traceInterceptions) {
        this.traceInterceptions = traceInterceptions;
        // если в базе оказались правила с одинаковым условием, то берем с наибольшим ID, а остальные игнорируем
        this.traceCondition2Interception = traceInterceptions.stream()
                .collect(Collectors.toMap(TraceInterception::getCondition, Function.identity(),
                        (l, r) -> l.getId() < r.getId() ? r : l));
    }

    public static Object getSemaphoreKey(TraceInterception traceInterception) {
        return nvl(traceInterception.getAction().getSemaphoreKey(), traceInterception.getId());
    }


    public void refreshSemaphoresMap() {
        Map<Object, Integer> permits = new HashMap<>();
        StreamEx.of(traceInterceptions)
                .distinct()
                .mapToEntry(TraceInterceptionsStorage::getSemaphoreKey, i -> i.getAction().getSemaphorePermits())
                .nonNullValues()
                .forKeyValue((key, value) -> {
                    permits.merge(key, value, (oldValue, newValue) -> {
                        if (!oldValue.equals(newValue)) {
                            int max = Math.max(oldValue, newValue);
                            logger.warn("Semaphore {} have different permits: {} {}. Chosen {}",
                                    key, oldValue, newValue, max);
                            return max;
                        }
                        return oldValue;
                    });
                });

        // 1. если правило новое - создаем семафор
        // 2. если поменялось значение - делаем release или reduce (без блокировки), чтобы поддержать изменение в
        // начальном количестве permit'ов
        // 3. если правило на семафор выключено или удалено, то выкидываем его из мапы
        permits.forEach((semaphoreKey, newMaxPermits) -> SEMAPHORES.compute(semaphoreKey, (key, value) -> {
                    MetricRegistry registry = SOLOMON_REGISTRY.subRegistry("trace_interception_semaphore",
                            semaphoreKey.toString());
                    registry.gaugeInt64("limit").set(newMaxPermits);
                    if (value == null) {
                        ResizeableSemaphore semaphore = new ResizeableSemaphore(newMaxPermits);
                        registry.lazyGaugeInt64("used", () -> semaphore.maxPermits - semaphore.availablePermits());
                        registry.putMetricIfAbsent(new MetricId("errors", Labels.empty()), semaphore.errors);
                        return semaphore;
                    } else if (value.maxPermits < newMaxPermits) {
                        value.release(newMaxPermits - value.maxPermits);
                    } else if (value.maxPermits > newMaxPermits) {
                        value.reducePermits(value.maxPermits - newMaxPermits);
                    }
                    value.maxPermits = newMaxPermits;
                    return value;
                })
        );

        // удаляем семафоры выключенных правил
        var iterator = SEMAPHORES.entrySet().iterator();
        while (iterator.hasNext()) {
            var entry = iterator.next();
            if (!permits.containsKey(entry.getKey())) {
                SOLOMON_REGISTRY.removeSubRegistry("trace_interception_semaphore", entry.getKey().toString());
                iterator.remove();
            }
        }
    }

    public List<TraceInterception> getTraceInterceptions() {
        return traceInterceptions;
    }

    public TraceInterception findInterception(TraceInterceptionCondition condition) {
        return traceCondition2Interception.get(condition);
    }

    public Semaphore getSemaphore(TraceInterception traceInterception) {
        Object key = getSemaphoreKey(traceInterception);
        return SEMAPHORES.get(key);
    }

    private static final class ResizeableSemaphore extends Semaphore {
        private int maxPermits;
        private final Rate errors;
        ResizeableSemaphore(int permits) {
            super(permits);
            maxPermits = permits;
            errors = new Rate();
        }

        @Override
        public boolean tryAcquire() {
            boolean result = super.tryAcquire();
            if (!result) {
                errors.inc();
            }
            return result;
        }

        @SuppressWarnings("EmptyMethod")
        @Override
        protected void reducePermits(int reduction) {
            super.reducePermits(reduction);
        }
    }
}
