package ru.yandex.direct.traceinterception.entity.traceinterception.repository;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppcdict.enums.TraceInterceptionsStatus;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.traceinterception.model.TraceInterception;
import ru.yandex.direct.traceinterception.model.TraceInterceptionAction;
import ru.yandex.direct.traceinterception.model.TraceInterceptionCondition;
import ru.yandex.direct.traceinterception.model.TraceInterceptionStatus;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.dbschema.ppcdict.tables.TraceInterceptions.TRACE_INTERCEPTIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class TraceInterceptionsRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<TraceInterception> jooqMapper;

    public TraceInterceptionsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(TraceInterception::new)
                .map(property(TraceInterception.ID, TRACE_INTERCEPTIONS.ID))
                .map(convertibleProperty(TraceInterception.CONDITION, TRACE_INTERCEPTIONS.CONDITION,
                        TraceInterceptionsRepository::convertConditionFromSource,
                        TraceInterceptionsRepository::convertConditionToSource))
                .map(convertibleProperty(TraceInterception.ACTION, TRACE_INTERCEPTIONS.ACTION,
                        TraceInterceptionsRepository::convertActionFromSource,
                        TraceInterceptionsRepository::convertActionToSource))
                .map(convertibleProperty(TraceInterception.STATUS, TRACE_INTERCEPTIONS.STATUS,
                        TraceInterceptionStatus::fromSource, TraceInterceptionStatus::toSource))
                .build();
    }

    private static TraceInterceptionCondition convertConditionFromSource(String condition) {
        return JsonUtils.fromJson(condition, TraceInterceptionCondition.class);
    }

    private static String convertConditionToSource(TraceInterceptionCondition condition) {
        return JsonUtils.toJson(condition);
    }

    private static TraceInterceptionAction convertActionFromSource(String action) {
        return JsonUtils.fromJson(action, TraceInterceptionAction.class);
    }

    private static String convertActionToSource(TraceInterceptionAction action) {
        return JsonUtils.toJson(action);
    }

    /**
     * Возвращает полный список правил перехвата.
     */
    public List<TraceInterception> getAll() {
        return dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(TRACE_INTERCEPTIONS)
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Возвращает список активных правил перехвата.
     */
    public List<TraceInterception> getActive() {
        return dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(TRACE_INTERCEPTIONS)
                .where(TRACE_INTERCEPTIONS.STATUS.eq(TraceInterceptionsStatus.on))
                .orderBy(TRACE_INTERCEPTIONS.ID)
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Добавляет новое правило перехвата.
     */
    public void add(TraceInterception traceInterception) {
        add(ImmutableList.of(traceInterception));
    }

    /**
     * Добавляет новые правила перехвата.
     */
    public void add(List<TraceInterception> traceInterceptions) {
        new InsertHelper<>(dslContextProvider.ppcdict(), TRACE_INTERCEPTIONS)
                .addAll(jooqMapper, traceInterceptions)
                .execute();
    }

    public boolean updateActionById(Long id, TraceInterceptionAction action) {
        return dslContextProvider.ppcdict()
                .update(TRACE_INTERCEPTIONS)
                .set(TRACE_INTERCEPTIONS.ACTION, convertActionToSource(action))
                .where(TRACE_INTERCEPTIONS.ID.eq(id))
                .execute() > 0;
    }

    public boolean updateStatusById(Long id, TraceInterceptionStatus status) {
        return dslContextProvider.ppcdict()
                .update(TRACE_INTERCEPTIONS)
                .set(TRACE_INTERCEPTIONS.STATUS, TraceInterceptionStatus.toSource(status))
                .where(TRACE_INTERCEPTIONS.ID.eq(id))
                .execute() > 0;
    }

    public boolean delete(Long id) {
        return dslContextProvider.ppcdict()
                .deleteFrom(TRACE_INTERCEPTIONS)
                .where(TRACE_INTERCEPTIONS.ID.equal(id))
                .execute() > 0;
    }
}
