package ru.yandex.direct.traceinterception.entity.traceinterception.service;

import java.util.concurrent.Semaphore;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.traceinterception.entity.traceinterception.exception.TraceInterceptionActionException;
import ru.yandex.direct.traceinterception.entity.traceinterception.exception.TraceInterceptionSemaphoreException;
import ru.yandex.direct.traceinterception.entity.traceinterception.model.TraceInterceptionsStorage;
import ru.yandex.direct.traceinterception.model.TraceInterception;
import ru.yandex.direct.traceinterception.model.TraceInterceptionAction;
import ru.yandex.direct.traceinterception.model.TraceInterceptionCondition;
import ru.yandex.direct.tracing.TraceInterceptor;
import ru.yandex.direct.tracing.TraceProfile;

@Service
public class TraceInterceptionsService implements TraceInterceptor {
    private static final Logger logger = LoggerFactory.getLogger(TraceInterceptionsService.class);

    // Комбинации для поиска правила по "условию" TraceInterceptionCondition
    // Нужны, чтобы можно было сделать расширенное правило, например на весь сервис разом
    // Сначала смотрим полное совпадение по Service, Method, Func, Tags
    // Далее предпочтение отдаем функции, и перебираем отсутствие Service и Method
    // Если "по функции" не удалось найти, то уже перебираем по Service и Method без нее
    // Каждая комбинация рассматривается сначала с тегами, а потом сразу без (образуются как-бы пары из подряд идущих правил)
    // Первое подходящее правило отдаем как результат
    // Возможно, этот порядок нужно обозначить на страничке в InternalTools
    private static final boolean I = true;
    private static final boolean O = false;
    private static final boolean[][] COMBINATIONS = {
            {I, I, I, I},
            {I, I, I, O},
            {O, I, I, I},
            {O, I, I, O},
            {I, O, I, I},
            {I, O, I, O},
            {O, O, I, I},
            {O, O, I, O},
            {I, I, O, I},
            {I, I, O, O},
            {O, I, O, I},
            {O, I, O, O},
            {I, O, O, I},
            {I, O, O, O},
            {O, O, O, I}};

    private final TraceInterceptionsStorageCache traceInterceptionsStorageCache;

    public TraceInterceptionsService(TraceInterceptionsStorageCache traceInterceptionsStorageCache) {
        this.traceInterceptionsStorageCache = traceInterceptionsStorageCache;
    }

    @Override
    public void checkInterception(TraceProfile traceProfile) {
        TraceInterceptionsStorage storage = traceInterceptionsStorageCache.getStorage();
        if (storage.getTraceInterceptions().isEmpty()) {
            return;
        }

        TraceInterception interception = findTraceInterception(storage, traceProfile);
        if (interception == null || interception.getAction() == null) {
            return;
        }

        TraceInterceptionAction action = interception.getAction();
        // порядок, в котором проверяется наличие "действия" важен
        if (action.getSleepDuration() != null && action.getSleepDuration() >= 0) {
            try {
                logger.trace("Sleeping for {} seconds (interception: {})", action.getSleepDuration(), interception);
                Thread.sleep(action.getSleepDuration());
            } catch (InterruptedException e) {
                logger.trace("Thread was interrupted");
                Thread.currentThread().interrupt();
            }
        }

        if (action.getExceptionMessage() != null) {
            throw new TraceInterceptionActionException(action.getExceptionMessage());
        }

        if (action.getSemaphorePermits() != null) {
            Semaphore semaphore = storage.getSemaphore(interception);
            if (semaphore != null) {
                if (!semaphore.tryAcquire()) {
                    throw new TraceInterceptionSemaphoreException("Failed to acquire semaphore (interception: " +
                            interception.toString() + ")");
                }
                logger.trace("Semaphore is acquired (interception: {})", interception);

                traceProfile.addCommandOnClose(() -> {
                    semaphore.release();
                    logger.trace("Semaphore is released (interception: {})", interception);
                });
            }
        }
    }

    private TraceInterception findTraceInterception(TraceInterceptionsStorage storage, TraceProfile traceProfile) {
        TraceInterceptionCondition condition = new TraceInterceptionCondition();

        for (var comb : COMBINATIONS) {
            condition.withService(comb[0] == I ? traceProfile.getService() : null);
            condition.withMethod(comb[1] == I ? traceProfile.getMethod() : null);
            condition.withFunc(comb[2] == I ? traceProfile.getFunc() : null);
            condition.withTags(comb[3] == I ? traceProfile.getTags() : null);

            TraceInterception interception = storage.findInterception(condition);
            if (interception != null) {
                return interception;
            }
        }
        return null;
    }
}
