package ru.yandex.direct.traceinterception.entity.traceinterception.service;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import javax.annotation.PreDestroy;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.traceinterception.entity.traceinterception.model.TraceInterceptionsStorage;
import ru.yandex.direct.traceinterception.entity.traceinterception.repository.TraceInterceptionsRepository;
import ru.yandex.direct.traceinterception.model.TraceInterception;

@Service
public class TraceInterceptionsStorageCache {
    private static final Logger logger = LoggerFactory.getLogger(TraceInterceptionsStorageCache.class);

    private final TraceInterceptionsRepository traceInterceptionsRepository;

    private final ScheduledExecutorService executor;

    private volatile TraceInterceptionsStorage storage = new TraceInterceptionsStorage(Collections.emptyList());

    public TraceInterceptionsStorageCache(TraceInterceptionsRepository traceInterceptionsRepository,
                                          @Value("${trace.interceptions.storageReloadIntervalInMs:30000}") long reloadIntervalMS,
                                          @Value("${trace.interceptions.enabled:true}") boolean enabled) {
        this.traceInterceptionsRepository = traceInterceptionsRepository;
        if (enabled) {
            this.executor = createDefaultExecutor();
            executor.scheduleAtFixedRate(
                    this::updateStorage,
                    0L,
                    reloadIntervalMS,
                    TimeUnit.MILLISECONDS
            );
        } else {
            this.executor = null;
        }
    }

    public TraceInterceptionsStorage getStorage() {
        return storage;
    }

    private void updateStorage() {
        try {
            // так как обновление идет в отдельном треде не под трейсингом, то можно не выставлять флаг пропуска перехвата
            List<TraceInterception> traceInterceptions = traceInterceptionsRepository.getActive();

            // если поменялись правила, то обновляем сторадж
            if (!traceInterceptions.equals(storage.getTraceInterceptions())) {
                storage = new TraceInterceptionsStorage(traceInterceptions);
                // немного лениво (не атомарно с предыдущей операцией) обновляем мапу для семафоров вместе с новыми правилами
                storage.refreshSemaphoresMap();
            }
        } catch (Exception e) {
            logger.error("Unhandled exception", e);
        }
    }

    private ScheduledExecutorService createDefaultExecutor() {
        ThreadFactory threadFactory =
                new ThreadFactoryBuilder().setNameFormat("trace-interceptions-updater-%02d").setDaemon(true).build();
        return Executors.newScheduledThreadPool(1, threadFactory);
    }

    @PreDestroy
    public void stopExecutor() {
        if (executor != null) {
            executor.shutdown();
        }
    }
}
