package ru.yandex.direct.tvm;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.inside.passport.tvm2.exceptions.IncorrectTvmServiceTicketException;
import ru.yandex.inside.passport.tvm2.exceptions.IncorrectTvmUserTicketException;
import ru.yandex.inside.passport.tvm2.exceptions.MissingTvmServiceTicketException;
import ru.yandex.inside.passport.tvm2.exceptions.MissingTvmUserTicketException;

import static java.util.Arrays.asList;
import static ru.yandex.direct.tvm.TvmIntegration.TVM_SERVICE_ID_ATTRIBUTE_NAME;

/**
 * Разрешает, либо отклоняет доступ к API-endpoint на основании присутствующих {@link AllowServices}.
 * Наибольший приоритет у аннотации на методе. Если аннотации на методе нет - используется аннотация класса.
 * Доступ не ограничивается если ни у метода, ни у класса нет аннотации {@link AllowServices}.
 * <p>
 * Родительские классы на наличие аннотаций не проверяются.
 *
 * AbstractTvmInterceptor - абстрактный класс с логикой интерсептора, для использования нужно в наследние
 * переопределить логгер и способ возвращения ошибок
 */
public abstract class AbstractTvmInterceptor extends HandlerInterceptorAdapter {
    private final TvmIntegration tvmIntegration;
    private final EnvironmentType environmentType;

    public AbstractTvmInterceptor(TvmIntegration tvmIntegration, EnvironmentType environmentType) {
        this.tvmIntegration = tvmIntegration;
        this.environmentType = environmentType;
    }

    protected abstract Logger getLogger();
    protected abstract Exception accessDeniedExceptionInvalidTvmTicket(Exception cause);
    protected abstract Exception accessDeniedExceptionInvalidUserTicket(Exception cause);
    protected abstract Exception accessDeniedExceptionNoTvmTicket();
    protected abstract Exception sourceApplicationAccessDeniedException(TvmService sourceService);

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler)
            throws Exception {
        if (!(handler instanceof HandlerMethod)) {
            return true;
        }
        if (!tvmIntegration.isEnabled()) {
            getLogger().debug("Tvm service disabled");
            return true;
        }

        HandlerMethod handlerMethod = ((HandlerMethod) handler);
        AllowServices allowedServices = handlerMethod.getMethod().getAnnotation(AllowServices.class);
        if (allowedServices == null) {
            allowedServices = handlerMethod.getBeanType().getAnnotation(AllowServices.class);
        }

        if (allowedServices == null) {
            return true;
        }

        String serviceTicket = request.getHeader(TvmHeaders.SERVICE_TICKET);
        if (serviceTicket == null) {
            getLogger().debug("Absent service ticket in header {}", TvmHeaders.SERVICE_TICKET);
            throw accessDeniedExceptionNoTvmTicket();
        }

        TvmService sourceService;
        try {
            sourceService = tvmIntegration.getTvmService(serviceTicket);
        } catch (IncorrectTvmServiceTicketException
                | IncorrectTvmUserTicketException
                | MissingTvmServiceTicketException
                | MissingTvmUserTicketException
                | UnknownTvmServiceException exception) {
            throw accessDeniedExceptionInvalidTvmTicket(exception);
        }
        request.setAttribute(TVM_SERVICE_ID_ATTRIBUTE_NAME, sourceService.getId());

        TvmService[] allowed;
        if (environmentType.isProductionOrPrestable()) {
            allowed = allowedServices.production();
        } else if (environmentType.isProductionSandbox()) {
            allowed = allowedServices.sandbox();
        } else if (environmentType.isSandbox()) {
            allowed = allowedServices.sandboxTesting();
        } else {
            allowed = allowedServices.testing();
        }

        if (!ArrayUtils.contains(allowed, sourceService)) {
            getLogger().warn("TVM check failed. ClientId {} is not in AllowServices {}", sourceService, asList(allowed));
            throw sourceApplicationAccessDeniedException(sourceService);
        }

        String userTicket = request.getHeader(TvmHeaders.USER_TICKET);
        // todo в будущем планируется добавить возможность обязательности указания User-тикета для определенных ручек
        if (userTicket != null) {
            try {
                tvmIntegration.checkUserTicket(userTicket);
            } catch (IncorrectTvmUserTicketException
                    | MissingTvmUserTicketException exception) {
                throw accessDeniedExceptionInvalidUserTicket(exception);
            }
        }
        return true;
    }

}
