package ru.yandex.direct.tvm;

import java.time.Duration;
import java.util.Collections;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.TaskScheduler;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.liveresource.LiveResource;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.liveresource.PollingLiveResourceWatcher;
import ru.yandex.direct.solomon.SolomonExternalSystemMonitorService;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.inside.passport.tvm2.Tvm2ApiClient;
import ru.yandex.inside.passport.tvm2.TvmClientCredentials;
import ru.yandex.inside.passport.tvm2.exceptions.IncorrectTvmServiceTicketException;
import ru.yandex.inside.passport.tvm2.exceptions.IncorrectTvmUserTicketException;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.passport.tvmauth.BlackboxEnv;
import ru.yandex.passport.tvmauth.CheckedServiceTicket;
import ru.yandex.passport.tvmauth.CheckedUserTicket;

import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_2XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_4XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_5XX;


@ParametersAreNonnullByDefault
public class TvmIntegrationImpl implements TvmIntegration {
    public static final long LIVE_CONFIG_CHECK_RATE = TimeUnit.SECONDS.toMillis(30);
    private static final Logger logger = LoggerFactory.getLogger(TvmIntegrationImpl.class);

    private static final String EXTERNAL_SYSTEM = "tvm";
    private static final String METHOD_CHECK_SERVICE_TICKET = "checkServiceTicket";
    private static final String METHOD_CHECK_USER_TICKET = "checkUserTicket";
    private static final String METHOD_GET_SERVICE_TICKET = "getServiceTicket";

    private static final SolomonExternalSystemMonitorService monitorService = new SolomonExternalSystemMonitorService(
            EXTERNAL_SYSTEM,
            Set.of(METHOD_CHECK_SERVICE_TICKET, METHOD_CHECK_USER_TICKET, METHOD_GET_SERVICE_TICKET)
    );

    private Tvm2 tvm;
    private final TvmConfig tvmConfig;

    public TvmIntegrationImpl(TvmConfig tvmConfig) {
        this.tvmConfig = tvmConfig;
    }

    @Override
    public TvmService getTvmService(String ticketBody) throws IncorrectTvmServiceTicketException {
        try {
            CheckedServiceTicket ticket = tvm.checkServiceTicket(ticketBody);
            if (!ticket.booleanValue()) {
                logger.warn("invalid ticket given {}, status {}", ticket.debugInfo(), ticket.getStatus());
                monitorService.write(METHOD_CHECK_SERVICE_TICKET, STATUS_4XX);
                throw new IncorrectTvmServiceTicketException();
            }

            TvmService tvmService = TvmService.fromId(ticket.getSrc());
            if (tvmService == TvmService.UNKNOWN) {
                monitorService.write(METHOD_CHECK_SERVICE_TICKET, STATUS_4XX);
                throw new UnknownTvmServiceException(ticket.getSrc());
            }
            monitorService.write(METHOD_CHECK_SERVICE_TICKET, STATUS_2XX);
            return tvmService;
        } catch (RuntimeException e) {
            monitorService.write(METHOD_CHECK_SERVICE_TICKET, STATUS_5XX);
            throw e;
        }
    }

    @Override
    public long checkUserTicket(String ticketBody) throws IncorrectTvmUserTicketException {
        try {
            CheckedUserTicket ticket = tvm.checkUserTicket(ticketBody);
            if (!ticket.booleanValue()) {
                logger.warn("invalid user ticket given {}, status {}", ticket.debugInfo(), ticket.getStatus());
                monitorService.write(METHOD_CHECK_USER_TICKET, STATUS_4XX);
                throw new IncorrectTvmUserTicketException();
            }
            monitorService.write(METHOD_CHECK_USER_TICKET, STATUS_2XX);
            return ticket.getDefaultUid();
        } catch (RuntimeException e) {
            monitorService.write(METHOD_CHECK_USER_TICKET, STATUS_5XX);
            throw e;
        }
    }

    @Override
    public TvmService currentTvmService() {
        return tvmConfig.getService();
    }

    @Override
    public void close() {
        tvm.stop();
    }

    @Override
    public boolean isEnabled() {
        return true;
    }

    @Override
    public String getTicket(TvmService service) {
        try {
            String ticket = tvm.getServiceTicket(service.getId()).getOrNull();
            if (ticket == null) {
                monitorService.write(METHOD_GET_SERVICE_TICKET, STATUS_4XX);
                throw new IllegalStateException("Can not obtain ticket for service " + service);
            }
            monitorService.write(METHOD_GET_SERVICE_TICKET, STATUS_2XX);
            return ticket;
        } catch (RuntimeException e) {
            monitorService.write(METHOD_GET_SERVICE_TICKET, STATUS_5XX);
            throw e;
        }
    }

    /**
     * Создаёт экземпляр класса TvmServiceImpl и запускает поток обновления сервисных тикетов {@link Tvm2#start()}
     * Предполагается, что TvmServiceImpl — синглтон (аля @Service)
     */
    public static TvmIntegrationImpl create(DirectConfig directConfig, TaskScheduler taskScheduler) {
        TvmService service = TvmService.fromIdStrict(directConfig.getInt("tvm.app_id"));
        String tmvSecretUrl = directConfig.getString("tvm.secret");
        String tvmApiUrl = directConfig.getString("tvm.api.url");
        Duration soTimeout = directConfig.getDuration("tvm.api.so_timeout");
        Duration connectTimeout = directConfig.getDuration("tvm.api.connect_timeout");
        Duration updateDelay = directConfig.getDuration("tvm.api.update_delay");
        Duration errorDelay = directConfig.getDuration("tvm.api.error_delay");

        TvmConfig tvmConfig = new TvmConfig(service, tvmApiUrl, soTimeout, connectTimeout, updateDelay, errorDelay);

        //Создаём эезкмпляр сервиса. Методы initAndWatch обновляют данные в tvmConfig.
        //Только после обновления можно создать Tvm2 из конфига
        TvmIntegrationImpl tvmIntegration = new TvmIntegrationImpl(tvmConfig);
        tvmIntegration.initAndWatchSecret(tmvSecretUrl, taskScheduler);

        Tvm2 tvm = createTvm(tvmConfig);
        tvmIntegration.tvm = tvm;
        tvm.start();

        return tvmIntegration;
    }

    /**
     * Создаёт и настраивает экземпляр {@link Tvm2}
     */
    private static Tvm2 createTvm(TvmConfig tvmConfig) {
        TvmClientCredentials tvmClientCredentials =
                new TvmClientCredentials(tvmConfig.getServiceClientId(), tvmConfig.getTvmSecret());
        Tvm2ApiClient tvm2ApiClient = new Tvm2ApiClient(tvmConfig.getTvmApiUrl(),
                new Timeout(tvmConfig.getSoTimeout().toMillis(), tvmConfig.getConnectTimeout().toMillis()));
        Tvm2 tvm = new Tvm2(tvm2ApiClient, tvmClientCredentials,
                org.joda.time.Duration.millis(tvmConfig.getUpdateDelay().toMillis()),
                org.joda.time.Duration.millis(tvmConfig.getErrorDelay().toMillis()));
        configureTvm(tvm, tvmConfig);
        return tvm;
    }

    /**
     * Донастройка tvm. Параметры, которые не передать в конструктор
     */
    private static void configureTvm(Tvm2 tvm, TvmConfig tvmConfig) {
        tvm.setSrcClientIds(Collections.singletonList(tvmConfig.getServiceClientId()));
        tvm.setDstClientIds(TvmService.getKnownServices());
        // в случае с Mimino также используется Prod
        // https://wiki.yandex-team.ru/passport/tvm2/user-ticket/#0-opredeljaemsjasokruzhenijami
        tvm.setBlackboxEnv(BlackboxEnv.PROD);
    }

    /**
     * Для обновления секрета требуется создать новый экземпляр {@link Tvm2}.
     * Останавливаем текущий tvm
     * Обновляем значение в конфиге, затем пересоздаем объект
     * Запускаем {@link Tvm2#start()}
     */
    private void updateSecret(String tvmSecret) {
        tvmConfig.setTvmSecret(tvmSecret);
        tvm.stop();
        tvm = createTvm(tvmConfig);
        tvm.start();
        logger.info("Secret updated");
    }

    /**
     * Инициализирует секрет в конфиге и запускает {@link PollingLiveResourceWatcher}
     */
    private void initAndWatchSecret(String tmvSecretUrl, TaskScheduler taskScheduler) {
        if (tmvSecretUrl.contains("~/")) {
            tmvSecretUrl = tmvSecretUrl.replace("~/", System.getProperty("user.home") + "/");
        }

        LiveResource tvmSecret = LiveResourceFactory.get(tmvSecretUrl);
        PollingLiveResourceWatcher secretWatcher =
                new PollingLiveResourceWatcher(tvmSecret, tvmSecret.getContent(), taskScheduler, LIVE_CONFIG_CHECK_RATE);
        tvmConfig.setTvmSecret(tvmSecret.getContent());
        secretWatcher.addListener(e -> updateSecret(e.getCurrentContent()));
        secretWatcher.watch();
    }
}
