package ru.yandex.direct.tvm;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Список известных сервисов, использующих tvm.
 * Список также есть на вики: https://wiki.yandex-team.ru/direct/development/security/tvm/
 * <p>
 * Поиск TVM-приложений по их ID или оригинальным названиям:
 * https://abc.yandex-team.ru/resources/?supplier=14&type=47&state=requested&state=approved&state=granted
 * <p>
 * Как разработчику сгенерировать сервсиный TVM-тикет: https://st.yandex-team.ru/DIRECTKNOL-12
 */
@SuppressWarnings("SpellCheckingInspection")
public enum TvmService {
    /**
     * Директ: perl-скрипты и java-jobs
     */
    DIRECT_SCRIPTS_PROD(2000389),
    DIRECT_SCRIPTS_SANDBOX(2000922),
    DIRECT_SCRIPTS_TEST(2000767),
    DIRECT_SCRIPTS_SANDBOX_TEST(2000928),
    /**
     * Директ: perl- и java-intapi
     */
    DIRECT_INTAPI_PROD(2000390),
    DIRECT_INTAPI_TEST(2000693),
    DIRECT_INTAPI_SANDBOX(2000920),
    DIRECT_INTAPI_SANDBOX_TEST(2000926),
    /**
     * Директ: perl- и java-web
     */
    DIRECT_WEB_PROD(2000769),
    DIRECT_WEB_TEST(2000771),
    /**
     * Директ: perl- и java-api
     */
    DIRECT_API_PROD(2000773),
    DIRECT_API_SANDBOX(2000918),
    DIRECT_API_TEST(2000775),
    DIRECT_API_SANDBOX_TEST(2000924),
    /**
     * Директ: ess
     */
    DIRECT_ESS_PROD(2009823),
    DIRECT_ESS_TEST(2009825),
    /**
     * Директ: canvas
     */
    DIRECT_CANVAS_PROD(2012206),
    DIRECT_CANVAS_TEST(2012208),

    /**
     * Директ: chassis
     */
    DIRECT_CHASSIS(2031222),
    DIRECT_CHASSIS_TEST(2031224),
    /**
     * Bannerstorage: harvester
     */
    BANNERSTORAGE_HARVESTER_PROD(2030471),
    BANNERSTORAGE_HARVESTER_TEST(2030473),
    /**
     * Автотесты Директа
     */
    DIRECT_AUTOTESTS(2009913),
    /**
     * "Рукояточник" - сервис-помощник для ручных тестировщиков
     */
    DIRECT_HANDLES(2018694),
    /**
     * Разработчики Директа
     */
    DIRECT_DEVELOPER(2009921),
    /**
     * Супермодерация (новая модерация)
     */
    SUPERMODERATION_PROD(2002030),
    SUPERMODERATION_TEST(2002032),

    /**
     * Приложение на стороне модерации, которое блокирует клиентов
     */
    CLIENT_BAN_PROD(2030731),
    CLIENT_BAN_TEST(2030729),

    /**
     * Ручная модерация - продакшен
     */
    MODADVERT_MANUAL_PROD(2011058),

    /**
     * Робот на стороне модерации, делающий запросы в директ из нирваны
     */
    MODQUALITY_NIRVANA_ROBOT(2035077),

    /**
     * Антифрод
     * https://abc.yandex-team.ru/services/afraud/
     */
    FURY_SUPERMOD_PROD(2025378),
    FURY_SUPERMOD_PREPROD(2025376),
    FURY_SUPERMOD_TEST(2025374),
    /**
     * MBI API Маркета
     */
    MARKET_MBI_API_PROD(2011130),
    MARKET_MBI_API_TEST(2011132),
    MARKET_MBI_API_DEV(2011134),
    /**
     * Автотесты Модерации ("М1 для интеграционных тестов")
     */
    MODERATE_AUTOTESTS(2010046),
    /**
     * Внутреннее API Метрики
     */
    METRIKA_INTERNAL_API_PROD(2000270),
    METRIKA_INTERNAL_API_TEST(2000269),
    /**
     * https://st.yandex-team.ru/DIRECT-80134#5b30c90cffa28d001b2c431d
     */
    METRIKA_AUDIENCE_PROD(2000306),
    METRIKA_AUDIENCE_TEST(2000305),
    /**
     * clicklogd Метрики (обогащение кликовых логов?)
     * https://st.yandex-team.ru/METR-31343
     */
    METRIKA_CLICKLOGD_PROD(2010172),
    METRIKA_CLICKLOGD_TEST(2010176),
    /**
     * https://st.yandex-team.ru/METR-31021
     */
    METRIKA_AUDIENCE_CRYPTA_SENDER_PROD(2010138),
    METRIKA_AUDIENCE_CRYPTA_SENDER_TEST(2010136),
    METRIKA_MARKEDPHONED_PROD(2010134),
    METRIKA_MARKEDPHONED_TEST(2010132),
    METRIKA_SCHEDULERD_PROD(2010130),
    METRIKA_SCHEDULERD_TEST(2010128),
    /**
     * https://st.yandex-team.ru/METR-41976
     */
    METRIKA_DIRECT_DICT_UPDATER_PROD(2032600),
    METRIKA_DIRECT_DICT_UPDATER_TEST(2032598),
    /**
     * metrika faced
     */
    METRIKA_API_PROD(2000268),
    METRIKA_API_TEST(2000233),
    /**
     * фронтэнд метрики
     */
    METRIKA_FRONTEND_PROD(2000246),
    METRIKA_FRONTEND_TEST(2000234),
    EXPERT_API_PRODUCTION(2001476),
    EXPERT_API_TESTING(2001462),
    UGC_DB_PROD(2000868),
    UGC_DB_TEST(2000870),

    /**
     * Аппметрика, внутренее API
     */
    APP_METRIKA_API_PROD(2002840),
    APP_METRIKA_API_TEST(2002950),

    /**
     * оригинальные названия этих приложений: jafar-prod / jafar-test
     */
    LAUNCHER_PROD(2001918),
    LAUNCHER_TEST(2001914),
    /**
     * Продакшн инсталляция LogBroker
     */
    LOGBROKER_PRODUCTION(2001059),
    /**
     * Prestable инсталляция LogBroker
     */
    LOGBROKER_PRESTABLE(2001147),
    /**
     * aka binlogbroker
     */
    DIRECT_PPCDATA_BINLOG(2001403),
    DIRECT_PPCDATA_BINLOG_TEST(2001405),
    /**
     * Геопродукт
     * https://st.yandex-team.ru/DIRECT-89467
     * оригинальные названия этих приложений: geoadv-back / geoadv-back-testing
     */
    GEO_PRODUCT_PROD(2000731),
    GEO_PRODUCT_TEST(2000757),
    /**
     * Геопродукт
     * https://st.yandex-team.ru/DIRECT-89467
     * оригинальные названия этих приложений: geoadv-back / geoadv-back-testing
     */
    GEO_SEARCH_PROD(2001886),
    GEO_SEARCH_TEST(2008261),
    /**
     * Tycoon ака справочник
     * https://st.yandex-team.ru/DIRECT-93449
     */
    TYCOON_PROD(2002622),
    TYCOON_TEST(2002604),
    TYCOON_FRONTEND_PROD(2002649),
    TYCOON_FRONTEND_TEST(2002608),
    /**
     * Публичный API Справочника
     * https://st.yandex-team.ru/DIRECT-96364
     */
    SPRAV_API_PROD(2011746),
    SPRAV_API_TEST(2011748),
    /**
     * Сервис управления лэндингами в ЯБизнесе
     */
    LANDLORD_API_PROD(2024467),
    LANDLORD_API_TEST(2024465),
    /**
     * Сервис Яндекс Услуг
     */
    YDO_PROD(2002420),
    YDO_TEST(2002414),
    /**
     * Тикет для комдепа для похода в ручку создания клиента
     */
    SMB(2015659),
    /**
     * Тикет для отдела модерации и генерации контента для похода в ручку создания клиента
     */
    FREESERVICE(2016339),
    /**
     * Приложение рестаппа Еды
     */
    EDA_RESTAPP_PROD(2023319),
    EDA_RESTAPP_TEST(2023317),
    /**
     * Ещё одно приложение рестаппа Еды
     */
    EDA_RESTAPP_MARKETING_PROD(2025037),
    EDA_RESTAPP_MARKETING_TEST(2025035),
    EATS_ADVERTS(2030946),
    /**
     * Inventory
     * оригинальные названия этих приложений: inventori_to_direct_internal_api / inventori_to_direct_internal_api_test
     */
    INVENTORI_PROD(2009954),
    INVENTORI_TEST(2009990),
    /**
     * CRM
     */
    CRM_PROD(2001277),
    CRM_TEST(2001283),
    /**
     * CMS видеохостинга
     * https://wiki.yandex-team.ru/video/VideoPoisk/devops/vh-ugc/external_services/
     */
    CMS_PROD(2010946),
    CMS_TEST(2006059),
    /**
     * Партнерский интерфейс РСЯ
     */
    RSYA_PARTNER_INTERFACE_PROD(2010230),
    RSYA_PARTNER_INTERFACE_TEST(2010232),
    /**
     * Диалоги aka paskills
     */
    DIALOGS_PROD(2000252),
    DIALOGS_TEST(2000245),
    /**
     * Турбо-аппы
     */
    TURBO_APP_PROD(2018238),
    TURBO_APP_TEST(2018236),
    /**
     * Телефония
     */
    TELEPHONY_PROD(2019743),
    TELEPHONY_TEST(2019745),
    /**
     * takeout
     */
    TAKEOUT_PROD(2009785),
    TAKEOUT_TEST(2009783),
    /**
     * ADV (yandex.ru/adv)
     * оригинальные названия этих приложений: Advertising production / Advertising frontend testing
     */
    ADV_PROD(205),
    ADV_TEST(2000159),
    /**
     * IDM (Управлятор)
     */
    IDM_PROD(2001600),
    IDM_TEST(2001602),
    /**
     * IDM для проекта Коннект
     */
    IDM_B2B_PROD(2014580),
    IDM_B2B_TEST(2014382),
    /**
     * Коннект
     * В тестинге действительно тот же TVM ID
     * https://wiki.yandex-team.ru/Connect/projects/cabinetb2b/instruction/resources/backend/
     */
    DIRECTORY_PROD(2000205),
    /**
     * Баланс
     * оригинальные названия этих приложений: paysys-balance-medium / paysys-balance-medium-test
     */
    BALANCE_PROD(2000599),
    BALANCE_TEST(2000601),
    /**
     * Сверка Баланса.
     * оригинальные названия этих приложений: balance-dcs / paysys-balance-dcs-test
     */
    BALANCE_DCS_PROD(184),
    BALANCE_DCS_TEST(2002192),
    /**
     * Яндекс.Взгляд
     * обновление ab сегментов в кампаниях
     */
    SURVEYS_PROD(2011008),
    SURVEYS_TEST(2011006),
    /**
     * Staff
     * API Cтаффa, предоставляет информацию о структуре подразделений и о сотрудниках
     */
    STAFF_PROD(2001974),
    STAFF_TEST(2001976),
    /**
     * Внутренняя реклама и дистрибуция на Портале
     * оригинальные названия этих приложений: yabs_banana / Banana testing
     */
    YABS_BANANA_PROD(2000725),
    YABS_BANANA_TEST(2011108),

    /**
     * Приложение БК для Директ-Мониторинга. DIRECT-114748
     */
    YABS_DIRECT_MONITORING(2019411),

    /**
     * Новая модерация объектов Директа
     */
    LB_MODERATION_PROD(2009765),
    LB_MODERATION_TEST(2009767),

    /**
     * БК наружная реклама
     */
    BS_DOOH_PROD(2011864),

    /**
     * Конструктор турбо-страниц
     * https://abc.yandex-team.ru/services/switchconstructor/
     */
    LPC_SWITCHCONSTRUCTOR_PROD(2002004),
    LPC_SWITCHCONSTRUCTOR_TEST(2002010),

    /**
     * Blackbox
     */
    BLACKBOX_PROD(222),
    BLACKBOX_MIMINO(239),
    BLACKBOX_TEST(224),
    BLACKBOX_PROD_YATEAM(223),

    /**
     * Яндекс.Коллекции
     */
    COLLECTIONS_TEST(2000635),
    COLLECTIONS_PROD(2000637),

    /**
     * Аватарница
     */
    AVATARS_PROD(2002150),

    /**
     * Adfox
     * https://st.yandex-team.ru/DIRECT-103474
     */
    ADFOX_LOGIN(2001946),
    ADFOX_LOGIN_TEST(2002000),

    /**
     * Zora
     */
    ZORA_ONLINE(2000193),
    ZORA_GO(2023123),

    /**
     * LOGVIEWER
     */
    LOGVIEWER_PROD(2017957),
    LOGVIEWER_TEST(2017959),

    /**
     * создание рекомендаций DIRECT-111215
     */
    TURBO_ECOM_PROD(2009789),
    TURBO_ECOM_TEST(2011240),

    /**
     * API паспорта для отправки SMS
     * https://doc.yandex-team.ru/Passport/YaSMSDevGuide/concepts/About.xml
     */
    YA_SMS_PROD(2000836),

    /**
     * Cross-DC YDB cluster
     */
    YDB_CROSSDC(2002490),

    /**
     * UAC backend
     * https://abc.yandex-team.ru/services/mobileproducts/
     */
    RMP_PROD(2024339),
    RMP_TEST(2024337),

    /**
     * Антифродовый сервис Комдепа
     */
    COMDEP_CHECK_FRAUD(2020687),

    /**
     * Генерация id'шников БК
     */
    BS_ID_GENERATOR_PROD(2018015),
    BS_ID_GENERATOR_TEST(2018542),


    PASSPORT_PROD(2000078),

    /**
     * Отправка сервер-пушей
     * */
    XIVA_API_PROD(2008797),
    XIVA_API_TEST(2002788),

    /**
     * Получение всякого из ABC
     */
    ABC_PROD(2012190),

    /**
     * Заведение кампаний Дзена
     */
    ZEN_PROD(2023448),
    ZEN_TEST(2023446),

    /**
     * Генерация текстов баннеров
     */
    BANGEN_PROXY_PROD(2031388),

    /**
     * Экспериментальные ML продукты
     * Генерация 3d видео
     */
    MILAB_PROD(2011644),
    MILAB_TEST(2018752),

    /**
     * Антифрод Паспорта
     */
    PASSPORT_ANTIFRAUD_API_PROD(2019898),
    PASSPORT_ANTIFRAUD_API_TEST(2024973),

    /**
     * Для использования в юнит-тестах
     */
    DUMMY(-2),
    /**
     * Неизвестный сервис. Реальный ID не сохранен
     */
    UNKNOWN(-1);

    private final int id;

    TvmService(int id) {
        this.id = id;
    }

    public int getId() {
        return id;
    }

    private static final Logger logger = LoggerFactory.getLogger(TvmService.class);

    static final Map<Integer, TvmService> SERVICES = Arrays.stream(values())
            .filter(s -> s.getId() > 0)
            .collect(ImmutableMap.toImmutableMap(TvmService::getId, Function.identity()));

    /**
     * Получить TvmService по id приложения
     *
     * @param id TVM Client ID
     * @return инстанс, соответствующшй приложению или {@link #UNKNOWN}, если соответствия не наншлось
     */
    public static TvmService fromId(int id) {
        return SERVICES.getOrDefault(id, UNKNOWN);
    }

    /**
     * Получить TvmService по id приложения
     *
     * @param id TVM Client ID
     * @return инстанс, соответствующшй приложению
     * @throws IllegalArgumentException если переданному id не соответствует ни одно приложение
     */
    public static TvmService fromIdStrict(int id) {
        TvmService service = SERVICES.get(id);
        if (service == null) {
            throw new IllegalArgumentException("Unknown TVM service with id: " + id);
        }
        return SERVICES.get(id);
    }

    static List<Integer> getKnownServices() {
        return EntryStream.of(SERVICES).keys().toList();
    }

    /**
     * Получить сервис по строковому названию.
     * <p>
     * В буквенной записи в конфигурации - легко ошибиться, лучше использовать числовую запись и метод {@code fromId}
     *
     * @throws IllegalArgumentException если переданному id не соответствует ни одно приложение
     */
    @JsonCreator
    public static TvmService fromStringStrict(String value) {
        return valueOf(StringUtils.upperCase(value).replace("-", "_"));
    }

    @Override
    public String toString() {
        return name() + "(" + id + ")";
    }
}
