package ru.yandex.direct.web.auth;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.util.Assert;

/**
 * Проверяет полученные authenticationProviders до первой ошибки.
 * Возвращает результат последней успешной аутентификации или null в случае, когда первая же аутентификации оказалась неуспешной.
 */
public class BestEffortAuthenticationManager implements AuthenticationManager {

    private static final Logger logger = LoggerFactory.getLogger(BestEffortAuthenticationManager.class);

    private final List<AuthenticationProvider> authenticationProviders;

    public BestEffortAuthenticationManager(List<AuthenticationProvider> authenticationProviders) {
        Assert.notNull(authenticationProviders, "authenticationProviders is required");
        Assert.notEmpty(authenticationProviders, "authenticationProviders list must contain at least one provider");
        this.authenticationProviders = authenticationProviders;
    }

    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        logger.debug("trying to authenticate in one of defined providers {}", authentication);

        boolean hasSuccessEffort = false;
        for (AuthenticationProvider provider : authenticationProviders) {
            if (!provider.supports(authentication.getClass())) {
                throw new AuthenticationServiceException("invalid authentication manager configuration: " +
                        provider.getClass().getCanonicalName() + " doesn't support " +
                        authentication.getClass().getName() + " returned by previous provider");
            }

            try {
                authentication = provider.authenticate(authentication);
            } catch (Exception e) {
                break;
            }
            hasSuccessEffort = true;
        }
        if (!hasSuccessEffort) {
            return null;
        }
        return authentication;
    }
}
