package ru.yandex.direct.web.auth;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.util.Assert;

/**
 * Authentication entry point that delegates building response
 * to one of exception handlers that supports that exception type.
 */
public class ExceptionHandlingEntryPoint implements AuthenticationEntryPoint {

    private static final Logger logger = LoggerFactory.getLogger(ExceptionHandlingEntryPoint.class);

    private ExceptionHandler defaultHandler;
    private List<ExceptionHandler> handlers;

    public ExceptionHandlingEntryPoint(ExceptionHandler defaultHandler) {
        this(defaultHandler, new ArrayList<>());
    }

    public ExceptionHandlingEntryPoint(ExceptionHandler defaultHandler, List<ExceptionHandler> handlers) {
        Assert.notNull(defaultHandler, "defaultHandler is required");
        Assert.notNull(handlers, "handlers is required");
        this.defaultHandler = defaultHandler;
        this.handlers = handlers;
    }

    @Override
    public void commence(HttpServletRequest request, HttpServletResponse response,
                         AuthenticationException authException) {
        logger.debug("entering authentication entry point with {}", authException);
        for (ExceptionHandler exceptionHandler : handlers) {
            if (exceptionHandler.supports(authException.getClass())) {
                logger.debug("exception will be handled by {}", exceptionHandler.getClass().getName());
                exceptionHandler.handle(request, response, authException);
                return;
            }
        }
        logger.debug("exception will be handled by default handler {}", defaultHandler.getClass().getName());
        defaultHandler.handle(request, response, authException);
    }
}
