package ru.yandex.direct.web.auth;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.util.Assert;

/**
 * Authenticates at all defined authentication providers
 */
public class SequenceAuthenticationManager implements AuthenticationManager {

    private static final Logger logger = LoggerFactory.getLogger(SequenceAuthenticationManager.class);

    private List<AuthenticationProvider> authenticationProviders;

    public SequenceAuthenticationManager(List<AuthenticationProvider> authenticationProviders) {
        Assert.notNull(authenticationProviders, "authenticationProviders is required");
        Assert.notEmpty(authenticationProviders, "authenticationProviders list must contain at least one provider");
        this.authenticationProviders = authenticationProviders;
    }

    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        logger.debug("trying to authenticate in all defined providers {}", authentication);

        for (AuthenticationProvider provider : authenticationProviders) {
            logger.debug("trying to authenticate in provider {}", provider.getClass().getName());

            if (!provider.supports(authentication.getClass())) {
                throw new AuthenticationServiceException("invalid authentication manager configuration: " +
                        provider.getClass().getCanonicalName() + " doesn't support " +
                        authentication.getClass().getName() + " returned by previous provider");
            }

            authentication = provider.authenticate(authentication);
            if (authentication == null) {
                throw new AuthenticationServiceException("authentication provider returned null: " +
                        provider.getClass().getCanonicalName());
            }

            logger.debug("successfully authenticated in {}", provider.getClass().getName());
        }
        return authentication;
    }
}
