package ru.yandex.direct.web.auth.blackbox;

import java.util.Objects;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.CredentialsExpiredException;
import org.springframework.security.authentication.DisabledException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;

import ru.yandex.direct.blackbox.client.BlackboxClient;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.web.auth.blackbox.exception.BadBlackboxCredentialsException;
import ru.yandex.inside.passport.blackbox2.protocol.BlackboxException;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxSessionIdException;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxSessionIdException.BlackboxSessionIdStatus;
import ru.yandex.misc.ip.IpAddress;

import static com.google.common.base.Preconditions.checkState;

/**
 * Provides authentication at blackbox by sessionid method (by cookies)
 */
public class BlackboxCookieAuthProvider implements AuthenticationProvider {

    private static final String TRACE_BLACKBOX_AUTH = "blackbox:auth";
    private static final Logger logger = LoggerFactory.getLogger(BlackboxCookieAuthProvider.class);

    private final BlackboxClient blackboxClient;

    @Autowired
    public BlackboxCookieAuthProvider(BlackboxClient blackboxClient) {
        this.blackboxClient = Objects.requireNonNull(blackboxClient, "blackboxClient");
    }


    @Override
    public boolean supports(Class<?> authentication) {
        return BlackboxCookieAuthRequest.class.isAssignableFrom(authentication);
    }


    /**
     * Authentication at blackbox by cookies credentials
     *
     * @param auth <code>BlackboxCookieAuthRequest</code> object
     *             with <code>BlackboxCookieCredentials</code> as credentials
     * @return <code>BlackboxCookieAuth2</code> with username and Blackbox sessionid response
     * @throws AuthenticationException
     */
    @Override
    public BlackboxCookieAuth authenticate(Authentication auth) throws AuthenticationException {
        logger.debug("blackbox authentication requested ({})", auth);
        if (auth == null) {
            throw new IllegalArgumentException("no authentication request provided");
        }
        BlackboxCookieAuthRequest blackboxCookieAuthRequest = (BlackboxCookieAuthRequest) auth;
        BlackboxCorrectResponse blackboxResponse = getBlackboxResponse(blackboxCookieAuthRequest);
        checkBlackboxResponse(blackboxResponse);
        BlackboxCookieCredentials credentials = blackboxCookieAuthRequest.getCredentials();
        return createAuthenticationFromResponse(blackboxResponse, credentials);
    }


    /**
     * Makes request to blackbox and returns response
     *
     * @param auth blackbox authentication request with populated credentials
     * @return blackbox sessionId response
     */
    private BlackboxCorrectResponse getBlackboxResponse(BlackboxCookieAuthRequest auth) {
        BlackboxCookieCredentials credentials = auth.getCredentials();
        if (credentials == null || !credentials.isPopulated()) {
            throw new BadBlackboxCredentialsException();
        }
        logger.debug("send sessionId-request to blackbox");
        try (TraceProfile ignored = Trace.current().profile(TRACE_BLACKBOX_AUTH)) {
            IpAddress ipAddress = IpAddress.parse(credentials.getUserIp());
            Optional<String> sslSessionId = Optional.of(credentials.getSslSessionId());
            return blackboxClient.sessionId(ipAddress, credentials.getSessionId(),
                    credentials.getHost(), null, false, sslSessionId,
                    auth.getTvmTicket() != null, auth.getTvmTicket());
        } catch (BlackboxSessionIdException e) {
            logger.error("Invalid SessionId", e);
            switch (e.getStatus()) {
                case DISABLED:
                    throw new DisabledException("blackbox authentication is disabled", e);
                case INVALID:
                    throw new CredentialsExpiredException("blackbox authentication must be refreshed", e);
                default:
                    throw new BadCredentialsException("Invalid sessionId", e);
            }
        } catch (BlackboxException e) {
            logger.error("Can not authenticate at blackbox", e);
            throw new AuthenticationServiceException("Can not authenticate at blackbox", e);
        }
    }

    private void checkBlackboxResponse(BlackboxCorrectResponse response) {
        if (response.getStatus() == BlackboxSessionIdStatus.NEED_RESET.value()) {
            throw new CredentialsExpiredException("blackbox authentication must be refreshed");
        }
        // такого быть не должно, все проверки, кроме NEED_RESET, делаются в getBlackboxResponse
        checkState(response.getStatus() == BlackboxSessionIdStatus.VALID.value());
    }

    private BlackboxCookieAuth createAuthenticationFromResponse(
            BlackboxCorrectResponse response, BlackboxCookieCredentials credentials) {
        return new BlackboxCookieAuth(response, credentials);
    }

}
