package ru.yandex.direct.web.auth.blackbox;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.utils.UrlUtils;

import static ru.yandex.direct.common.util.HttpUtil.HEADER_X_REAL_IP;

/**
 * All credentials required for blackbox sessionid request
 */
public class BlackboxCookieCredentials {

    public static final String SESSION_ID = "Session_id";
    public static final String SSL_SESSION_ID = "sessionid2";
    private static final String X_REAL_IP_HEADER = "X-Real-IP";
    private String sessionId;
    private String sslSessionId;
    private String host;
    private String userIp;

    private BlackboxCookieCredentials() {
    }

    public BlackboxCookieCredentials(String host, String userIp, String sessionId, String sslSessionId) {
        this.host = host;
        this.userIp = userIp;
        this.sessionId = sessionId;
        this.sslSessionId = sslSessionId;
    }

    /**
     * Extracts all credentials required for blackbox sessionid request
     *
     * @param request client request to be authenticated
     * @return credentials
     */
    public static BlackboxCookieCredentials extract(HttpServletRequest request) {
        BlackboxCookieCredentials cookiesCredentials = new BlackboxCookieCredentials();

        cookiesCredentials.host = HttpUtil.getHostHeader(request)
                .map(UrlUtils::trimPort)
                .orElseGet(request::getServerName);
        cookiesCredentials.userIp = HttpUtil.getHeaderValue(request, HEADER_X_REAL_IP)
                .orElseGet(request::getRemoteAddr);

        Cookie[] cookies = request.getCookies();
        if (cookies == null) {
            return cookiesCredentials;
        }
        for (Cookie cookie : request.getCookies()) {
            String name = cookie.getName();
            if (SESSION_ID.equals(name)) {
                cookiesCredentials.sessionId = cookie.getValue();
            } else if (SSL_SESSION_ID.equals(name)) {
                cookiesCredentials.sslSessionId = cookie.getValue();
            }
        }
        return cookiesCredentials;
    }

    public String getSslSessionId() {
        return sslSessionId;
    }

    public String getSessionId() {
        return sessionId;
    }

    public String getHost() {
        return host;
    }

    public String getUserIp() {
        return userIp;
    }

    public boolean isPopulated() {
        return sessionId != null &&
                sslSessionId != null &&
                host != null &&
                userIp != null;
    }

}

