package ru.yandex.direct.web.auth.blackbox;

import java.util.Objects;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.CredentialsExpiredException;
import org.springframework.security.authentication.DisabledException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;

import ru.yandex.direct.blackbox.client.BlackboxClient;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.web.auth.blackbox.exception.BadBlackboxCredentialsException;
import ru.yandex.inside.passport.blackbox2.protocol.BlackboxException;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxOAuthException;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxOAuthStatus;
import ru.yandex.misc.ip.IpAddress;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;


/**
 * Provides authentication at blackbox by oauth method
 */
public class BlackboxOauthAuthProvider implements AuthenticationProvider {

    private static final String TRACE_BLACKBOX_AUTH = "blackbox:auth";
    private static final Logger logger = LoggerFactory.getLogger(BlackboxOauthAuthProvider.class);

    private final BlackboxClient blackboxClient;

    @Autowired
    public BlackboxOauthAuthProvider(BlackboxClient blackboxClient) {
        this.blackboxClient = Objects.requireNonNull(blackboxClient, "blackboxClient");
    }

    @Override
    public boolean supports(Class<?> authentication) {
        return BlackboxOauthAuthRequest.class.isAssignableFrom(authentication);
    }

    /**
     * Authentication at blackbox by provided credentials
     *
     * @param auth <code>BlackboxOauthAuthRequest</code> object
     *             with <code>BlackboxOauthCredentials</code> as credentials
     * @return <code>BlackboxOauthAuth</code> with username and Blackbox response
     * @throws AuthenticationException if auth failed
     */
    @Override
    public BlackboxOauthAuth authenticate(Authentication auth) throws AuthenticationException {
        logger.debug("blackbox authentication requested ({})", auth);
        if (auth == null) {
            throw new IllegalArgumentException("no authentication request provided");
        }
        BlackboxOauthAuthRequest oauthAuthRequest = (BlackboxOauthAuthRequest) auth;
        BlackboxCorrectResponse blackboxResponse = getBlackboxResponse(oauthAuthRequest);
        checkBlackboxResponse(blackboxResponse);
        return createAuthenticationFromResponse(blackboxResponse, oauthAuthRequest.getCredentials());
    }

    /**
     * Makes request to blackbox and returns response
     *
     * @param auth blackbox authentication request with populated credentials
     * @return blackbox oauth response
     */
    private BlackboxCorrectResponse getBlackboxResponse(BlackboxOauthAuthRequest auth) {
        BlackboxOauthCredentials credentials = auth.getCredentials();
        if (credentials == null || !credentials.isPopulated()) {
            throw new BadBlackboxCredentialsException();
        }
        logger.debug("send oauth request to blackbox");
        try (TraceProfile ignored = Trace.current().profile(TRACE_BLACKBOX_AUTH)) {
            return blackboxClient.oAuth(
                    /* userIp = */ IpAddress.parse(credentials.getUserIp().getHostAddress()),
                    /* token = */ credentials.getOauthToken(),
                    /* dbFields = */ emptyList(),
                    /* attributes = */ emptyList(),
                    /* emails = */ Optional.empty(),
                    /* aliases = */ Optional.empty(),
                    /* getUserTicket = */ auth.getTvmTicket() != null,
                    /* tvmTicket = */ auth.getTvmTicket());
        } catch (BlackboxOAuthException e) {
            logger.info("Oauth failed");
            switch (e.getStatus()) {
                case DISABLED:
                    throw new DisabledException("blackbox authentication is disabled", e);
                case INVALID:
                    throw new CredentialsExpiredException("blackbox authentication must be refreshed", e);
                default:
                    throw new BadCredentialsException("Invalid oauth", e);
            }
        } catch (BlackboxException e) {
            logger.error("Can not authenticate at blackbox", e);
            throw new AuthenticationServiceException("Can not authenticate at blackbox", e);
        }
    }

    private void checkBlackboxResponse(BlackboxCorrectResponse response) {
        // такого быть не должно, все проверки делаются в getBlackboxResponse
        checkState(response.getStatus() == BlackboxOAuthStatus.VALID.value());
    }

    private BlackboxOauthAuth createAuthenticationFromResponse(
            BlackboxCorrectResponse response, BlackboxOauthCredentials credentials) {
        return new BlackboxOauthAuth(response, credentials);
    }
}
