package ru.yandex.direct.web.auth.blackbox;

import java.util.Arrays;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import com.google.common.net.UrlEscapers;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.util.URIUtil;

import ru.yandex.direct.common.util.HttpUtil;

@SuppressWarnings("squid:S2068")
public enum PassportUrls {
    AUTH("https://passport.%s/auth?retpath=%s"),
    AUTH_UPDATE("https://passport.%s/auth/update?retpath=%s"),
    POST_REGISTRATION("https://passport.%s/passport?create_login=1&create_password=1&retpath=%s");

    private static final String YANDEX_ZONE = "yandex.";
    private static final String DEFAULT_ZONE = YANDEX_ZONE + "com";
    private static final List<String> PASSPORT_ZONES = Arrays.asList(
            YANDEX_ZONE + "ru",
            YANDEX_ZONE + "com",
            YANDEX_ZONE + "com.tr",
            YANDEX_ZONE + "kz",
            YANDEX_ZONE + "ua",
            YANDEX_ZONE + "by");

    private final String urlPattern;

    PassportUrls(String urlPattern) {
        this.urlPattern = urlPattern;
    }

    public String build(HttpServletRequest request) {
        return String.format(
                urlPattern,
                getZoneByServiceHost(realServerName(request)),
                UrlEscapers.urlFormParameterEscaper().escape(buildRetpath(request))
        );
    }

    public String buildWithoutRetpath(HttpServletRequest request) {
        return String.format(
                urlPattern,
                getZoneByServiceHost(realServerName(request)),
                ""
        );
    }

    /**
     * Сначала берём имя сервера из заголовка {@value HttpUtil#HEADER_X_REAL_HOST}.
     * Если не получилось, то из {@link HttpServletRequest#getServerName()}
     */
    private static String realServerName(HttpServletRequest request) {
        return HttpUtil.getHostHeader(request)
                .orElseGet(request::getServerName);
    }

    private static String buildRetpath(HttpServletRequest request) {
        String urlPath = getRequestURLWithRealHost(request);
        String query = request.getQueryString();
        return query != null ? (urlPath + "?" + query) : urlPath;
    }

    /**
     * Копипаста {@link Request#getRequestURL()} с точностью до получения {@code serverName}
     */
    private static String getRequestURLWithRealHost(HttpServletRequest request) {
        final StringBuffer url = new StringBuffer(128);
        String serverName = realServerName(request);
        URIUtil.appendSchemeHostPort(url, request.getScheme(), serverName, request.getServerPort());
        url.append(request.getRequestURI());
        return url.toString();
    }

    private static String getZoneByServiceHost(String serviceHost) {
        serviceHost = serviceHost.toLowerCase();
        for (String zone : PASSPORT_ZONES) {
            if (serviceHost.endsWith(zone)) {
                return zone;
            }
        }
        return DEFAULT_ZONE;
    }
}
