package ru.yandex.direct.web.core.configuration;

import com.fasterxml.classmate.TypeResolver;
import com.google.common.collect.Ordering;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;
import org.springframework.plugin.core.SimplePluginRegistry;
import org.springframework.web.servlet.config.annotation.ResourceHandlerRegistry;
import org.springframework.web.servlet.config.annotation.ViewControllerRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.builders.RequestHandlerSelectors;
import springfox.documentation.schema.plugins.SchemaPluginsManager;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spi.service.OperationBuilderPlugin;
import springfox.documentation.spi.service.contexts.Orderings;
import springfox.documentation.spring.web.plugins.Docket;
import springfox.documentation.swagger.readers.operation.OperationNotesReader;
import springfox.documentation.swagger.readers.operation.OperationSummaryReader;
import springfox.documentation.swagger2.annotations.EnableSwagger2;

import ru.yandex.direct.config.EssentialConfiguration;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.web.core.swagger.SwaggerApiRedirectController;
import ru.yandex.direct.web.core.swagger.SwaggerRedirectController;

import static java.util.Collections.singletonList;


/**
 * Конфигурация Swagger-сервиса, который отдаёт описание API.
 * <p>
 * {@link SwaggerConfiguration Swagger конфигурация} состоит из статичного swagger-ui.html с обвязкой, раздаваемой через
 * webjar, и swagger-сервиса, описывающего API нашего web-api
 * <p>
 * URL для swagger-сервиса кастомизируется через свойство "{@code springfox.documentation.swagger.v2.path}". Подробнее
 * <a href="springfox.documentation.swagger.v2.path">в документации</a>.
 * <p>
 * UI со статикой раздаётся по URL, указанному в конфигурации под ключом "{@code springfox.documentation.swagger.ui.baseurl}".
 * Для того, чтобы это сделать, был использован подход из документации к Swagger'у:
 * <a href="http://springfox.github.io/springfox/docs/current/#q13">Q. How does one configure swagger-ui for non-springboot applications?</a>
 *
 * @see SwaggerRedirectController
 * @see SwaggerApiRedirectController
 */
@Configuration
@EnableSwagger2
@Import(EssentialConfiguration.class)
@ComponentScan(
        basePackages = "ru.yandex.direct.web.core.swagger",
        excludeFilters = @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION)
)
public class SwaggerConfiguration extends WebMvcConfigurerAdapter {

    @Value("${springfox.documentation.swagger.ui.baseurl}")
    String swaggerUiBaseUrl;

    @Value("${springfox.documentation.swagger.v2.path}")
    String apiBaseUrl;

    @Value("${springfox.documentation.swagger.api.url}")
    String apiUrl;

    @Autowired
    private EnvironmentType environmentType;

    /**
     * Переопределение {@link SchemaPluginsManager} чтобы в проде не показывать лишнюю информацию
     */
    @Bean
    public SchemaPluginsManager schemaPluginsManager(TypeResolver typeResolver) {
        return new SchemaPluginsManager(
                SimplePluginRegistry.create(singletonList(new DirectModelPropertyBuilderPlugin(environmentType))),
                SimplePluginRegistry.create(singletonList(new DirectApiModelBuilder(typeResolver, environmentType))));
    }

    /**
     * Переопределение {@link OperationSummaryReader} чтобы в проде не показывать лишнюю информацию
     * название бина совпадает с {@link OperationSummaryReader}, чтобы заменить его поведение(т.к. плагины выстраиваются в список и применяются последовательно)
     */
    @Bean("operationSummaryReader")
    public OperationBuilderPlugin operationSummaryReader(EnvironmentType environmentType) {
        return new DirectOperationSummaryReader(environmentType);
    }

    /**
     * Переопределение {@link OperationNotesReader} чтобы в проде не показывать лишнюю информацию
     * название бина совпадает с {@link OperationNotesReader}, чтобы заменить его поведение(т.к. плагины выстраиваются в список и применяются последовательно)
     */
    @Bean("operationNotesReader")
    public OperationBuilderPlugin operationNotesReader(EnvironmentType environmentType) {
        return new DirectOperationNotesReader(environmentType);
    }

    /**
     * <b>Комментарий по реализации</b>
     * <p>
     * При использовании конфигурации {@link Docket} по умолчанию к {@code operationId} добавляется магический
     * постфикс "_7". Это происходит из-за того, что Springfox swagger сначала для каждого из 8ми известных ему
     * HTTP-методов (полученных из {@link org.springframework.web.bind.annotation.RequestMethod#values()}) генерирует
     * по одному endpoint'у, которым даёт имена типа "auth", "auth_1", ..., "auth_7". Затем из них он выбирает тот,
     * который является первым по его default'ным правилам сортировки.
     * <p>
     * Мы разворачиваем default'ные правила сортировки однотипных операций, добавляя reverse при инициализации,
     * чтобы выбирался "auth", а не "auth_7"
     *
     * @see springfox.documentation.spi.service.contexts.Defaults#operationOrdering()
     * @see springfox.documentation.spring.web.readers.operation.ApiOperationReader#read(springfox.documentation.spi.service.contexts.RequestMappingContext)
     * @see org.springframework.web.bind.annotation.RequestMethod
     */
    @Bean
    public Docket api() {
        Docket docket = new Docket(DocumentationType.SWAGGER_2)
                .operationOrdering(Ordering.from(Orderings.positionComparator().reversed())
                        .compound(Orderings.nickNameComparator()));

        // документация доступна по https://localhost:9443/docs/
        return docket
                .pathMapping(apiUrl)
                .select()
                .apis(RequestHandlerSelectors.any())
                .paths(PathSelectors.any())
                .build();
    }

    @Override
    public void addResourceHandlers(ResourceHandlerRegistry registry) {
        if (!environmentType.isProductionOrPrestable()) {
            registry.addResourceHandler(swaggerUiBaseUrl + "/**")
                    .addResourceLocations("classpath:/META-INF/resources/");
        }
    }

    @Override
    public void addViewControllers(ViewControllerRegistry registry) {
        registry.addRedirectViewController(swaggerUiBaseUrl + apiBaseUrl, apiBaseUrl);
        if (!environmentType.isProductionOrPrestable()) {
            registry.addRedirectViewController(swaggerUiBaseUrl + "", swaggerUiBaseUrl + "/swagger-ui.html");
            registry.addRedirectViewController(swaggerUiBaseUrl + "/", swaggerUiBaseUrl + "/swagger-ui.html");
        }
    }
}
